#/*
# *  Copyright 2007 hkrn <hikarin@users.sourceforge.jp>
# *
# *  Licensed under the Apache License, Version 2.0 (the "License");
# *  you may not use this file except in compliance with the License.
# *  You may obtain a copy of the License at
# *
# *      http://www.apache.org/licenses/LICENSE-2.0
# *
# *  Unless required by applicable law or agreed to in writing, software
# *  distributed under the License is distributed on an "AS IS" BASIS,
# *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# *  See the License for the specific language governing permissions and
# *  limitations under the License.
# */
#
# $Id: Pool.pm 146 2007-01-28 11:56:19Z hikarin $
#

package Zeromin::Pool;

use strict;

sub new {
    my ( $zClass, $iObject, $key ) = @_;
    my $class = ref $iObject || '';
    my ( $zPool, $iBBS );

    if ( $class eq 'Zeromin::Thread' ) {
        $key ||= $iObject->get_key();
        $iBBS  = $iObject->get_bbs();
        $zPool = bless {
            __bbs => $iBBS,
            __key => $key,
        }, $zClass;
        $iObject->load();
        $zPool->{__thread} = $iObject;
    }
    elsif ( $class eq 'Img0ch::BBS' ) {
        $iBBS = $iObject;
        $key ||= 0;
        $zPool = bless {
            __bbs => $iBBS,
            __key => $key
        }, $zClass;
        require Zeromin::Thread;
        my $zThread = Zeromin::Thread->new( $iBBS, $key );
        $zThread->load();
        $zPool->{__thread} = $zThread;
    }
    else {
        Img0ch::Kernel->throw_exception(
            'Zeromin::Thread or Img0ch::BBS not given');
    }

    require Img0ch::Pool::Subject;
    require Img0ch::Pool::Thread;
    require Zeromin::Subject;
    require Zeromin::Upload;
    my $zSubject = Zeromin::Subject->new($iBBS);
    $zSubject->load();
    $zPool->{__pool_subject} = Img0ch::Pool::Subject->new($iBBS);
    $zPool->{__pool_thread}  = Img0ch::Pool::Thread->new( $iBBS, $key );
    $zPool->{__subject}      = $zSubject;
    $zPool->{__upload}       = Zeromin::Upload->new( $iBBS, $key );

    return $zPool;
}

sub load {
    my ($zPool) = @_;
    $zPool->{__pool_subject}->load();
    $zPool->{__pool_thread}->load();
    $zPool->{__thread}->load();
    $zPool->{__upload}->load();
    return 1;
}

sub set_key {
    my ( $zPool, $key ) = @_;
    $zPool->{__key} = $key;
    $zPool->{__thread}->set_key($key);
    $zPool->{__pool_thread}->set_key($key);
    return;
}

sub create {
    my ($zPool) = @_;
    my $zPThread = $zPool->{__pool_thread};
    $zPThread->count() and return 0;

    my $zThread   = $zPool->{__thread};
    my $zPSubject = $zPool->{__pool_subject};
    my $count     = $zThread->count();
    $zPThread->set( [ @{ $zThread->get(1) }, $zThread->get_subject() ], 1 );
    for ( my $i = 2; $i <= $count; $i++ ) {
        $zPThread->set( $zThread->get($i), $i );
    }

    my $key = $zPool->{__key};
    $zPThread->save();
    $zPSubject->set( $key, $count, $zThread->get_subject() );
    $zPSubject->save();
    $zPool->{__thread}->remove();
    $zPool->{__upload}->freeze();
    $zPool->_plugin('zeromin.pool.thread');

    return 1;
}

sub restore {
    my ($zPool) = @_;
    my $zPThread = $zPool->{__pool_thread};
    $zPThread->count() or return 0;

    my $zThread = $zPool->{__thread};
    my $count   = $zPThread->count();
    $zThread->set( [ @{ $zPThread->get(1) }, $zPThread->get_subject() ], 1 );
    for ( my $i = 2; $i <= $count; $i++ ) {
        $zThread->set( $zPThread->get($i), $i );
    }
    $zThread->save();

    my $zSubject = $zPool->{__subject};
    $zSubject->load();
    $zSubject->age( $zPool->{__key}, $count, $zPThread->subject() );
    $zSubject->save();
    $zPool->remove();
    $zPool->{__upload}->release();
    $zPool->_plugin('zeromin.restore.thread');

    return 1;
}

sub remove {
    my ($zPool) = @_;
    my $zPSubject = $zPool->{__pool_subject};
    my $zPThread  = $zPool->{__pool_thread};
    $zPThread->count() or return 0;

    if ( $zPSubject->remove( $zPool->{__key} ) ) {
        $zPSubject->save();
        my $ret = $zPThread->remove();
        $zPool->{__upload}->remove();
        $zPool->_plugin('zeromin.remove.pool');
        return $ret;
    }
    else {
        return 0;
    }
}

sub to_array { $_[0]->{__pool_subject}->to_array() }

sub _plugin {
    my ( $zPool, $at ) = @_;
    require Zeromin::Plugin;
    my $iBBS    = $zPool->{__bbs};
    my $key     = $zPool->{__key};
    my $zPlugin = Zeromin::Plugin->new($iBBS);
    $zPlugin->do( $at, $zPool, [ $iBBS, $key ] );
    return;
}

1;
__END__
