#/*
# *  Copyright 2007 hkrn <hikarin@users.sourceforge.jp>
# *
# *  Licensed under the Apache License, Version 2.0 (the "License");
# *  you may not use this file except in compliance with the License.
# *  You may obtain a copy of the License at
# *
# *      http://www.apache.org/licenses/LICENSE-2.0
# *
# *  Unless required by applicable law or agreed to in writing, software
# *  distributed under the License is distributed on an "AS IS" BASIS,
# *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# *  See the License for the specific language governing permissions and
# *  limitations under the License.
# */
#
# $Id: MailPost.pm 78 2007-01-11 12:24:39Z hikarin $
#

package Img0ch::App::MailPost;

use strict;
use base qw(Img0ch::App::BBS);
use Email::MIME qw();
use Net::POP3 qw();

sub new {
    my ( $iClass, $iKernel, @args ) = @_;
    my $iApp = $iClass->SUPER::new( $iKernel, @args );
    $iApp->{_request} = Img0ch::App::MailPost::Request->new(
        Img0ch::Request->new(@args));
    return $iApp;
}

sub run {
    my ($iApp)   = @_;
    my $mail_rec = [];
    my $fetch    = 0;
    if ($iApp->get_mail($mail_rec)) {
        my @args = ( 'mobile', 'shiftjis',
            sub { ${ $_[0] } =~ s/\n//gxms; ${ $_[0] } } );
        $iApp->process($mail_rec) or return $iApp->redirect_error(@args);
        $iApp->validate()     or return $iApp->redirect_error(@args);
        $iApp->is_writeable() or return $iApp->redirect_error(@args);
        $iApp->get_date();
        $iApp->post() or return $iApp->redirect_error(@args);
        $iApp->update_subback( undef, 1 );
        $iApp->update_index();
        $fetch++;
    }
    $iApp->redirect_success($fetch);
    return 1;
}

sub redirect_success {
    my ($iApp, $fetch) = @_;
    my $iBBS     = $iApp->bbs();
    my $iConfig  = $iApp->{_config};
    my $iRequest = $iApp->{_request};

    if (!$iBBS) {
        require Img0ch::BBS;
        my $iKernel = $iApp->kernel();
        my @path = split '/', $iRequest->path_info();
        $path[1] =~ /\A([\w\-]+)\z/xms;
        $iBBS = Img0ch::BBS->new( $iKernel, { bbs => $1 } );
    }

    require Img0ch::Metadata;
    require Img0ch::Template;
    my $iMeta     = Img0ch::Metadata->new($iBBS);
    my $iTemplate = Img0ch::Template->new(
        $iBBS,
        {   file    => 'mobile',
            setting => $iApp->setting(),
            version => $iRequest->credit(),
        }
    );
    $iTemplate->param({
        Banner => $iMeta->mobile_banner(),
        Flag => 0,
        Fetch => $fetch,
        Referer => $iRequest->get_header('referer') || '""',
    });

    my $charset = $iConfig->get('DefaultCharset');
    $iRequest->send_http_header( 'text/html', $charset );
    $iTemplate->flush(sub { ${ $_[0] } =~ s/\n//gxms; ${ $_[0] } });
    1
}

sub get_mail {
    my ($iApp, $mail_rec) = @_;
    my $iConfig = $iApp->{_config};

    my $user = $iConfig->get('POP3User');
    my $pass = $iConfig->get('POP3Pass');
    my $pop3 = Net::POP3->new($iConfig->get('POP3Server'));
    my $re = $iConfig->get('UseAPOP') ?
             $pop3->apop( $user, $pass ) :
             $pop3->login( $user, $pass );
    if ($re) {
        my $msgnums = $pop3->list();
        my @msgs    = sort { $a <=> $b } keys %$msgnums;
        my $msgnum  = shift @msgs;
        if ($msgnum) {
            my $msg = $pop3->get($msgnum);
            @$mail_rec = @$msg;
            $pop3->delete($msgnum);
            return 1;
        }
        return 0;
    }
    else {
        $iApp->kernel()->throw_exception($pop3->message());
    }
}

sub process {
    my ($iApp, $mail_rec) = @_;
    my $iConfig = $iApp->{_config};
    my $iKernel = $iApp->kernel();
    my $iRequest = $iApp->request();
    my $mime = Email::MIME->new(join "\r\n", @$mail_rec);
    $iApp->_get_host($iConfig, $mime) or return 0;

    require Unicode::Japanese;
    my $found = 0;
    for my $part ($mime->parts()) {
        if (my $fn = $part->filename()) {
            my $cte = $part->header('Content-Transfer-Encoding');
            if ($cte =~ s/\s+//gxms) {
                $part->header_set('Content-Transfer-Encoding', $cte);
            }
            require File::Temp;
            my ( $fh, $temp ) = File::Temp::tempfile(UNLINK => 1);
            binmode $fh;
            print {$fh} $part->body();
            close $fh;
            $iRequest->upload_internal( $temp, $fn );
            $found++;
        }
        else {
            $iApp->_parse_body(\$part->body()) or return 0
        }
    }
    if (!$found) {
        $iApp->set_error('INVALID_MAILPOST');
        return 0
    }

    my $subject = Unicode::Japanese->new($mime->header('Subject'))->sjis();
    $subject =~ s/\A\s+//xms;
    $subject =~ s/\s+\z//xms;
    _nonl($subject);
    if ($subject =~ /(.*?)<>(.*)/xms) {
        $iRequest->set_param_internal( 'FROM', $1 );
        $iRequest->set_param_internal( 'mail', $2 );
    }
    else {
        $iRequest->set_param_internal( 'FROM', $subject );
        $iRequest->set_param_internal( 'mail', '' );
    }

    $iApp->init() or return 0;
    my $iSetting = $iApp->setting();
    if (!$iSetting->is_checked('BBS_ACCEPT_MAILPOST')) {
        $iApp->set_error('NOT_ACCEPT_ERROR');
        return 0;
    }

    1
}

sub _parse_body {
    my ( $iApp, $body ) = @_;
    my $iRequest = $iApp->request();
    my ( $bbs, $key, $text ) = split '<>', $$body;

    $bbs ||= '';
    $key ||= '';
    if ($bbs =~ /\A(?:\r\n)*([\w\-]+)\z/xms) {
        $iRequest->set_param_internal( 'bbs', $1 );
        if ($key =~ /\A(\d{9,10})\z/xms) {
            $iRequest->set_param_internal( 'key', $1 );
            $text = Unicode::Japanese->new($text)->sjis();
            _nonl($text);
            $text =~ s/\r\n/\n/gxms;
            $text =~ s/\n\n/\n/gxms;
            $text =~ s/\n+\z//gxms;
            $iRequest->set_param_internal( 'MESSAGE', $text );
            return 1;
        }
    }

    $iApp->set_error('INVALID_MAILPOST');
    return 0
}

sub _get_host {
    my ($iApp, $iConfig, $mime) = @_;
    my $from;

    if ( !( $from = $mime->header('Reply-To') ) ) {
        if ( !( $from = $mime->header('Return-Path') ) ) {
            if ( !( $from = $mime->header('From') ) ) {
                $from = 'unknown';
            }
        }
    }

    $from =~ /<(.*?)>/xms and $from = $1;
    if (!$iConfig->get('AcceptMailPostFromAnother')) {
        if ($from !~ /docomo\.ne\.jp/xms
            and $from !~ /ezweb\.ne\.jp/xms
            and $from !~ /jp-[dhtckrnsq]\.ne\.jp/xms
            and $from !~ /vodafone\.ne\.jp/xms) {
            $iApp->set_error('MAILPOST_FROM_NOT_MOBILE_ADDRESS');
            return 0;
        }
    }

    my $ip = $mime->header('Received') || '';
    $ip =~ /.+[\[\(]([\d\.]+)[\]\)]/ixms;
    $ip = $1 || '127.0.0.1';
    my $ip_int = pack 'C*', split('\.', $ip);
    $iApp->{_ip} = $ip;
    $iApp->{_ip_int} = $ip_int;
    $iApp->{_host} = gethostbyaddr($ip_int, 2) || $ip;

    require Digest::MD5;
    my $seed = $iConfig->get('Seed');
    $iApp->{_serial} = Digest::MD5::md5_hex( $from, $seed );
    1
}

sub _nonl {
    $_[0] =~ s/\A(?:\r\n)+//xms;
    $_[0] =~ s/(?:\r\n)+\z//xms;
}

package Img0ch::App::MailPost::Request;

use base qw(Img0ch::RequestIF);

sub new {
   my ( $iClass, $iRequest ) = @_;
   bless { __orig => $iRequest }, $iClass;
}

sub signature { $_[0]->{__orig}->signature() }

*credit = \&signature;

sub init {
    my ( $iRequest, $iConfig ) = @_;
    $iRequest->{__orig}->init($iConfig);
    my $bbs = $iRequest->param('bbs');
    $bbs =~ /\A([\w\-]+)\z/xms;
    $iRequest->{__bbs} = $1 || '';
    my $key = $iRequest->param('key');
    $key =~ /\A(\d\d\d\d\d\d\d\d\d\d?)\z/xms;
    $iRequest->{__key} = $1 || '';
    return 1;
}

sub param {
    my ( $iRequest, $key, $unescape ) = @_;
    $unescape ||= 0;

    if ( !wantarray ) {
        my $value = $iRequest->{"__param_${key}"};
        if ( !$unescape ) {
            $value = Img0ch::Kernel::escape_html_entities($value);
        }
        return $value;
    }
    elsif ( wantarray and !$key ) {
        my @result = grep { /\A__param_\w+\z/xms } keys %$iRequest;
        return @result;
    }
    else {
        my $value = $iRequest->{"__param_${key}"};
        return Img0ch::Kernel::escape_html_entities($_);
    }
}

sub get_header { shift->{__orig}->get_header(@_) }

sub set_header { shift->{__orig}->set_header(@_) }

sub query { $_[0]->{__orig}->query() }

sub path_info { $_[0]->{__orig}->path_info() }

sub is_uploadable {1}

sub tempfile { $_[0]->{__tmp} }

sub filename {
    my ( $iRequest ) = @_;
    my $filename = $iRequest->{__file};
    $filename =~ tr/\0'"<>&//d;
    return $filename;
}

sub cookie { shift->{__orig}->cookie(@_) }

sub send_http_header { shift->{__orig}->send_http_header(@_) }

sub set_param_internal {
    my ( $iRequest, $key, $value ) = @_;
    caller() eq 'Img0ch::App::MailPost' or return;
    $iRequest->{"__param_${key}"} = $value;
    return;
}

sub upload_internal {
    my ( $iRequest, $tempfile, $filename ) = @_;
    caller() eq 'Img0ch::App::MailPost' or return;
    $iRequest->{__file} = $filename;
    $iRequest->{__fsize} = -s $tempfile || 0;
    $iRequest->{__tmp} = $tempfile;
    return;
}

1;
__END__
