#/*
# *  Copyright 2007 hkrn <hikarin@users.sourceforge.jp>
# *
# *  Licensed under the Apache License, Version 2.0 (the "License");
# *  you may not use this file except in compliance with the License.
# *  You may obtain a copy of the License at
# *
# *      http://www.apache.org/licenses/LICENSE-2.0
# *
# *  Unless required by applicable law or agreed to in writing, software
# *  distributed under the License is distributed on an "AS IS" BASIS,
# *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# *  See the License for the specific language governing permissions and
# *  limitations under the License.
# */
#
# $Id: BBS.pm 60 2006-12-31 00:29:40Z hikarin $
#

package Img0ch::BBS;

use strict;

sub new {
    my ( $iClass, $iKernel, $hash ) = @_;
    my $iConfig = $iKernel->get_config();
    my $iRepos  = $iKernel->get_repos( $iKernel->get_repos_path('common') );
    my ( $dir, $id ) = ( $hash->{bbs}, $hash->{id} );

    if ( $iConfig->get('compatible.bbs') ) {
        require Img0ch::Compat::BBS;
        Img0ch::Compat::BBS::update( $iRepos, $dir );
    }
    if ($dir) {
        $id = $iRepos->get_int("I:B.d.${dir}");
        $dir = $id ? $dir : '';
    }
    else {
        $dir = $iRepos->get("I:B.i.${id}");
        $id = $dir ? $id : 0;
    }

    bless {
        _dir    => $dir,
        _id     => $id,
        _path   => join( '/', $iConfig->get('BBSPath'), $dir ),
        _url    => join( '/', $iConfig->get('Server'), $dir ),
        _kernel => $iKernel,
        _repos  => $iRepos,
    }, $iClass;
}

sub get_kernel { $_[0]->{_kernel} }

sub get_common_repos { $_[0]->{_repos} }

sub id { $_[0]->{_id} }

sub bbs { $_[0]->{_dir} }

sub get_repos_path {
    my ( $iBBS, @dirs ) = @_;
    unshift @dirs, $iBBS->{_dir};
    join '/', $iBBS->{_kernel}->get_repos_path( join( '/', @dirs ) );
}

sub path {
    my ( $iBBS, @dirs ) = @_;
    join '/', $iBBS->{_path}, @dirs;
}

sub url {
    my ( $iBBS, @dirs ) = @_;
    join '/', $iBBS->{_url}, @dirs;
}

1;
__END__

=head1 NAME

Img0ch::BBS - 掲示板を管理するクラス

=head1 SYNOPSYS

  use Img0ch::BBS

  my $iBBS = Img0ch::BBS->new($iKernel, $bbs);
  my $id = $iBBS->id();
  my $bbs = $iBBS->bbs();

=head1 DESCRIPTION

掲示板単体を1つのオブジェクトとするクラスです。

=head2 new

=over 4

=item Arguments

$iKernel (Img0ch::Kernel), $bbs_dir

=item Return Value

$iBBS (Img0ch::BBS itself)

=back

I<Img0ch::BBS>のオブジェクトを作成します。

=head2 get_kernel

=over 4

=item Arguments

none

=item Return Value

$iKernel

=back

I<Img0ch::Kernel>のオブジェクトを返します。

=head2 get_common_repos

=over 4

=item Arguments

none

=item Return Value

$iRepos

=back

共通レポジトリのオブジェクトを返します。
この関数はバージョンが2.1.x-3.xでのみ使用可能です。

=head2 id

=over 4

=item Arguments

none

=item Return Value

$bbs_id

=back

現在の掲示板のIDを返します。存在しない場合は0を返します。

=head2 bbs

=over 4

=item Arguments

none

=item Return Value

$bbs_dir

=back

現在の掲示板のディレクトリ名を返します。存在しない場合はナルストリングを返します。

=head2 path

=over 4

=item Arguments

@dirs

=item Return Value

$joined_path

=back

引数から掲示板のディレクトリの含めたパスを作成します。

=head2 url

=over 4

=item Arguments

@dirs

=item Return Value

$joined_url

=back

引数から掲示板のディレクトリの含めたURLを作成します。

=head1 AUTHOR

hkrn E<lt>hikarin@users.sourceforge.jpE<gt>

=cut
