#include "main.h"
#include <stdio.h>
#include <string.h>

extern struct tasks *tss;

void console_task(struct sheet *sht)
{
	struct task *task = task_now();
	struct console cons;
	struct fhandle fh[8];
	int i;

	for (i = 0; i < 8; i++) {
		fh[i].buf = 0;
	}
	cons.sht = sht;
	cons.cx = 8;
	cons.cy = 28;
	cons.cc = -1;
	task->cons = &cons;
	task->fh = fh;

	console_putc(&cons, '>', 1);

	for (;;) {
		io_cli();
		if (fifo_stat(&task->fifo) == 0) {
			task_sleep(task);
			io_sti();
		} else {
			i = fifo_get(&task->fifo);
			io_sti();

			if (i == 2) {	/* J[\ON */
				cons.cc = COLOR_FFFFFF;
			} else if (i == 3) {	/* J[\OFF */
				if (cons.sht != 0) {
					console_putc(&cons, ' ', 0);
				}
				cons.cc = -1;
			} else if (i == 4) {
				console_exit(&cons, task->fat);
			} else if (256 <= i && i <= 511) {
				i -= 256;
				if (i == 8) {
					if (cons.cx > 16) {
						console_putc(&cons, ' ', 0);
						cons.cx -= 8;
					}
				} else if (i == 10) {
					cons.line[cons.cx / 8 - 2] = 0;
					console_putc(&cons, ' ', 0);
					console_nline(&cons);
					console_runcmd(&cons, task->fat);
					console_nline(&cons);
					console_putc(&cons, '>', 1);
					if (cons.sht == 0) {
						console_exit(&cons, task->fat);
					}
				} else {
					if (cons.cx < 240) {
						cons.line[cons.cx / 8 - 2] = i;
						console_putc(&cons, (unsigned char) i, 1);
					}
				}
			}
			if (cons.sht != 0) {
				if (cons.cc >= 0) {
					graphic_box(sht, cons.cc, cons.cx, cons.cy, cons.cx + 7, cons.cy + 15);
				}
				sheet_refresh(sht, cons.cx, cons.cy, cons.cx + 8, cons.cy + 16);
			}
		}
	}
}

struct sheet *console_open(int *fat)
{
	struct memory *mem = (struct memory *) MEMORY_ADDR;
	struct sheet *sht = sheet_alloc();
	unsigned char *buf = (unsigned char *) memory_alloc(mem, 42240);
	struct task *task;

	sheet_set(sht, buf, 256, 165, -1);
	window_alloc(sht, "R\[", 0);
	window_textbox(sht, COLOR_000000, 8, 28, 240, 128);
	task = console_open0(sht, fat);
	sht->task = task;
	sht->flag |= SHEET_CUR;
	return sht;
}

struct task *console_open0(struct sheet *sht, int *fat)
{
	struct memory *mem = (struct memory *) MEMORY_ADDR;
	struct task *task = task_alloc();
	int *fifo = (int *) memory_alloc(mem, 512);

	task = task_alloc();
	task->stack = memory_alloc(mem, 65536);
	task->tss.esp = task->stack + 65536 - 8;
	task->tss.eip = (int) &console_task;
	task->tss.es = 1 * 8;
	task->tss.cs = 2 * 8;
	task->tss.ss = 1 * 8;
	task->tss.ds = 1 * 8;
	task->tss.fs = 1 * 8;
	task->tss.gs = 1 * 8;
	*((int *) (task->tss.esp + 4)) = (int) sht;
	fifo_init(&task->fifo, fifo, 128, task);
	task->fat = fat;
	task_run(task, 2, 2);
	return task;
}

void console_close(struct sheet *sht)
{
	struct memory *mem = (struct memory *) MEMORY_ADDR;
	struct task *task = sht->task;

	memory_free(mem, (int) sht->buf, 42240);
	sheet_free(sht);
	console_close0(task);
	return;
}

void console_close0(struct task *task)
{
	struct memory *mem = (struct memory *) MEMORY_ADDR;

	task_sleep(task);
	memory_free(mem, task->stack, 65536);
	memory_free(mem, (int) task->fifo.buf, 512);
	io_cli();
	task->flag = 0;
	if (tss->tfpu == task) {
		tss->tfpu = 0;
	}
	io_sti();
	return;
}

void console_putc(struct console *cons, unsigned char c, char m)
{
	struct sheet *sht = cons->sht;
	unsigned char s[2];

	s[0] = c;
	s[1] = 0;
	if (s[0] == 0x09) {	/* ^u */
		for (;;) {
			if (cons->sht != 0) {
				graphic_box(sht, COLOR_000000, cons->cx, cons->cy, cons->cx + 7, cons->cy + 15);
				graphic_puts(sht, COLOR_FFFFFF, cons->cx, cons->cy, " ");
				sheet_refresh(sht, cons->cx, cons->cy, cons->cx + 8, cons->cy + 16);
			}
			cons->cx += 8;
			if (cons->cx == 248) {
				console_nline(cons);
			}
			if (((cons->cx - 8) & 0x1f) == 0) {
				break;	/* TCY4 */
			}
		}
	} else if (s[0] == 0x0a) {	/* s */
		console_nline(cons);
	} else if (s[0] == 0x0d) {	/* AiȂj */
	} else {	/* ʏ핶 */
		if (cons->sht != 0) {
			graphic_box(sht, COLOR_000000, cons->cx, cons->cy, cons->cx + 7, cons->cy + 15);
			graphic_puts(sht, COLOR_FFFFFF, cons->cx, cons->cy, s);
			sheet_refresh(sht, cons->cx - 8, cons->cy, cons->cx + 8, cons->cy + 16);
		}
		if (m != 0) {
			cons->cx += 8;
			if (cons->cx == 248) {
				console_nline(cons);
			}
		}
	}
	return;
}

void console_puts(struct console *cons, unsigned char *s, int n)
{
	int i;

	if (n != -1) {
		for (i = 0; i < n; i++) {
			console_putc(cons, s[i], 1);
		}
	} else {
		for (; *s != 0; s++) {
			console_putc(cons, *s, 1);
		}
	}
	return;
}

void console_nline(struct console *cons)
{
	struct task *task = task_now();
	struct sheet *sht = cons->sht;
	int x, y;

	if (cons->cy < 140) {
		cons->cy += 16;
	} else if (sht != 0) {
		for (y = 28; y < 140; y++) {
			for (x = 8; x < 248; x++) {
				sht->buf[y * sht->sx + x] = sht->buf[(y + 16) * sht->sx + x];
			}
		}
		for (y = 140; y < 156; y++) {
			for (x = 8; x < 248; x++) {
				sht->buf[y * sht->sx + x] = COLOR_000000;
			}
		}
		sheet_refresh(sht, 8, 28, 248, 156);
	}
	cons->cx = 8;
	if (task->lbyte != 0) {
		cons->cx += 8;
	}
	return;
}

void console_runcmd(struct console *cons, int *fat)
{
	if (strcmp(cons->line, "mem") == 0 && cons->sht != 0) {
		console_mem(cons);
	} else if (strcmp(cons->line, "cls") == 0 && cons->sht != 0) {
		console_cls(cons);
	} else if (strcmp(cons->line, "dir") == 0 && cons->sht != 0) {
		console_dir(cons);
	} else if (strcmp(cons->line, "exit") == 0) {
		console_exit(cons, fat);
	} else if (strncmp(cons->line, "start ", 6) == 0) {
		console_start(cons, fat);
	} else if (strncmp(cons->line, "ncst ", 5) == 0) {
		console_ncst(cons->line + 5, fat);
	} else if (strncmp(cons->line, "create ", 7) == 0) {
		console_create(cons, fat);
	} else if (strncmp(cons->line, "delete ", 7) == 0) {
		console_delete(cons, fat);
	} else if (strncmp(cons->line, "rename ", 7) == 0) {
		console_rename(cons);
	} else if (cons->line[0] != 0) {
		if (console_app(cons, fat) == 0) {	/* słȂR}h */
			console_puts(cons, "R}hsł܂B", -1);
		}
	}
	return;
}

void console_mem(struct console *cons)
{
	struct memory *mem = (struct memory *) MEMORY_ADDR;
	unsigned char s[20];

	sprintf(s, " %4d / %4d MB", memory_stat(mem) / 1048576, mem->allsize / 1048576);
	console_puts(cons, s, -1);
	return;
}

void console_cls(struct console *cons)
{
	struct sheet *sht = cons->sht;
	int x, y;

	for (y = 28; y < 156; y++) {
		for (x = 8; x < 248; x++) {
			sht->buf[y * sht->sx + x] = COLOR_000000;
		}
	}
	sheet_refresh(sht, 8, 28, 248, 156);
	cons->cy = 12;
	return;
}

void console_dir(struct console *cons)
{
	struct file *file = (struct file *) (DISK_ADDR + 0x00002600);
	int i, j;
	unsigned char s[21];

	for (i = 0; i < 224; i++) {
		if (file[i].name[0] == 0x00) {	/* I[ */
			break;
		}
		if (file[i].name[0] != 0xe5) {	/*  */
			if ((file[i].type & 0x18) == 0) {	/* t@C */
				sprintf(s, "FILENAME.EXT %7d\n", file[i].size);
				for (j = 0; j < 8; j++) {
					s[j] = file[i].name[j];
				}
				s[9] = file[i].ext[0];
				s[10] = file[i].ext[1];
				s[11] = file[i].ext[2];
				if (file[i].ext[0] == ' ') {
					s[8] = ' ';
				}
				console_puts(cons, s, -1);
			}
		}
	}
	return;
}

void console_exit(struct console *cons, int *fat)
{
	struct task *task = task_now();

	io_cli();
	if (cons->sht != 0) {
		console_exitreq(cons->sht);
	} else {
		console_exitreq0(task);
	}
	io_sti();
	for (;;) {
		task_sleep(task);
	}
}

void console_start(struct console *cons, int *fat)
{
	struct sheet *sht = console_open(fat);
	struct fifo *fifo = &sht->task->fifo;
	int i;

	sheet_app(sht);
	window_refresh();
	for (i = 6; cons->line[i] != 0; i++) {
		fifo_put(fifo, 256 + cons->line[i]);
	}
	fifo_put(fifo, 266);
	return;
}

void console_ncst(unsigned char *line, int *fat)
{
	struct task *task = console_open0(0, fat);
	struct fifo *fifo = &task->fifo;
	int i;

	for (i = 0; line[i] != 0; i++) {
		fifo_put(fifo, 256 + line[i]);
	}
	fifo_put(fifo, 266);
	return;
}

void console_create(struct console *cons, int *fat)
{
	struct file *file = file_create(fat, cons->line + 7);

	if (file == 0) {
		console_puts(cons, "t@C쐬ł܂B", -1);
	}
	return;
}

void console_delete(struct console *cons, int *fat)
{
	struct file *file = file_search(cons->line + 7);

	if (file != 0) {
		file_delete(fat, file);
	} else {
		console_puts(cons, "t@C܂B", -1);
	}
	return;
}

void console_rename(struct console *cons)
{
	struct file *file;
	unsigned char *s;
	int i;

	for (s = cons->line + 7; *s != ' '; s++) { }
	*s = 0x00;
	s++;
	for (; *s == ' '; s++) { }
	file = file_search(cons->line + 7);

	if (file != 0) {
		i = file_rename(file, s);
		if (i == 0) {
			console_puts(cons, "t@Cł܂B", -1);
		}
	} else {
		console_puts(cons, "t@C܂B", -1);
	}
	return;
}

int console_app(struct console *cons, int *fat)
{
	struct file *file;
	struct task *task = task_now();
	struct memory *mem = (struct memory *) MEMORY_ADDR;
	unsigned char s[13], *p, *q;
	int apps, segs, esp, dats, dhrb, ahrb, i;

	for (i = 0; i < 13; i++) {
		if (cons->line[i] <= ' ') {
			break;
		}
		s[i] = cons->line[i];
	}
	s[i] = 0;
	file = file_search(s);
	if (file == 0 && s[i - 1] != '.') {
		s[i] = '.';
		s[i + 1] = 'H';
		s[i + 2] = 'R';
		s[i + 3] = 'B';
		s[i + 4] = 0;
		file = file_search(s);
	}

	if (file != 0) {
		/* t@CAȃZOgp */
		apps = file->size;
		p = file_read(fat, file->caddr, apps);
		if (apps >= 36 && strncmp(p + 4, "Hari", 4) == 0 && *p == 0x00) {
			segs = *((int *) p);
			esp = *((int *) (p + 0x0c));
			dats = *((int *) (p + 0x10));
			dhrb = *((int *) (p + 0x14));
			q = (unsigned char *) memory_alloc(mem, segs);
			task->ds = (int) q;
			gdt_set(task->ldt, (int) p, file->size - 1, RXCODE + 0x60);
			gdt_set(task->ldt + 1, (int) q, segs - 1, RWDATA + 0x60);
			for (i = 0; i < dats; i++) {
				q[esp + i] = p[dhrb + i];
			}
			task->mem = 0;

			/* HariMainĂяoA߂ƂɏIpAPIĂяo */
			ahrb = *((int *) (p + 0x1c)) + 0x20 - 9;
			p[4] = 0xe8;
			p[5] = ahrb & 0xff;
			p[6] = (ahrb >> 8) & 0xff;
			p[7] = (ahrb >> 16) & 0xff;
			p[8] = (ahrb >> 24) & 0xff;
			p[9] = 0xb8;
			p[10] = 0x00;
			p[11] = 0x00;
			p[12] = 0x00;
			p[13] = 0x00;
			p[14] = 0xcd;
			p[15] = 0x30;
			runapp(4, 4, 12, esp, &(task->tss.esp0));

			sheet_clean(task);
			timer_allcancel(&task->fifo);
			beep(0);
			for (i = 0; i < 8; i++) {
				if (task->fh[i].buf != 0) {
					file_flushbuf(&task->fh[i]);
					memory_free(mem, (int) task->fh[i].buf, 512);
					task->fh[i].buf = 0;
				}
			}
			task->lbyte = 0;
			memory_free(mem, (int) q, segs);
		} else {
			console_puts(cons, "Ȏst@Cł͂܂B", -1);
		}
		memory_free(mem, (int) p, apps);
		return 1;
	}
	return 0;
}
