#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59bbs, blog like bulletin board system.
#    Copyright (C) 2007-2010 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::DateTime;
use Lib::Ping;
use Lib::String;
use Lib::Filter;
use Lib::Util;
use Lib::Topic;
use Lib::Archive;
use Lib::Search;
use Lib::Keyword;
use Lib::Conf;
use Lib::Language;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $conf = Lib::Conf->new();

sub act() {
    my ($act, $cgi) = @_;
    my ($operation, $object) = split(/_/, $act);

    if ($act eq 'new_post') {
        _new_post($cgi);
    }
    elsif ($act eq 'add_post') {
        _add_post($cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

sub _new_post() {
    my ($cgi) = @_;

    my $topic = Lib::Topic->new();

    my $ptopicid = $cgi->param('topicid');
    if (($ptopicid ne '') && ($ptopicid !~ /\A[0-9]+\z/)) {
        $error->print_error('00002e');
    }

    my $keyword = '';
    if ($ptopicid ne '') {
        $topic->load($ptopicid, 0);
        $keyword = $topic->get_keyword();
    }

    my $url = $cgi->param('url');
    my $title = '';
    my $blockquote = '';
    my $pattern = "s?https?://[-_.!~*'()a-zA-Z0-9;/?:@&=+$,%#]+";
    if (($url ne '') && ($url =~ /\A$pattern\z/)) {
        my $util = Lib::Util->new();
        $url = $util->tag_invalidate($url);
        ($title, $blockquote) = $util->get_blockquote($url);
    }

    my $topicdata;
    $topicdata->{title}   = $title;
    $topicdata->{handle}  = '';
    $topicdata->{text}    = '';
    if ($url ne '') {
        $topicdata->{text} = "$title\n";
        $topicdata->{text} .= "$url\n\n";
        $topicdata->{text} .= "＞$blockquote\n";
    }
    $topicdata->{keyword} = $keyword;

    _show_newtopic($ptopicid, $topicdata, '');
}

sub _add_post() {
    my ($cgi) = @_;

    my $ptopicid = $cgi->param('topicid');
    if (($ptopicid ne '') && ($ptopicid !~ /\A[0-9]+\z/)) {
        $error->print_error('00002e');
    }

    # パラメーターの取得
    my $topicdata;
    $topicdata->{title}   = $cgi->param('title');
    $topicdata->{handle}  = $cgi->param('handle');
    $topicdata->{keyword} = $cgi->param('keyword');
    $topicdata->{text}    = $cgi->param('text');
    $topicdata->{status}  = $conf->get_default_status();

    $topicdata->{keyword} =~ s/　/ /g;
    $topicdata->{keyword} =~ s/\A[ ]+//g;
    $topicdata->{keyword} =~ s/[ ]+\z//g;
    $topicdata->{keyword} =~ s/[ ]+/ /g;

    my $util = Lib::Util->new();
    $topicdata->{title} = $util->tag_invalidate($topicdata->{title});
    $topicdata->{handle} = $util->tag_invalidate($topicdata->{handle});
    $topicdata->{keyword} = $util->tag_invalidate($topicdata->{keyword});
    $topicdata->{text} = $util->tag_invalidate($topicdata->{text});

    my $language = Lib::Language->new();
    # 自サイトのコンテンツからの呼び出しかチェック
    my $docrootdir = $conf->get_docroot_dir();
    my $referer = $ENV{'HTTP_REFERER'};# 呼び出し元を取得
    if ($referer !~ /\A$docrootdir/) {
        _show_newtopic($ptopicid, $topicdata, $language->get_usererror('deny-post'));
    }

    my $topic = Lib::Topic->new();
    $topic->load($ptopicid, 1);

    # 入力チェック
    _check_newtopic($ptopicid, $topicdata);

    my $string = Lib::String->new();

    $topicdata->{editor}     = '';
    $topicdata->{ad1}        = '';
    $topicdata->{ad2}        = '';
    $topicdata->{ad3}        = '';
    $topicdata->{updateinfo} = 1;
#    $topicdata->{updateinfo} = 0;
    $topicdata->{convcrlf}   = 1;
    $topicdata->{updatedate} = 1;
    $topicdata->{url}        = '';
    $topicdata->{trackback}  = '';

    # 新規トピックIDを取得
    my $archive = Lib::Archive->new();
    my $newtopicid = $archive->get_counter() + 1;
    $archive->update_counter($newtopicid);

    my $topicurl = "./?func=detail&id=$newtopicid";
    my $htmlurl = $topicurl;

    my $linknum = 0;
    if ($ptopicid ne '') {
        my @links = $topic->get_links();
        my $linkid = (split(/\,/, $links[$#links]))[0];
        $linkid++;
        my $title = $topicdata->{title};
        my $summary = $string->cut($topicdata->{text}, $conf->get_summary_length());
        $title =~ s/,/enc_conma/g;
        $summary =~ s/,/enc_conma/g;
        my $linkdata = "$linkid,$title,$htmlurl,$summary";
        push @links, $linkdata;
        $topic->set_linknum(scalar(@links));
        $topic->set_links(@links);
        $topic->update();
    }

    my $ipaddr = $ENV{'REMOTE_ADDR'}; # 投稿者のIPアドレス
    my $dt = Lib::DateTime->new();
    my $lastdate = $dt->local_datetime(0);

    my $newtopic = Lib::Topic->new();

    if ($ptopicid ne '') {
        # 新規トピックの関連リンクに親トピックを追加
        my $ptext = $topic->get_text();
        my $ptitle = $topic->get_title();
        my $psummary = $string->cut($ptext, $conf->get_summary_length());
        my $ptopicurl = "./?func=detail&id=$ptopicid";
        $ptitle =~ s/,/enc_conma/g;
        $psummary =~ s/,/enc_conma/g;
        my $linkdata = "1,$ptitle,$ptopicurl,$psummary";
        $newtopic->set_links($linkdata);
        $linknum++;
    }

    # 改行の処理
    if ($topicdata->{convcrlf}) {
        $topicdata->{text} =~ s/\r?\n/<br \/>enc_crlf/g;
    }
    else {
        $topicdata->{text} =~ s/\r?\n/enc_crlf/g;
    }

    $newtopic->set_topicid($newtopicid);
    $newtopic->set_status(1);
    $newtopic->set_editor($topicdata->{editor});
    $newtopic->set_lastdate($lastdate);
    $newtopic->set_title($topicdata->{title});
    $newtopic->set_handle($topicdata->{handle});
    $newtopic->set_text($topicdata->{text});
    $newtopic->set_commentnum(0);
    $newtopic->set_linknum($linknum);
    $newtopic->set_trackbacknum(0);
    if ($topicdata->{ad1} ne 'hidden') {
        $newtopic->set_ad1($topicdata->{ad1});
    }
    if ($topicdata->{ad2} ne 'hidden') {
        $newtopic->set_ad2($topicdata->{ad2});
    }
    if ($topicdata->{ad3} ne 'hidden') {
        $newtopic->set_ad3($topicdata->{ad3});
    }
    $newtopic->set_point(0);
    $newtopic->set_keyword($topicdata->{keyword});
    $newtopic->set_url($topicdata->{url});
    $newtopic->set_ipaddr($ipaddr);
    $newtopic->update();

    # キーワードが設定されていある場合はキーワードデータに追加
    if ($topicdata->{keyword} ne '') {
        my @newarray = split(/ /, $topicdata->{keyword});
        my $keywordobj = Lib::Keyword->new();
        foreach my $keyworditem (@newarray) {
            $keywordobj->add($keyworditem, $newtopicid);
        }
    }

    my $str = "$topicdata->{title} $topicdata->{handle} $topicdata->{editor} $topicdata->{text} $topicdata->{keyword}";
    my $search = Lib::Search->new();
    $search->addindex($newtopicid, $str);

    my $ping = Lib::Ping->new();
    if ($topicdata->{updateinfo}) {
        my $archive = Lib::Archive->new();
        $archive->add_archive($newtopicid, $lastdate);
        $archive->add($newtopicid);
        $ping->weblogupdates($conf->get_docroot_dir(), $conf->get_site_title());
    }

    print "Location: $topicurl", "\n\n";
}

# 新規投稿入力チェック
sub _check_newtopic() {
    my ($ptopicid, $topicdata) = @_;

    my $language = Lib::Language->new();

    # トピックのタイトル
    if ($topicdata->{title} eq '') {
        _show_newtopic($ptopicid, $topicdata, $language->get_usererror('input-title'));
    }

    # トピックの本文
    if ($topicdata->{text} eq '') {
        _show_newtopic($ptopicid, $topicdata, $language->get_usererror('input-body'));
    }
    else {
        my $langcode = $language->get_langcode();
        if (($langcode eq 'ja') && ($topicdata->{text} =~ /\A[\x20-\x7E\r\n]+\z/)) {
            _show_newtopic($ptopicid, $topicdata, $language->get_usererror('deny-post'));
        }
    }

    # キーワードのチェック
    my $keywordobj = Lib::Keyword->new();
    if ($keywordobj->check($topicdata->{keyword})) {
        _show_newtopic($ptopicid, $topicdata, $language->get_usererror('keyword-format'));
    }

    my $filter = Lib::Filter->new();
    if ($filter->check_filter("$topicdata->{title} $topicdata->{handle} $topicdata->{text}")) {
        _show_newtopic($ptopicid, $topicdata, $language->get_usererror('deny-post'));
    }
}

sub _show_newtopic() {
    my ($ptopicid, $topicdata, $msg) = @_;

    if ($msg ne '') {
        $msg = "<p><div id=\"message\">$msg</div></p>";
    }

    my $language = Lib::Language->new();
    my $caption = $language->get_statictext('new-thread');
    my $homelabel = $language->get_statictext('home');
    my $path = '';
    if ($ptopicid ne '') {
        my $topicurl = "./?func=detail&id=$ptopicid";
        my $topic = Lib::Topic->new();
        $topic->load($ptopicid, 0);
        my $ptitle = $topic->get_title();
        my $string = Lib::String->new();
        $ptitle = $string->cut($ptitle, 50);
        $path = <<"END_PATH";
<a class="path" href="./">$homelabel</a>&nbsp;
&gt;&nbsp;
<a class="path" href="$topicurl">$ptitle</a>
&gt;&nbsp;
$caption
END_PATH
    }
    else {
        $path = <<"END_PATH";
<a class="path" href="./">$homelabel</a>&nbsp;
&gt;&nbsp;
$caption
END_PATH
    }

    my $sitetitle = $conf->get_site_title();
    my $subtitle = $conf->get_sub_title();
    my $systemdir = $conf->get_system_dir();

    my $titlelabel = $language->get_statictext('title');
    my $handlelabel = $language->get_statictext('handle');
    my $bodylabel = $language->get_statictext('body');
    my $keywordlabel = $language->get_statictext('keyword-guide');
    my $postbutton = $language->get_statictext('post');
    my $cancelbutton = $language->get_statictext('cancel');

    open my $templatefh, '<', "$systemdir/tmpl/post.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$TOPICID\$/$ptopicid/g;
    $template =~ s/\$TOPICTITLE\$/$topicdata->{title}/g;
    $template =~ s/\$HANDLE\$/$topicdata->{handle}/g;
    $template =~ s/\$TEXT\$/$topicdata->{text}/g;
    $template =~ s/\$KEYWORD\$/$topicdata->{keyword}/g;
    $template =~ s/\$TITLELABEL\$/$titlelabel/g;
    $template =~ s/\$HANDLELABEL\$/$handlelabel/g;
    $template =~ s/\$BODYLABEL\$/$bodylabel/g;
    $template =~ s/\$KEYWORDLABEL\$/$keywordlabel/g;
    $template =~ s/\$POSTBUTTON\$/$postbutton/g;
    $template =~ s/\$CANCELBUTTON\$/$cancelbutton/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

1;
