#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59bbs, blog like bulletin board system.
#    Copyright (C) 2007-2010 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use URI::Escape;

use constant {
    LOCK_SH => 1,
    LOCK_EX => 2,
    LOCK_NB => 4,
    LOCK_UN => 8,
};

use Lib::Logger;
use Lib::Error;
use Lib::Calendar;
use Lib::DateTime;
use Lib::User;
use Lib::String;
use Lib::Util;
use Lib::Ads;
use Lib::Archive;
use Lib::Keyword;
use Lib::Search;
use Lib::Mail;
use Lib::Conf;
use Lib::Language;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $conf = Lib::Conf->new();

sub act() {
    my ($act, $cgi) = @_;
    if ($act eq 'exec_main') {
        _exec_main($cgi);
    }
    else {
        _view_main($cgi);
    }
}

sub _view_main() {
    my ($cgi) = @_;
    my $func = $cgi->param('func');
    my ($sessionid, $session_userid) = _check_login($func, $cgi);
    if ($func eq '') {
        $func = 'list';
    }

    # func: login, signup, pwdreset, list, detail
    if ($func eq 'login') {
        _input_login();
    }
    elsif ($func eq 'signup') {
        _input_signup();
    }
    elsif ($func eq 'pwdreset') {
        _input_pwdreset();
    }
    elsif ($func eq 'list') {
        _view_list($cgi);
    }
    elsif ($func eq 'detail') {
        _view_detail($session_userid, $cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

sub _exec_main() {
    my ($cgi) = @_;

    my $func = $cgi->param('func');
    my ($sessionid, $session_userid) = _check_login($func, $cgi);
#    $logger->write("_exec_main - sessionid:$sessionid");   # for debug
    if (($func eq 'logout') && ($session_userid eq '')) {
        $error->print_error('00002e');
    }

    # func: login, logout, signup, pwdreset
    if ($func eq 'login') {
        _admit_login($cgi);
    }
    elsif ($func eq 'logout') {
        _exec_logout($cgi);
    }
    elsif ($func eq 'signup') {
        _exec_signup($cgi);
    }
    elsif ($func eq 'pwdreset') {
        _exec_pwdreset($cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

sub _check_login() {
    my ($func, $cgi) = @_;

    my $sessionid = '';
    my $session_userid = '';
    if ($func eq 'logout' || $func eq 'detail') {
        $sessionid = $cgi->cookie('cookie59b');
        if ($sessionid ne '') {
            my $user = Lib::User->new();
            $session_userid = $user->check_session($sessionid);
        }
    }

    my @session = ($sessionid, $session_userid);
    return @session;
}

sub _input_login() {
    my $logindata;
    $logindata->{userid} = '';
    $logindata->{password} = '';
    $logindata->{redirect} = $ENV{'HTTP_REFERER'};
    _show_login($logindata, '');
}

sub _admit_login() {
    my ($cgi) = @_;

    my $logindata;
    $logindata->{userid}   = $cgi->param('userid');
    $logindata->{password} = $cgi->param('password');
    $logindata->{redirect} = $cgi->param('redirect');

    my $util = Lib::Util->new();
    $logindata->{userid} = $util->tag_invalidate($logindata->{userid});

    $logindata->{redirect} = uri_unescape($logindata->{redirect});
    my $pattern = "s?https?://[-_.!~*'()a-zA-Z0-9;/?:@&=+$,%#]+";
    if (($logindata->{redirect} ne '') && ($logindata->{redirect} !~ /\A$pattern\z/)) {
        $logindata->{redirect} = '';
    }

    my $language = Lib::Language->new();
    if ($logindata->{userid} eq '') {
        _show_login($logindata, $language->get_usererror('input-userid'));
    }
    if ($logindata->{password} eq '') {
        _show_login($logindata, $language->get_usererror('input-password'));
    }

    my $user = Lib::User->new();
    my $sessionid = $user->login($logindata->{userid}, $logindata->{password}, $ENV{'REMOTE_ADDR'});
    if (!$sessionid) {
        _show_login($logindata, $language->get_usererror('invalid-account'));
    }
#    $logger->write("_admit_login - sessionid:$sessionid");   # for debug

    _send_cookie($sessionid, $logindata->{redirect});
}

sub _send_cookie() {
    my ($sessionid, $redirect) = @_;

    if ($redirect eq '') {
        $redirect = './';
    }
    else {
        my $docrootdir = $conf->get_docroot_dir();
        my $pattern = "$docrootdir/detail([0-9]+)[.]html";
        $redirect =~ s/\A$pattern\z/$docrootdir\/?func=detail&id=$1/g;
    }
#    $logger->write("_send_cookie - redirect:$redirect");   # for debug

    my $cookie = "Set-Cookie: cookie59b=$sessionid\n";
    my $html = <<"END_HTML";
<html>
<head>
<title>Redirecting...</title>
<script type="text/javascript">
<!--
var __onload_flag = window.onload;
window.onload = function() {
    if (__onload_flag) __onload_flag();
    location.href = '$redirect';
}
//-->
</script>
</head>
<body>
Redirecting...
</body>
</html>
END_HTML

    print "Content-Type: text/html\n";
    print "$cookie\n";
    print $html;
}

sub _show_login() {
    my ($logindata, $msg) = @_;

    my $systemdir = $conf->get_system_dir();
    my $sitetitle = $conf->get_site_title();
    my $subtitle = $conf->get_sub_title();

    if ($msg ne '') {
        $msg = "<p><div id=\"message\">$msg</div></p>";
    }

    my $language = Lib::Language->new();
    my $caption = $language->get_statictext('login');
    my $homelabel = $language->get_statictext('home');

    my $path = <<"END_PATH";
<a class="path" href="./">$homelabel</a>&nbsp;
&gt;&nbsp;
$caption
END_PATH

    my $loginguide = $language->get_statictext('login-guide');
    $loginguide =~ s/\$SITETITLE\$/$sitetitle/g;
    my $useridlabel = $language->get_statictext('userid');
    my $passwordlabel = $language->get_statictext('password');
    my $loginbutton = $language->get_statictext('login');
    my $cancelbutton = $language->get_statictext('cancel');
    my $forgetpassword = $language->get_statictext('forget-password');

    open my $templatefh, '<', "$systemdir/tmpl/login.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$LOGINGUIDE\$/$loginguide/g;
    $template =~ s/\$USERIDLABEL\$/$useridlabel/g;
    $template =~ s/\$PASSWORDLABEL\$/$passwordlabel/g;
    $template =~ s/\$LOGINBUTTON\$/$loginbutton/g;
    $template =~ s/\$CANCELBUTTON\$/$cancelbutton/g;
    $template =~ s/\$FORGETPASSWORD\$/$forgetpassword/g;
    $template =~ s/\$USERID\$/$logindata->{userid}/g;
    $template =~ s/\$REDIRECT\$/$logindata->{redirect}/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

sub _exec_logout() {
    my ($cgi) = @_;

    my $sessionid = $cgi->cookie('cookie59b');
    if ($sessionid ne '') {
        my $user = Lib::User->new();
        $user->logout($sessionid);
    }

    print "Location: ./", "\n\n";
}

sub _input_signup() {

    my $userdata;
    $userdata->{userid}     = '';
    $userdata->{mailaddr}   = '';

#    $logger->write('_input_signup');   # for debug
    my $freesignup = $conf->get_free_signup();
    if (!$freesignup) { # ユーザー登録制限
        $error->print_error('00004e');
    }

    _show_signup($userdata, '');
}

sub _exec_signup() {
    my ($cgi) = @_;

    my $userdata;
    $userdata->{userid}   = $cgi->param('userid');
    $userdata->{mailaddr} = $cgi->param('mailaddr');

    my $util = Lib::Util->new();
    $userdata->{userid} = $util->tag_invalidate($userdata->{userid});
    $userdata->{mailaddr} = $util->tag_invalidate($userdata->{mailaddr});

    my $user = Lib::User->new();
    my $language = Lib::Language->new();

    if ($userdata->{userid} eq '') {
        _show_signup($userdata, $language->get_usererror('input-userid'));
    }
    if ($userdata->{userid} =~ /[^0-9a-zA-Z]/) {
        _show_signup($userdata, $language->get_usererror('userid-format'));
    }
    if (!$user->check_valid($userdata->{userid})) {
        _show_signup($userdata, $language->get_usererror('userid-exist'));
    }
    if ($userdata->{mailaddr} eq '') {
        _show_signup($userdata, $language->get_usererror('input-mailaddr'));
    }
    if ($userdata->{mailaddr} !~ /[-_0-9a-z]+\@[-_0-9a-z]+\.[-_0-9a-z.]+/) {
        _show_signup($userdata, $language->get_usererror('mailaddr-format'));
    }

    # 10000 ～ 99999 の初期パスワードを生成
    my $password = int(rand(89999)) + 10000;

    my $userid = $userdata->{userid};
    my $mailaddr = $userdata->{mailaddr};
    my $username = $userid;

    my $auth = 1;

    # add user
    $user->set_userid($userid);
    $user->set_password($password);
    $user->set_mailaddr($mailaddr);
    $user->set_hpurl('');
    $user->set_username($username);
    $user->set_auth($auth);
    $user->set_status(1);
    $user->set_ad1('');
    $user->set_ad2('');
    $user->set_parts('');
    $user->add();

    my $systemdir = $conf->get_system_dir();

    my $sitetitle = $conf->get_site_title();
    my $subtitle = $conf->get_sub_title();

    my $useridlabel = $language->get_statictext('userid');
    my $passwordlabel = $language->get_statictext('init-password');
    my $maillabel = $language->get_statictext('mail-address');

    # send sign up mail to user.
    my $signupnotice = $language->get_message('signup-notice');
    my $signupmail1 = $language->get_message('signup-mail1');
    my $signupmail2 = $language->get_message('signup-mail2');

    $signupnotice =~ s/\$SITETITLE\$/$sitetitle/g;
    $signupmail1  =~ s/\$SITETITLE\$/$sitetitle/g;

    my $subject = $signupnotice;
    my $body = <<"END_BODY";
$signupmail1
$signupmail2

$useridlabel :
$userid

$passwordlabel :
$password

$maillabel :
$mailaddr
END_BODY

    my $mail = Lib::Mail->new();
    $mail->send($subject, $body, $mailaddr);

    # send sign up information mail to webmaster.
    my $signupinfo = $language->get_message('signup-info');
    my $signupinfobody = $language->get_message('signup-infobody');

    $signupinfo     =~ s/\$USER\$/$userdata->{userid}/g;
    $signupinfobody =~ s/\$USER\$/$userdata->{userid}/g;

    $subject = $signupinfo;
    $body = <<"END_BODY";
$signupinfobody

$maillabel : $mailaddr
END_BODY

    $mail->send($subject, $body, $conf->get_admin_address());

    my $homelabel = $language->get_statictext('home');
    my $signuplabel = $language->get_statictext('signup');
    my $caption = $language->get_statictext('signup-end');

    my $path = <<"END_PATH";
<a class="path" href="./">$homelabel</a>&nbsp;
&gt;&nbsp;
<a class="path" href="./?func=signup">$signuplabel</a>&nbsp;
&gt;&nbsp;
$caption
END_PATH

    # sign up complete message
    my $signupcomplete = $language->get_message('signup-complete');
    $signupcomplete =~ s/\$SITETITLE\$/$sitetitle/g;
    my $msg = "<p>$signupcomplete</p>";

    open my $templatefh, '<', "$systemdir/tmpl/message.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$MSG\$/$msg/g;

    print "Content-Type: text/html\n\n";
    print $template;
}

sub _show_signup() {
    my ($userdata, $msg) = @_;

    if ($msg ne '') {
        $msg = "<p><div id=\"message\">$msg</div></p>";
    }

    my $language = Lib::Language->new();
    my $caption = $language->get_statictext('signup');
    my $homelabel = $language->get_statictext('home');

    my $path = <<"END_PATH";
<a class="path" href="./">$homelabel</a>&nbsp;
&gt;&nbsp;
$caption
END_PATH

    my $systemdir = $conf->get_system_dir();
    my $sitetitle = $conf->get_site_title();
    my $subtitle = $conf->get_sub_title();

    my $useridlabel = $language->get_statictext('userid');
    my $maillabel = $language->get_statictext('mail-address');
    my $signupbutton = $language->get_statictext('signup');
    my $cancelbutton = $language->get_statictext('cancel');

    open my $templatefh, '<', "$systemdir/tmpl/signup.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$USERIDLABEL\$/$useridlabel/g;
    $template =~ s/\$MAILLABEL\$/$maillabel/g;
    $template =~ s/\$SIGNUPBUTTON\$/$signupbutton/g;
    $template =~ s/\$CANCELBUTTON\$/$cancelbutton/g;
    $template =~ s/\$USERID\$/$userdata->{userid}/g;
    $template =~ s/\$MAILADDR\$/$userdata->{mailaddr}/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

sub _input_pwdreset() {
    _show_pwdreset('', '');
}

sub _exec_pwdreset() {
    my ($cgi) = @_;

    my $language = Lib::Language->new();
    my $userid = $cgi->param('userid');

    my $util = Lib::Util->new();
    $userid = $util->tag_invalidate($userid);

    if ($userid eq '') {
        _show_pwdreset($userid, $language->get_usererror('input-userid'));
    }

    my $systemdir = $conf->get_system_dir();
    my $userfile = "$systemdir/user/$userid/profile.txt";
    if (!(-f $userfile)) {
        _show_pwdreset($userid, $language->get_usererror('not-signup'));
    }

    my $user = Lib::User->new();
    $user->load($userid);
    my $mailaddr = $user->get_mailaddr();
    my $status   = $user->get_status();
    if ($status != 1) {
        _show_pwdreset($userid, $language->get_usererror('invalid-userid'));
    }

    # 10000 ～ 99999 の初期パスワードを生成
    my $password = int(rand(89999)) + 10000;
    $user->set_password($password);
    $user->update();

    my $mail = Lib::Mail->new();
    my $subject = $language->get_statictext('password-reset');
    my $body = $language->get_message('pwdreset-body');
    $body =~ s/\$PASSWORD\$/$password/g;
    $mail->send($subject, $body, $mailaddr);

    my $caption = $language->get_statictext('password-reset');
    my $loginlink = $language->get_statictext('login');
    my $homelabel = $language->get_statictext('home');

    my $path = <<"END_PATH";
<a class="path" href="./">$homelabel</a>&nbsp;
&gt;&nbsp;
<a class="path" href="./?func=login">$loginlink</a>&nbsp;
&gt;&nbsp;
$caption
END_PATH

    my $pwdresetmail = $language->get_message('pwdreset-mail');
    my $msg = <<"END_MSG";
<p>
<b><big>$pwdresetmail</big></b>
</p>
<p>
<a href=\"./?func=login\">$loginlink</a>
</p>
END_MSG

    my $sitetitle = $conf->get_site_title();
    my $subtitle = $conf->get_sub_title();

    open my $templatefh, '<', "$systemdir/tmpl/message.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$MSG\$/$msg/g;

    print "Content-Type: text/html\n\n";
    print $template;
}

sub _show_pwdreset() {
    my ($userid, $msg) = @_;

    if ($msg ne '') {
        $msg = "<p><div id=\"message\">$msg</div></p>";
    }

    my $language = Lib::Language->new();
    my $caption = $language->get_statictext('password-reset');
    my $loginlink = $language->get_statictext('login');
    my $homelabel = $language->get_statictext('home');

    my $path = <<"END_PATH";
<a class="path" href="./">$homelabel</a>&nbsp;
&gt;&nbsp;
<a class="path" href="./?func=login">$loginlink</a>&nbsp;
&gt;&nbsp;
$caption
END_PATH

    my $systemdir = $conf->get_system_dir();
    my $sitetitle = $conf->get_site_title();
    my $subtitle = $conf->get_sub_title();

    my $pwdresetguide = $language->get_statictext('pwdreset-guide');
    my $useridlabel = $language->get_statictext('userid');
    my $executebutton = $language->get_statictext('execute');
    my $cancelbutton = $language->get_statictext('cancel');

    open my $templatefh, '<', "$systemdir/tmpl/pwdreset.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$USERID\$/$userid/g;
    $template =~ s/\$PWDRESETGUIDE\$/$pwdresetguide/g;
    $template =~ s/\$USERIDLABEL\$/$useridlabel/g;
    $template =~ s/\$EXECUTEBUTTON\$/$executebutton/g;
    $template =~ s/\$CANCELBUTTON\$/$cancelbutton/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

sub _view_list() {
    my ($cgi) = @_;

    my $session_userid = '';
    my $user = Lib::User->new();
    my $sessionid = $cgi->cookie('cookie59b');
    if ($sessionid ne  '') {
        $session_userid = $user->check_session($sessionid);
    }
    my $date = $cgi->param('date');
    my $q = $cgi->param('q');
    my $page = $cgi->param('page');
    my $f = $cgi->param('f');
    $q =~ s/　/ /g;
    my $util = Lib::Util->new();
    $q = $util->tag_invalidate($q);
    if ($page ne '' && $page !~ /\A[0-9]+\z/) {
        $error->print_error('00002e');
    }
    $page = ($page eq '') ? 1 : $page;

    my @items = ();
    my $format = 0;
    my ($year, $month, $day) = ('', '', '');
    my $archive = Lib::Archive->new();

    # 日時を取得
    my $dt = Lib::DateTime->new();
    $dt->local_datetime(0);
    $year = $dt->get_year();
    $month = $dt->get_month();

    if ($q eq '') { # 更新情報
        @items = $archive->get_newtopics();
    }
    else { # 検索
        if ($q =~ /\A[0-9\,]+\z/) { # 数字とコンマだけの場合はID指定
            my $datadir = $conf->get_data_dir();
            @items = split(/\,/, $q);
            # データファイルが存在するかチェック
            my @checkeditems = ();
            foreach my $item (@items) {
                my $filename = "$datadir/$item" . "tp.data";
                if (-f $filename) {
                    push @checkeditems, $item;
                }
            }
            @items = @checkeditems;
        }
        elsif ($q =~ /\Anew:\z/i) { # new: で新着トピックを表示
            @items = _new_topics();
        }
        elsif ($q =~ /\Adate:[0-9]+\z/i) { # date:yyyymm or date:yyyymmdd でアーカイブの検索
            $format = 1;
            $year = substr($q, 5, 4);
            $month = substr($q, 9, 2);
            if (length(substr($q, 5)) > 6) {
                $day = substr($q, 11, 2);
                $format = 2;
            }
            @items = $archive->get_archivetopics($year, $month, $day);
        }
        else {
            my $search = Lib::Search->new();
            @items = $search->search($q);
        }
        $format = 3;
        @items = reverse (sort { $a <=> $b; } @items); # 降順にソート
    }

    if ($f eq 'rss') { # RSS出力指定
        _list_rss($q, @items);
    }
    else {
        my $listhtml = _list_html($session_userid, $format, "$year/$month/$day", $page, $q, @items);
        print "Content-Type: text/html\n\n";
        print $listhtml;
    }
}

# 新着トピックの一覧を取得
sub _new_topics {
    my $archive = Lib::Archive->new();
    my $counter = $archive->get_counter();
    my $topic = Lib::Topic->new();

    my $datadir = $conf->get_data_dir();
    my $newnum = $conf->get_new_item_num();
#    $logger->write("_new_topics");   # for debug

    my @newtopics = ();
    my $topicid  = $counter;
    my $i = 0;
    while ($topicid >= 0) {
        my $filename = "$datadir/$topicid" . "tp.data";
        if (-f $filename) {
            push @newtopics, $topicid;
        }
        $topicid--;
        $i++;
        last if ($i >= $newnum);
    }
    return @newtopics;
}

# 検索結果をRSS出力
sub _list_rss {
    my $q = shift;
    my @result = @_;
#    $logger->write("_list_rss");   # for debug

    my $maxnum = $conf->get_result_count();
    $maxnum--;
    @result = @result[0 .. $maxnum];

    my $archive = Lib::Archive->new();
    my $rssxml = $archive->get_rss2(@result);

    print "Content-Type: text/xml; charset=utf-8\n\n";
    print $rssxml;
}

# listページのhtmlソース作成
sub _list_html {
    my $session_userid = shift;
    my $format = shift;
    my $date = shift;
    my $page = shift;
    my $q = shift;
    my @result = @_;

    my $docrootdir = $conf->get_docroot_dir();
    my $sitetitle = $conf->get_site_title();
    my $subtitle = $conf->get_sub_title();
    my $language = Lib::Language->new();

    my $caption = $sitetitle;
    my $listtitle = $language->get_statictext('update-info');
    my $metadisc = "<meta name=\"description\" content=\"$subtitle\">";
    my $metakeywords = "<meta name=\"keywords\" content=\"blog,software,social,bookmark,media\">";

    my $currentq = ''; 
    my ($year, $month, $day) = split(/\//, $date);
    my $currentdate = '';
    if ($format == 1) {
        $listtitle = "$year/$month";
        $caption = "$listtitle：$sitetitle";
        $currentdate = "$year$month";
        $metadisc = "<meta name=\"description\" content=\"$listtitle\">";
    }
    elsif ($format == 2) {
        $listtitle = "$year/$month/$day";
        $caption = "$listtitle：$sitetitle";
        $currentdate = "$year$month$day";
        $metadisc = "<meta name=\"description\" content=\"$listtitle\">";
    }
    elsif ($format == 3) {
        $listtitle = $language->get_statictext('search-result');
        $listtitle =~ s/\$QUERY\$/$q/g;
        $caption = "$listtitle：$sitetitle";
        $metadisc = "<meta name=\"description\" content=\"$listtitle\">";
        $metakeywords = "<meta name=\"keywords\" content=\"$q\">";
        $currentq = uri_escape($q);
    }

    $sitetitle = "<a class=\"sitetitle\" href=\"$docrootdir/\">$sitetitle</a>";

    my $systemdir = $conf->get_system_dir();
    my $rssurl = "$docrootdir/rss.xml";
    if ($q ne '') {
        $rssurl = "$docrootdir/?q=$currentq&f=rss";
    }

    my $homelabel = $language->get_statictext('home');

    my $path = <<"END_PATH";
<a class="path" href="$docrootdir/">$homelabel</a>
END_PATH

    my $signup = $language->get_statictext('signup');
    my $subscribe = $language->get_statictext('subscribe');
    my $login = $language->get_statictext('login');
    my $toplist = $language->get_statictext('top-list');
    my $prevlist = $language->get_statictext('prev-list');
    my $nextlist = $language->get_statictext('next-list');
#    my $post = $language->get_statictext('post');
    my $post = $language->get_statictext('new-thread');

    my $freesignup = $conf->get_free_signup();
    my $usersignup = '';
    if ($freesignup) {
        $usersignup = "<a class=\"menu\" href=\"./?func=signup\">$signup</a>&nbsp;";
    }
    my $menu = <<"END_MENU";
<a class="menu" href="$rssurl">$subscribe</a>&nbsp;<a class="menu" href="./?act=new_post">$post</a>&nbsp;$usersignup<a class="menu" href="./?func=login">$login</a>
END_MENU

    if ($session_userid ne '') {
        $menu = _get_menu($session_userid);
    }

    my $resultlist = $language->get_message('not-found');
    my $pagenavi = '';
    if (scalar(@result) > 0) {
        my $itemnum = $conf->get_result_count();
        my $pagenum = int(scalar(@result) / $itemnum); 
        if (scalar(@result) / $itemnum > $pagenum) {
            $pagenum++;
        }

        my @items = ();
        for my $i (0 .. $#result) {
            if (($i >= ($page - 1) * $itemnum) && ($i < $page * $itemnum)) {
                push @items, $result[$i];
            }
        }

        $resultlist = _get_topiclist($session_userid, @items);

        my $base = 1;
        my $currentnum = ($pagenum < 5) ? $pagenum : 5;
        if ($page > 5) {
            $base = $page - 4;
        }
        my @pagelist = ();
        for my $pagecount ($base .. ($base + $currentnum - 1)) {
            push @pagelist, $pagecount;
        }
        my $current = '';
        foreach my $pagecount (@pagelist) {
            if ($pagecount != $page) {
                $current .= "<a href=\"./?q=$currentq&page=$pagecount\">$pagecount</a>&nbsp;";
            }
            else {
                $current .= "$pagecount&nbsp;";
            }
        }
        my $prev = ''; 
        if ($page > 1) {
            my $prevpage = $page - 1;
            $prev = "<a href=\"./?q=$currentq&page=1\">$toplist</a>&nbsp;&nbsp;";
            $prev .= "<a href=\"./?q=$currentq&page=$prevpage\">$prevlist</a>";
        }
        my $next = ''; 
        if ($page < $pagenum) {
            my $nextpage = $page + 1;
            $next = "<a href=\"./?q=$currentq&page=$nextpage\">$nextlist</a>";
        }
        $pagenavi = <<"END_PAGENAVI";
<p><strong>$prev&nbsp;&nbsp;$current&nbsp;$next</strong></p>
END_PAGENAVI
    }

    # 更新情報を取得
    my $archive = Lib::Archive->new();

    my $calendar = '';
    my $archivelist = '';
    my $newtrackback = '';
    my $newcomment = '';
    # カレンダーを取得
    my $cal = Lib::Calendar->new();
    $calendar = $cal->get_calendar($year, $month);

    $archivelist = $archive->archive_list();

    # 広告の取得
    my $ads = Lib::Ads->new();
    my $ad1 = $ads->get_ad1();
    my $ad2 = $ads->get_ad2();
    my $ad3 = $ads->get_ad3();
    my $adlist = $ads->get_adlist_table();

    # クラウドを取得
    my $keywordobj = Lib::Keyword->new();
    my $clouds = $keywordobj->clouds(1);

    my $searchbutton = $language->get_statictext('search');
    my $updateinfo = $language->get_statictext('update-info');

    my $maincontent = <<"END_MAINCONTENT";
<h2>$listtitle</h2>

$resultlist
END_MAINCONTENT

    # テンプレートの読み込み
    open my $templatefh, '<', "$systemdir/tmpl/main.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$METADISC\$/$metadisc/g;
    $template =~ s/\$METAKEYWORDS\$/$metakeywords/g;
    $template =~ s/\$RSSURL\$/$rssurl/g;
    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MENU\$/$menu/g;
    $template =~ s/\$SITETITLE\$/$sitetitle/g;
    $template =~ s/\$SUBTITLE\$/$subtitle/g;
    $template =~ s/\$PREVQ\$/$q/g;
    $template =~ s/\$SEARCHBUTTON\$/$searchbutton/g;
    $template =~ s/\$MAIN\$/$maincontent/g;
    $template =~ s/\$PAGENAVI\$/$pagenavi/g;
    $template =~ s/\$AD1\$/$ad1/g;
    $template =~ s/\$AD2\$/$ad2/g;
    $template =~ s/\$AD3\$/$ad3/g;
    $template =~ s/\$ADLIST\$/$adlist/g;
    $template =~ s/\$CALENDAR\$/$calendar/g;
    $template =~ s/\$ARCHIVELIST\$/$archivelist/g;
    $template =~ s/\$CLOUDS\$/$clouds/g;
    $template =~ s/\$BLOGPARTS\$//g;
    $template =~ s/\$UPDATEINFO\$/$updateinfo/g;

    return $template;
}

# 指定されたトピックの一覧を作成する
sub _get_topiclist {
    my $session_userid = shift;
    my @topics = @_;

    my $topiclist = '';
    my $topic = Lib::Topic->new();

    for my $i (0 .. $#topics) {
        my $topicid = $topics[$i];
        my $ret = $topic->load($topicid, 0);
        next if (!$ret);
        my $open = ($i < $conf->get_open_item_num()) ? 1 : 0;
        $topiclist .= _topic_outline($topic, $open, $session_userid);
    }

    return $topiclist;
}

# トピック概要のHTMLを作成して返す、loadが前の処理で呼び出されていることが前提
sub _topic_outline {
    my ($topic, $open, $session_userid) = @_;
    my $topicid = $topic->get_topicid();
    my $status = $topic->get_status();
    my $title = $topic->get_title();
    my $text = $topic->get_text();
    my $keywords = $topic->get_keyword();
    my $url = $topic->get_url();
    my $point = $topic->get_point();
    my $commentnum = $topic->get_commentnum();
    my $trackbacknum = $topic->get_trackbacknum();
    my $linknum = $topic->get_linknum();
    my $lastdate = $topic->get_lastdate();

#    $logger->write("_topic_outline");   # for debug

    my $language = Lib::Language->new();
    my $bookmarklink = $language->get_statictext('bookmark-link');
    my $deletenotice = $language->get_statictext('delete-notice');
    my $editlink = $language->get_statictext('edit');

    my $topicclass = 'A';
    my $bookmark = '';
    if ($url ne '') {
        my $bookmarktarget = $conf->get_bookmark_target();
        $topicclass = 'B';
        $bookmark = "&nbsp;&nbsp;<font size=\"-1\"><a href=\"$url\" $bookmarktarget>$bookmarklink</a></font>";
    }

    if ($status == 9) {
        $title = $deletenotice;
        $text = $deletenotice;
    }

    my $topichtml = "./?func=detail&id=$topicid";
    if ($conf->get_static_html() && $session_userid eq '') {
        $topichtml = "detail$topicid.html";
    }

    my $keyword = Lib::Keyword->new();

    my $editor = $topic->get_editor();
    my $handle = $topic->get_handle();
    if ($editor eq '') {
        $editor = $handle;
        if ($editor eq '') {
            $editor = $conf->get_anonymous();
            $editor = "<a href=\"./\">$editor</a>";
        }
        my $util = Lib::Util->new();
        $text = $util->url_link($text, "target=\"_blank\"");
        $text = $keyword->link_keyword($keywords, $text);
    }
    else {
        my $enceditor = uri_escape($editor);
        $editor = "<a href=\"./?q=$enceditor\">$editor</a>";
    }

    my $editmenu = '';
    if ($session_userid ne '') {
        my $loginuser = Lib::User->new();
        if ($loginuser->is_admin($session_userid) || ($session_userid eq $editor)) {
            $editmenu = "&nbsp;<a href=\"./?act=view_editor&func=edittopic&topicid=$topicid\">$editlink</a>";
        }
    }

    my $string = Lib::String->new();
    my $summary = $string->cut($text, $conf->get_summary_length());
    $summary = $keyword->link_keyword($keywords, $summary);

    my $showblock = $open ? "display:none"  : "display:block";
    my $hideblock = $open ? "display:block" : "display:none";

    my $item = <<"END_ITEM";
<div id="toggleShow$topicid" style="$showblock">
  <a class="toggle" href="javascript:;" onmousedown="toggleText('toggleText$topicid','toggleSummary$topicid','toggleShow$topicid','toggleHide$topicid');"><font class="toggle">＋</font></a>&nbsp;
  <b><font class="mark">$topicclass</font> <a href="$topichtml">$title</a></b>$bookmark
</div>
<div id="toggleHide$topicid" style="$hideblock">
  <a class="toggle" href="javascript:;" onmousedown="toggleText('toggleText$topicid','toggleSummary$topicid','toggleShow$topicid','toggleHide$topicid');"><font class="toggle">－</font></a>&nbsp;
  <b><font class="mark">$topicclass</font> <a href="$topichtml">$title</a></b>$bookmark
</div>
<div id="toggleSummary$topicid" style="$showblock">$summary</div>
<div id="toggleText$topicid" style="$hideblock">$text</div>
<div>
[$topicid] Posted by $editor at <font class="datetime">$lastdate</font> $editmenu<br />
<b><font class="point" size="+1">$point</font> point</b> | <a href="$topichtml#link">Link ($linknum)</a> | <a href="$topichtml#trackback">Trackback ($trackbacknum)</a> | <a href="$topichtml#comment">Comment ($commentnum)</a>
</div>
<br />
END_ITEM

    return $item;
}

sub _view_detail() {
    my ($session_userid, $cgi) = @_;

    my $topicid = $cgi->param('id');
    if ($topicid !~ /\A[0-9]+\z/) {
        $error->print_error('00002e');
    }

    my $user = Lib::User->new();
    my $isadmin = 0;
    if ($session_userid ne '') {
        $isadmin = $user->is_admin($session_userid);
    }
    else {
        if ($conf->get_static_html()) {
            my $topicurl = "detail$topicid.html";
            print "Status: 301 Moved Permanently\n";
            print "Location: $topicurl", "\n\n";
            exit;
        }
    }

    my $topic = Lib::Topic->new();
    $topic->load($topicid, 1);

    my $editor = $topic->get_editor();
    my $mode = 0;
    if ($session_userid ne '') {
        $mode = ($isadmin || ($session_userid eq $editor)) ? 1 : 2;
    }

    my $topichtml = $topic->get_html(0, $mode);

    if ($session_userid ne '') {
        my $hpurl = $user->get_hpurl();
        $topichtml =~ s/\$AUTHOR\$/$session_userid/g;
        $topichtml =~ s/\$HPURL\$/$hpurl/g;
    }
    else {
        $topichtml =~ s/\$AUTHOR\$//g;
        $topichtml =~ s/\$HPURL\$//g;
    }

    print "Content-Type: text/html\n\n";
    print $topichtml;
}

# メニューを取得
sub _get_menu {
    my $session_userid = shift;

    my $language = Lib::Language->new();
    my $post = $language->get_statictext('new-thread');
    my $setting = $language->get_statictext('setting');
    my $manage = $language->get_statictext('manage');
    my $logout = $language->get_statictext('logout');

    my $menu = '';
    my $user = Lib::User->new();
    if ($user->is_admin($session_userid)) {
        $menu = <<"END_MENU";
<a class="menu" href="./?act=view_editor">$post</a>&nbsp;
<a class="menu" href="./?act=view_setting">$setting</a>&nbsp;
<a class="menu" href="./?act=view_manage">$manage</a>&nbsp;
<a class="menu" href="./?act=exec_main&func=logout">$logout</a><br />
END_MENU
    }
    else {
        $menu = <<"END_MENU";
<a class="menu" href="./?act=view_editor">$post</a>&nbsp;
<a class="menu" href="./?act=view_setting">$setting</a>&nbsp;
<a class="menu" href="./?act=exec_main&func=logout">$logout</a><br />
END_MENU
    }

    return $menu;
}

1;
