#!/usr/bin/perl
#------------------------------------------------------------------------------
#    59bbs, blog like bulletin board system.
#    Copyright (C) 2007-2009 Kaga, Hiroaki
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#------------------------------------------------------------------------------

use strict;
use warnings;

use Lib::Logger;
use Lib::Error;
use Lib::User;
use Lib::Mail;
use Lib::String;
use Lib::App::Conf;

my $logger = Lib::Logger->new();
my $error = Lib::Error->new();
my $user = Lib::User->new();
my $conf = Lib::App::Conf->new();

sub act() {
    my ($act, $cgi) = @_;

    if ($act eq 'input_register') {
        _input_register();
    }
    elsif ($act eq 'add_register') {
        _add_register($cgi);
    }
    else {
        $error->print_error('00002e');
    }
}

sub _input_register() {

    my $freesignup = $conf->get_free_signup();
    if (!$freesignup) {
        $error->print_error('00004e');
    }

    my $userdata;
    $userdata->{userid} = '';
    $userdata->{mailaddr} = '';

    # 入力画面の表示
    _show_page($userdata, '');
}

sub _add_register() {
    my ($cgi) = @_;

    my $freesignup = $conf->get_free_signup();
    if (!$freesignup) {
        $error->print_error('00004e');
    }

    # パラメーターの取得
    my $userdata;
    $userdata->{userid}   = $cgi->param('userid');
    $userdata->{mailaddr} = $cgi->param('mailaddr');

    my $string = Lib::String->new();
    $userdata->{userid} = $string->tag_invalidate($userdata->{userid});
    $userdata->{mailaddr} = $string->tag_invalidate($userdata->{mailaddr});

    # 入力チェック
    if ($userdata->{userid} eq '') {
        _show_page($userdata, 'ユーザーIDが未入力です');
    }
    # ユーザーIDが英数字かチェック
    if ($userdata->{userid} =~ /[^0-9a-zA-Z]/) {
        _show_page($userdata, 'ユーザーIDは半角英数字で入力してください');
    }
    # ユーザーIDの重複チェック
    if (!$user->check_valid($userdata->{userid})) {
        _show_page($userdata, 'ユーザーIDが既に登録されています');
    }
    # メールアドレス
    if ($userdata->{mailaddr} eq '') {
        _show_page($userdata, 'メールアドレスが未入力です');
    }
    # メールアドレス形式が正しいかチェック
    if ($userdata->{mailaddr} !~ /[-_0-9a-z]+\@[-_0-9a-z]+\.[-_0-9a-z.]+/) {
        _show_page($userdata, 'メールアドレスに誤りがあります');
    }

    # 初期パスワードを生成
    my $password = int(rand(89999)) + 10000;	# 10000 ～ 99999 のパスワードを生成

    my $userid = $userdata->{userid};
    my $mailaddr = $userdata->{mailaddr};
    my $username = $userid;

    # ユーザーを追加
    $user->set_userid($userid);
    $user->set_password($password);
    $user->set_mailaddr($mailaddr);
    $user->set_hpurl('');
    $user->set_username($username);
    $user->set_auth(1);
    $user->set_status(1);
    $user->set_ad1('');
    $user->set_ad2('');
    $user->set_parts('');
    $user->add();

    my $sitetitle = $conf->get_site_title();
    my $mail = Lib::Mail->new();

    # 登録者にメールを送信
    my $subject = "$sitetitle にユーザー登録しました";
    my $body = <<"END_BODY";
$sitetitle にユーザー登録しました。
登録したユーザーID、初期パスワードでログインし、パスワード変更を行なってください。

ユーザーID    ：$userdata->{userid}
初期パスワード：$password
メールアドレス：$userdata->{mailaddr}

END_BODY

    $mail->send($subject, $body, $userdata->{mailaddr});

    # 管理人にメールを送信
    $subject = "$userdata->{userid} さんがユーザー登録しました";
    $body = <<"END_BODY";
$userdata->{userid} さんがユーザー登録しました。

メールアドレス：$userdata->{mailaddr}
END_BODY

    $mail->send($subject, $body, $conf->get_admin_address());

    # ユーザー登録完了メッセージを表示
    my $msg = <<"END_MESSAGE";
$sitetitle へのユーザー登録が完了しました。<br />
登録されたメールアドレスにユーザーID、初期パスワードを送信しました。<br />
ログインして初期パスワードの変更を行なってください。
END_MESSAGE

    _show_message('ユーザー登録完了', $msg);
}

sub _show_page() {
    my ($userdata, $msg) = @_;

    $msg = "<p><strong><font class=\"message\" size=\"+1\">$msg</font></strong></p>";

    my $caption = 'ユーザー登録';

    my $path = <<"END_PATH";
<p><strong>
<a href="./" class="path">ホーム</a>&nbsp;
&gt;&nbsp;
$caption
</strong></p>
END_PATH

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/register.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$CAPTION\$/$caption/g;
    $template =~ s/\$PATH\$/$path/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$USERID\$/$userdata->{userid}/g;
    $template =~ s/\$MAILADDR\$/$userdata->{mailaddr}/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

# メッセージを表示する
sub _show_message() {
    my ($title, $msg) = @_;

#    $logger->write("_show_message");  # for debug

    $msg = "<p><strong><font class=\"message\">$msg</font></strong></p>";
    my $pageexit = "<strong><a href=\"./?act=input_session\">ログイン</a></strong>";

    my $systemdir = $conf->get_system_dir();

    open my $templatefh, '<', "$systemdir/tmpl/message.tmpl";
    my $template = do { local $/; <$templatefh> };
    close $templatefh;

    $template =~ s/\$TITLE\$/$title/g;
    $template =~ s/\$MSG\$/$msg/g;
    $template =~ s/\$PAGEEXIT\$/$pageexit/g;

    print "Content-Type: text/html\n\n";
    print $template;

    exit;
}

1;
