/** @file
 */
#if defined(HAVE_CONFIG_H)
#  include "../config.h"
#endif
#include <list>
#include <iostream>
#include <boost/format.hpp>
#include <gtkmm/liststore.h>
#include <gtkmm/treemodelfilter.h>
#include <gtkmm/treeselection.h>
#include "../gettext.h"
#include <glibmm/i18n.h>
#include <gtk/gtkdialog.h>

#include "gdestraier.hpp"
#include "model/preferences.hpp"
#include "model/encodings.hpp"
#include "index_edit_dialog.hpp"
#include "preferences_dialog.hpp"

#include "glade-helper.hpp"

namespace gdestraier {
  namespace gui {

    namespace {

      /**
       * TreeModelColumnRecord for indexes.
       */
      class index_column_record : public Gtk::TreeModelColumnRecord {
      public:
        Gtk::TreeModelColumn<Glib::ustring> description_;

        Gtk::TreeModelColumn<bool>          active_;
        Gtk::TreeModelColumn<int>           index_location_;
        Gtk::TreeModelColumn<Glib::ustring> index_path_;
        Gtk::TreeModelColumn<Glib::ustring> index_user_;
        Gtk::TreeModelColumn<Glib::ustring> index_password_;

        Gtk::TreeModelColumn<Glib::ustring> document_path_;
        Gtk::TreeModelColumn<gdestraier::model::filetype const*> filetype_;
        Gtk::TreeModelColumn<gdestraier::model::encoding const*> document_encoding_;
        Gtk::TreeModelColumn<Glib::ustring> language_;
        Gtk::TreeModelColumn<bool>          use_Ngram_for_all_languages_;
        Gtk::TreeModelColumn<bool>          include_document_title_to_body_;
        Gtk::TreeModelColumn<bool>          include_document_uri_to_body_;
        Gtk::TreeModelColumn<bool>          is_enable_quick_build_;

        Gtk::TreeModelColumn<Glib::ustring> uri_replace_rule_regex_;
        Gtk::TreeModelColumn<Glib::ustring> uri_replace_rule_to_;
        Gtk::TreeModelColumn<Glib::ustring> title_replace_rule_regex_;
        Gtk::TreeModelColumn<Glib::ustring> title_replace_rule_to_;

        Gtk::TreeModelColumn<gdestraier::model::preferences::indexes_type::iterator> raw_iter_;
        Gtk::TreeModelColumn<bool>          still_exists_;

        Gtk::TreeModelColumn<int>           depth_;
        Gtk::TreeModelColumn<int>           max_documents_;
        Gtk::TreeModelColumn<bool>          use_snippet_;

      protected:
        index_column_record() {
          add(description_);

          add(index_location_); add(index_path_); add(index_user_); add(index_password_);

          add(document_path_);
          add(active_);
          add(filetype_);
          add(document_encoding_); add(language_);
          add(use_Ngram_for_all_languages_);
          add(include_document_title_to_body_);
          add(include_document_uri_to_body_);

          add(is_enable_quick_build_);

          add(uri_replace_rule_regex_); add(uri_replace_rule_to_);
          add(title_replace_rule_regex_); add(title_replace_rule_to_);

          add(depth_); add(max_documents_);
          add(use_snippet_);


          add(raw_iter_);
          add(still_exists_);
        }
      public:
        static index_column_record const& instance() {
          static index_column_record inst;
          return inst;
        }
      };



      
      /**
       * TreeModelColumnRecord for font and colors.
       */
      class font_columns_record : public Gtk::TreeModelColumnRecord {
      public:
        Gtk::TreeModelColumn<Glib::ustring>    name_;
        Gtk::TreeModelColumn<char const*>      target_;
        Gtk::TreeModelColumn<Glib::ustring>    font_name_;
        Gtk::TreeModelColumn<Glib::ustring>    normal_color_;
        Gtk::TreeModelColumn<Glib::ustring>    highlight_color_;

      protected:
        font_columns_record() {
          add(name_); add(target_);
          add(font_name_);
          add(normal_color_); add(highlight_color_);
        }

      public:
        static font_columns_record const& instance() {
          static font_columns_record inst;
          return inst;
        }
      };

    }



    /** @brief Constructor
     *  @param cobject C object
     *  @param glade   refptr of glade XML object.
     */
    preferences_dialog::preferences_dialog(GtkDialog* cobject,
                                           ::GladeXML* glade)
      : Gtk::Dialog(cobject)
    {
      set_default_response(GTK_RESPONSE_CANCEL);
      signal_response().connect(sigc::mem_fun(this, &preferences_dialog::on_response) );

      //
      // Save precious widgets.
      //
      // Indexes
      gdestraier::gui::get_widget(glade, "indexes_view", &indexes_view_);
      gdestraier::gui::get_widget(glade, "edit_index", &edit_index_button_);
      gdestraier::gui::get_widget(glade, "delete_index", &delete_index_button_);
      gdestraier::gui::get_widget(glade, "moveup_index", &moveup_index_button_);
      gdestraier::gui::get_widget(glade, "movedown_index", &movedown_index_button_);

      Gtk::Button* add_index_button;
      gdestraier::gui::get_widget(glade, "add_index", &add_index_button);
      add_index_button->signal_clicked().connect(sigc::mem_fun(this, &preferences_dialog::on_add_index_clicked));

      delete_index_button_->signal_clicked()
        .connect(sigc::mem_fun(this, &preferences_dialog::on_delete_index_clicked));
      edit_index_button_->signal_clicked()
        .connect(sigc::mem_fun(this, &preferences_dialog::on_edit_index_clicked));
      moveup_index_button_->signal_clicked()
        .connect(sigc::mem_fun(this, &preferences_dialog::on_moveup_index_clicked));
      movedown_index_button_->signal_clicked()
        .connect(sigc::mem_fun(this, &preferences_dialog::on_movedown_index_clicked));

      indexes_view_->get_selection()->signal_changed().connect(sigc::mem_fun(this, &preferences_dialog::on_index_row_activated));


      // External commands
      gdestraier::gui::get_widget(glade, "estcmd_path", &estcmd_path_entry_);

      // Font and colors
      gdestraier::gui::get_widget(glade, "color_target_list", &color_target_list_);
      gdestraier::gui::get_widget(glade, "font_chooser", &font_chooser_);
      gdestraier::gui::get_widget(glade, "normal_color_chooser", &normal_color_chooser_);
      gdestraier::gui::get_widget(glade, "highlight_color_chooser", &highlight_color_chooser_);
      gdestraier::gui::get_widget(glade, "font_sample_view", &font_sample_view_);

      color_target_list_->get_selection()->signal_changed()
        .connect(sigc::mem_fun(this, &preferences_dialog::on_color_target_row_activate));
      font_chooser_->signal_font_set().connect(sigc::mem_fun(this, &preferences_dialog::on_font_set) );
      normal_color_chooser_->signal_color_set()
        .connect(sigc::mem_fun(this, &preferences_dialog::on_normal_color_set));
      highlight_color_chooser_->signal_color_set()
        .connect(sigc::mem_fun(this, &preferences_dialog::on_highlight_color_set));


      // Defaults
      gdestraier::gui::get_widget(glade, "default_group_key", &default_group_key_combo_);
      gdestraier::gui::get_widget(glade, "default_sort_key", &default_sort_key_combo_);
      gdestraier::gui::get_widget(glade, "thumbnail_size_button", &thumbnail_size_button_);


      // Network
      gdestraier::gui::get_widget(glade, "network_timeout_spinbutton", &network_timeout_spinbutton_);
      gdestraier::gui::get_widget(glade, "proxy_server_entry", &proxy_server_entry_);
      gdestraier::gui::get_widget(glade, "proxy_port_spinbutton", &proxy_port_spinbutton_);
      gdestraier::gui::get_widget(glade, "use_gnome_proxy_checkbox", &use_gnome_proxy_checkbox_);

      use_gnome_proxy_checkbox_->signal_toggled()
        .connect(sigc::mem_fun(this, &preferences_dialog::on_use_gnome_proxy_toggled));
      


      //
      // Initialize dialog data.
      //
      gdestraier::model::preferences& pref = gdestraier::gui::get_preferences();

      // Indexes
      {
        index_column_record const& columns = index_column_record::instance();
        indexes_model_ = Gtk::ListStore::create(columns);

        Gtk::TreePath filtered_root;   // Compatibility: 古いGtkmmではTreeModelFilter::createの第２引数を省略できない。
        indexes_model_filter_ = Gtk::TreeModelFilter::create(indexes_model_, filtered_root);
        indexes_model_filter_->set_visible_column(columns.still_exists_);

        indexes_view_->set_model(indexes_model_filter_);
        indexes_view_->append_column_editable("Active", columns.active_);
        indexes_view_->append_column("Desc.", columns.description_);
        indexes_view_->append_column("Path", columns.index_path_);

        // Append indexes
        for (gdestraier::model::preferences::indexes_type::iterator i = pref.indexes_.begin();
             i != pref.indexes_.end(); i++) {
          Gtk::TreeRow row = *(indexes_model_->append());
          row[columns.description_] = i->description_.c_str();

          row[columns.active_] = i->active_;
          row[columns.index_path_] = i->database_path_.c_str();
          row[columns.index_location_] = i->database_location_;
          row[columns.index_user_] = i->user_;
          row[columns.index_password_] = i->password_;

          row[columns.document_path_] = i->document_path_.c_str();
          row[columns.filetype_] = i->filetype_;
          row[columns.document_encoding_] = i->document_encoding_;
          row[columns.language_] = i->language_;
          row[columns.use_Ngram_for_all_languages_] = i->use_Ngram_for_all_languages_;
          row[columns.include_document_title_to_body_] = i->include_title_to_body_;
          row[columns.include_document_uri_to_body_] = i->include_uri_to_body_;

          row[columns.is_enable_quick_build_] = i->is_enable_quick_build_;

          row[columns.uri_replace_rule_regex_] = i->get_uri_replace_regex();
          row[columns.uri_replace_rule_to_]    = i->get_uri_replace_to();
          row[columns.title_replace_rule_regex_] = i->get_title_replace_regex();
          row[columns.title_replace_rule_to_]    = i->get_title_replace_to();

          row[columns.depth_] = i->get_depth();
          row[columns.max_documents_] = i->get_max_documents();
          row[columns.use_snippet_] = i->get_use_snippet();

          row[columns.raw_iter_] = i;
          row[columns.still_exists_] = true;
        }
      }

      // External commands
      estcmd_path_entry_->set_text(pref.estcmd_path_);


      // Font and colors
      {
        font_chooser_->set_sensitive(false);
        normal_color_chooser_->set_sensitive(false);
        highlight_color_chooser_->set_sensitive(false);


        font_columns_record const& cols = font_columns_record::instance();
        Glib::RefPtr<Gtk::ListStore> model = Gtk::ListStore::create(cols);

        struct targets_type {
          char const* target_;
          char const* name_;
        };
        static targets_type targets[] = {
          { "background",    N_("Background") },
          { "group-header",  N_("Group header") },
          { "title",         N_("Title") },
          { "score",         N_("Score") },
          { "file-type",     N_("File type") },
          { "author",        N_("Author") },
          { "last-modified", N_("Last modified") },
          { "language",      N_("Language") },
          { "index",         N_("Index") },
          { "keyword-1",     N_("Keyword 1") },
          { "keyword-2",     N_("Keyword 2") },
          { "keyword-3",     N_("Keyword 3") },
          { 0, 0 }
        };
        for (targets_type* p = targets; p->target_ != 0; p++) {
          gdestraier::model::font_and_color* fac = pref.get_font_and_color(p->target_);
          if (fac == 0) continue;

          Gtk::TreeModel::iterator i = model->append();
          i->set_value(cols.name_,         Glib::ustring(::gettext(p->name_)));
          i->set_value(cols.target_,       p->target_);
          i->set_value(cols.font_name_,    fac->font_name_);
          i->set_value(cols.normal_color_, fac->normal_color_);
          i->set_value(cols.highlight_color_, fac->highlight_color_);
        }

        color_target_list_->set_model(model);
        color_target_list_->append_column("", cols.name_);
      }


      // Defaults
      switch (pref.group_key_) {
      default:
      case gdestraier::model::preferences::GROUP_BY_INDEX:
        default_group_key_combo_->set_active(0);
        break;
      case gdestraier::model::preferences::GROUP_BY_MIME_TYPE:
        default_group_key_combo_->set_active(1);
        break;
      case gdestraier::model::preferences::GROUP_BY_AUTHOR:
        default_group_key_combo_->set_active(2);
        break;
      case gdestraier::model::preferences::GROUP_BY_LANGUAGE:
        default_group_key_combo_->set_active(3);
        break;
      case gdestraier::model::preferences::GROUP_BY_LAST_MODIFIED:
        default_group_key_combo_->set_active(4);
        break;
      }


      switch (pref.sort_key_) {
      default:
      case gdestraier::model::preferences::SORT_BY_SCORE:
        default_sort_key_combo_->set_active(0);
        break;
      case gdestraier::model::preferences::SORT_BY_LAST_MODIFIED:
        default_sort_key_combo_->set_active(1);
        break;
      case gdestraier::model::preferences::SORT_BY_TITLE:
        default_sort_key_combo_->set_active(2);
        break;
      }


      thumbnail_size_button_->set_value(pref.thumbnail_size_);


      // Network
      network_timeout_spinbutton_->set_value(pref.network_.timeout_);
      proxy_server_entry_->set_text(pref.network_.proxy_host_);
      proxy_port_spinbutton_->set_value(pref.network_.proxy_port_);
      use_gnome_proxy_checkbox_->set_active(pref.network_.use_gnome_proxy_settings_);
    }



    preferences_dialog::~preferences_dialog()
    {
    }



    /** @brief インデックス一覧の選択状態が変化しました。
     */
    void
    preferences_dialog::on_index_row_activated()
    {
      bool row_selected = indexes_view_->get_selection()->count_selected_rows() > 0;

      edit_index_button_->set_sensitive(row_selected);
      delete_index_button_->set_sensitive(row_selected);
      moveup_index_button_->set_sensitive(row_selected);
      movedown_index_button_->set_sensitive(row_selected);
    }





    /** @brief ダイアログが終了しようとしています
     */
    void preferences_dialog::on_response(int response_id)
    {
      index_column_record const& columns = index_column_record::instance();

      if (response_id != GTK_RESPONSE_OK) return ;


      // ダイアログデータをモデルへ保存します
      gdestraier::model::preferences& pref = gdestraier::gui::get_preferences();

      //
      // インデックスの設定を保存します
      //
      {
        Gtk::TreeNodeChildren rows = indexes_model_->children();

        for (Gtk::TreeNodeChildren::iterator i = rows.begin(); i != rows.end(); i++) {
          gdestraier::model::preferences::indexes_type::iterator ii =
            i->get_value(columns.raw_iter_);

          if (! i->get_value(columns.still_exists_)) {  // エントリは削除された
            if (ii != pref.indexes_.end()) // 元々存在していたなら削除する
              pref.indexes_.erase(ii);
          } else {
            if (ii == pref.indexes_.end()) {
              pref.indexes_.push_back(gdestraier::model::index_type());
              ii = pref.indexes_.end();
              ii--;
            }
            ii->description_   = i->get_value(columns.description_);
            ii->database_location_ = i->get_value(columns.index_location_);
            ii->database_path_ = i->get_value(columns.index_path_);
            ii->user_          = i->get_value(columns.index_user_);
            ii->password_      = i->get_value(columns.index_password_);
            ii->active_        = i->get_value(columns.active_);

            ii->document_path_ = i->get_value(columns.document_path_);
            ii->filetype_      = i->get_value(columns.filetype_);
            ii->document_encoding_ = i->get_value(columns.document_encoding_);
            ii->language_      = i->get_value(columns.language_);
            ii->use_Ngram_for_all_languages_ =
              i->get_value(columns.use_Ngram_for_all_languages_);
            ii->include_title_to_body_ =
              i->get_value(columns.include_document_title_to_body_);
            ii->include_uri_to_body_ =
              i->get_value(columns.include_document_uri_to_body_);

            ii->is_enable_quick_build_ = i->get_value(columns.is_enable_quick_build_);

            ii->set_uri_replace(i->get_value(columns.uri_replace_rule_regex_).c_str(),
                                i->get_value(columns.uri_replace_rule_to_).c_str() );
            ii->set_title_replace(i->get_value(columns.title_replace_rule_regex_).c_str(),
                                  i->get_value(columns.title_replace_rule_to_).c_str() );

            ii->set_depth(i->get_value(columns.depth_));
            ii->set_max_documents(i->get_value(columns.max_documents_));
            ii->set_use_snippet(i->get_value(columns.use_snippet_));
          }
        }

        //
        // 外部コマンドの設定を保存します
        //
        pref.estcmd_path_ = estcmd_path_entry_->get_text();


        //
        // Save font and color settings
        //
        {
          font_columns_record const& cols = font_columns_record::instance();
          Gtk::TreeNodeChildren rows = color_target_list_->get_model()->children();

          for (Gtk::TreeNodeChildren::iterator i = rows.begin(); i != rows.end(); i++) {
            std::string target_name = i->get_value(cols.target_);
            gdestraier::model::font_and_color* fc = pref.get_font_and_color(target_name.c_str());
            if (fc != 0) {
              fc->font_name_ = i->get_value(cols.font_name_);
              fc->normal_color_ = i->get_value(cols.normal_color_);
              fc->highlight_color_ = i->get_value(cols.highlight_color_);
            }
          }
        }



        // 
        // Save defaults settings
        //
        switch (default_group_key_combo_->get_active_row_number()) {
        default:
        case 0: pref.group_key_ = gdestraier::model::preferences::GROUP_BY_INDEX; break;
        case 1: pref.group_key_ = gdestraier::model::preferences::GROUP_BY_MIME_TYPE; break;
        case 2: pref.group_key_ = gdestraier::model::preferences::GROUP_BY_AUTHOR; break;
        case 3: pref.group_key_ = gdestraier::model::preferences::GROUP_BY_LANGUAGE; break;
        case 4: pref.group_key_ = gdestraier::model::preferences::GROUP_BY_LAST_MODIFIED; break;
        }

        switch (default_sort_key_combo_->get_active_row_number()) {
        default:
        case 0: pref.sort_key_ = gdestraier::model::preferences::SORT_BY_SCORE; break;
        case 1: pref.sort_key_ = gdestraier::model::preferences::SORT_BY_LAST_MODIFIED; break;
        case 2: pref.sort_key_ = gdestraier::model::preferences::SORT_BY_TITLE; break;
        }

        pref.thumbnail_size_ = thumbnail_size_button_->get_value_as_int();


        //
        // Save network settings
        //
        pref.network_.timeout_ = network_timeout_spinbutton_->get_value_as_int();
        pref.network_.proxy_host_ = proxy_server_entry_->get_text();
        pref.network_.proxy_port_ = proxy_port_spinbutton_->get_value_as_int();
        pref.network_.use_gnome_proxy_settings_ = use_gnome_proxy_checkbox_->get_active();
      }

      pref.save();
    }





    /** @brief [indexes -> Add ]
     */
    void preferences_dialog::on_add_index_clicked()
    {
      index_edit_dialog* dlg;
      gdestraier::gui::load_toplevel_widget_derived(&dlg);

      int response;
      while ((response = dlg->run()) == 0 || dlg->has_error())
        ;

      if (response == GTK_RESPONSE_OK) {
        gdestraier::model::preferences& pref = gdestraier::gui::get_preferences();

        index_column_record const& columns = index_column_record::instance();
        Gtk::ListStore::iterator row = indexes_model_->append();

        row->set_value(columns.active_, dlg->get_active());
        row->set_value(columns.description_, dlg->get_description());
        row->set_value(columns.index_location_, dlg->get_index_location());
        row->set_value(columns.index_path_,  dlg->get_index_path());
        row->set_value(columns.index_user_, dlg->get_index_user());
        row->set_value(columns.index_password_, dlg->get_index_password());

        row->set_value(columns.document_path_, dlg->get_document_path());
        row->set_value(columns.filetype_, dlg->get_filetype());
        row->set_value(columns.document_encoding_, dlg->get_document_encoding());
        row->set_value(columns.language_, dlg->get_language());
        row->set_value(columns.use_Ngram_for_all_languages_,  dlg->get_use_Ngram_for_all_languages());
        row->set_value(columns.include_document_title_to_body_,  dlg->get_include_document_title_to_body());
        row->set_value(columns.include_document_uri_to_body_,  dlg->get_include_document_uri_to_body());

        row->set_value(columns.is_enable_quick_build_, dlg->is_enable_quick_build());

        row->set_value(columns.uri_replace_rule_regex_, dlg->get_uri_replace_rule_regex());
        row->set_value(columns.uri_replace_rule_to_,    dlg->get_uri_replace_rule_to());
        row->set_value(columns.title_replace_rule_regex_, dlg->get_title_replace_rule_regex());
        row->set_value(columns.title_replace_rule_to_,    dlg->get_title_replace_rule_to());

        row->set_value(columns.depth_, dlg->get_depth());
        row->set_value(columns.max_documents_, dlg->get_max_documents());
        row->set_value(columns.use_snippet_, dlg->get_use_snippet());

        row->set_value(columns.raw_iter_, pref.indexes_.end());
        row->set_value(columns.still_exists_, true);
      }

      delete dlg;
    }



    /** @brief [indexes -> delete ]
     */
    void preferences_dialog::on_delete_index_clicked()
    {
      index_column_record const& columns = index_column_record::instance();
      Glib::RefPtr<Gtk::TreeSelection> sel = indexes_view_->get_selection();
      Gtk::TreeModel::iterator i = sel->get_selected();
      if (bool(i))  {
        Gtk::TreeModel::iterator ii = indexes_model_filter_->convert_iter_to_child_iter(i);
        ii->set_value(columns.still_exists_, false);
      }
    }


    /** @brief [indexes -> edit ]
     */
    void preferences_dialog::on_edit_index_clicked()
    {
      Glib::RefPtr<Gtk::TreeSelection> selection = indexes_view_->get_selection();
      Gtk::TreeModel::iterator i = indexes_model_filter_->convert_iter_to_child_iter(selection->get_selected());

      index_edit_dialog* dlg;
      gdestraier::gui::load_toplevel_widget_derived(&dlg);

      index_column_record const& columns = index_column_record::instance();
      dlg->set_active(i->get_value(columns.active_));
      dlg->set_description(i->get_value(columns.description_));
      dlg->set_index_location(i->get_value(columns.index_location_));
      dlg->set_index_path(i->get_value(columns.index_path_));
      dlg->set_index_user(i->get_value(columns.index_user_));
      dlg->set_index_password(i->get_value(columns.index_password_));

      dlg->set_document_path(i->get_value(columns.document_path_));
      dlg->set_filetype(i->get_value(columns.filetype_));
      dlg->set_document_encoding(i->get_value(columns.document_encoding_));
      dlg->set_language(i->get_value(columns.language_));
      dlg->set_use_Ngram_for_all_languages(i->get_value(columns.use_Ngram_for_all_languages_));
      dlg->set_include_document_title_to_body(i->get_value(columns.include_document_title_to_body_));
      dlg->set_include_document_uri_to_body(i->get_value(columns.include_document_uri_to_body_));

      dlg->set_enable_quick_build(i->get_value(columns.is_enable_quick_build_));

      dlg->set_uri_replace_rule_regex(i->get_value(columns.uri_replace_rule_regex_));
      dlg->set_uri_replace_rule_to(i->get_value(columns.uri_replace_rule_to_));
      dlg->set_title_replace_rule_regex(i->get_value(columns.title_replace_rule_regex_));
      dlg->set_title_replace_rule_to(i->get_value(columns.title_replace_rule_to_));

      dlg->set_depth(i->get_value(columns.depth_));
      dlg->set_max_documents(i->get_value(columns.max_documents_));
      dlg->set_use_snippet(i->get_value(columns.use_snippet_));

      int response;
      while ((response = dlg->run()) == 0 || dlg->has_error())
        ;

      if (response == GTK_RESPONSE_OK) {
        i->set_value(columns.active_, dlg->get_active());
        i->set_value(columns.description_, dlg->get_description());
        i->set_value(columns.index_location_, dlg->get_index_location());
        i->set_value(columns.index_path_, dlg->get_index_path());
        i->set_value(columns.index_user_, dlg->get_index_user());
        i->set_value(columns.index_password_, dlg->get_index_password());

        i->set_value(columns.document_path_, dlg->get_document_path());
        i->set_value(columns.filetype_, dlg->get_filetype());
        i->set_value(columns.document_encoding_, dlg->get_document_encoding());
        i->set_value(columns.language_, dlg->get_language());
        i->set_value(columns.use_Ngram_for_all_languages_,
                     dlg->get_use_Ngram_for_all_languages());
        i->set_value(columns.include_document_title_to_body_,
                     dlg->get_include_document_title_to_body());
        i->set_value(columns.include_document_uri_to_body_,
                     dlg->get_include_document_uri_to_body());

        i->set_value(columns.is_enable_quick_build_, dlg->is_enable_quick_build());

        i->set_value(columns.uri_replace_rule_regex_, dlg->get_uri_replace_rule_regex());
        i->set_value(columns.uri_replace_rule_to_, dlg->get_uri_replace_rule_to());
        i->set_value(columns.title_replace_rule_regex_, dlg->get_title_replace_rule_regex());
        i->set_value(columns.title_replace_rule_to_, dlg->get_title_replace_rule_to());

        i->set_value(columns.depth_, dlg->get_depth());
        i->set_value(columns.max_documents_, dlg->get_max_documents());
        i->set_value(columns.use_snippet_, dlg->get_use_snippet());
      }


      delete dlg;
    }


    /** @brief [ indexes -> move up ]
     */
    void preferences_dialog::on_moveup_index_clicked()
    {
      /// @TODO
    }

    /** @brief [ indexes -> move down ]
     */
    void preferences_dialog::on_movedown_index_clicked()
    {
      /// @TODO
    }






    void preferences_dialog::on_color_target_row_activate()
    {
      Glib::RefPtr<Gtk::TreeModel> model = color_target_list_->get_model();
      Gtk::ListStore::iterator i = color_target_list_->get_selection()->get_selected();

      if (i == model->children().end()) {
        font_chooser_->set_sensitive(false);
        normal_color_chooser_->set_sensitive(false);
        highlight_color_chooser_->set_sensitive(false);
      } else {
        font_columns_record const& cols = font_columns_record::instance();

        font_chooser_->set_font_name(i->get_value(cols.font_name_));
        normal_color_chooser_->set_color(Gdk::Color(i->get_value(cols.normal_color_)) );
        highlight_color_chooser_->set_color(Gdk::Color(i->get_value(cols.highlight_color_)) );

        font_chooser_->set_sensitive(true);
        normal_color_chooser_->set_sensitive(true);
        highlight_color_chooser_->set_sensitive(true);
      }
    }


    void preferences_dialog::on_font_set()
    {
      Gtk::ListStore::iterator i = color_target_list_->get_selection()->get_selected();
      if (i != color_target_list_->get_model()->children().end())
        i->set_value(font_columns_record::instance().font_name_, font_chooser_->get_font_name());
    }


    void preferences_dialog::on_normal_color_set()
    {
      Gtk::ListStore::iterator i = color_target_list_->get_selection()->get_selected();
      if (i != color_target_list_->get_model()->children().end()) {
        Gdk::Color c = normal_color_chooser_->get_color();
        i->set_value(font_columns_record::instance().normal_color_, 
                     Glib::ustring((boost::format("#%02x%02x%02x") % (c.get_red() / 256) % (c.get_green() / 256) % (c.get_blue() / 256)).str()));
      }
    }


    void preferences_dialog::on_highlight_color_set()
    {
      Gtk::ListStore::iterator i = color_target_list_->get_selection()->get_selected();
      if (i != color_target_list_->get_model()->children().end()) {
        Gdk::Color c = highlight_color_chooser_->get_color();
        i->set_value(font_columns_record::instance().highlight_color_, 
                     Glib::ustring((boost::format("#%02x%02x%02x") % (c.get_red() /256) % (c.get_green() /256) % (c.get_blue() / 256)).str()));
      }
    }


    void
    preferences_dialog::on_use_gnome_proxy_toggled()
    {
      bool b = use_gnome_proxy_checkbox_->get_active();
      proxy_server_entry_->set_sensitive(! b);
      proxy_port_spinbutton_->set_sensitive(! b);
    }

  }
}
