/** @file
 */
#if !defined(__HYPERESTRAIER__NODE__HPP__)
#define __HYPERESTRAIER__NODE__HPP__

#define GNU_SOURCE
#include <cstring>

#include <boost/noncopyable.hpp>

#include <estraier.h>
#include <estnode.h>

#include "wrapper.hpp"
#include "doc.hpp"
#include "db.hpp"


namespace hyperestraier {

  namespace detail {
    template <> inline void raw_handle_deleter<ESTNODE>(ESTNODE* p) { ::est_node_delete(p); }
    template <> inline void raw_handle_deleter<ESTRESDOC>(ESTRESDOC*) { }
    template <> inline void raw_handle_deleter<ESTNODERES>(ESTNODERES* p) { ::est_noderes_delete(p); }
  }




  /** @brief ノード上にあるリモート文書
   *
   */
  class node_document : 
    public handle_wrapper<ESTRESDOC, node_document>,
    public document
  {
  public:
    node_document() { }
    explicit node_document(ESTRESDOC* raw_handle) : handle_wrapper<ESTRESDOC, node_document>(raw_handle) { }
    virtual ~node_document() { }

    virtual char const* get_uri() const { return ::est_resdoc_uri(raw_handle_); }
    virtual ::CBLIST* attr_names() const { return ::est_resdoc_attr_names(raw_handle_); }
    virtual char const* get_attr(char const* name, char const* dft) const {
      char const* v = ::est_resdoc_attr(raw_handle_, name);
      return (v == 0) ? dft : v;
    }

    /* @brief 文書テキストの要約を作成します
     * @returns 要約文字列。破棄する必要はありません。
     *
     * 要約文字列はタブで分割された文字列です。各々の行は表示する為の語で、ほとんどは1フィールドのみですが、2つの
     * フィールドを持っている場合があります。
     * 2つ目のフィールドが存在した場合、最初のフィールドはハイライト表示する為の語で、2番目は正規化されている語です。
     */
    inline char const* make_snippet() const { return ::est_resdoc_snippet(raw_handle_); }
  };



  class node_result : 
    public handle_wrapper<ESTNODERES, node_result>
  {
  public:
    node_result() : handle_wrapper<ESTNODERES, node_result>() { }
    explicit node_result(ESTNODERES* raw_handle) :
      handle_wrapper<ESTNODERES, node_result>(raw_handle) { }
    ~node_result() { clear(); }

    ::CBMAP* hints() const { return ::est_noderes_hints(raw_handle_); }
    int      doc_num() const { return ::est_noderes_doc_num(raw_handle_); }

    ESTRESDOC* get_doc(int index) const { return ::est_noderes_get_doc(raw_handle_, index); }
  };




  /** @brief ノード上にあるデータベース
   */
  class node_database :
    public handle_wrapper<ESTNODE, node_database>,
    public database
  {
  private:
    class network_environment {
    public:
      network_environment() { ::est_init_net_env(); }
      ~network_environment() { ::est_free_net_env(); }
    };

    static network_environment& use_network_environment() {
      static network_environment env;
      return env;
    }

  public:
    node_database() : handle_wrapper<ESTNODE, node_database>(), database() {
      use_network_environment();
    }
    node_database(char const* uri) { use_network_environment(); open(uri); }
    explicit node_database(ESTNODE* raw_handle) : handle_wrapper<ESTNODE, node_database>(raw_handle), database() { }
    ~node_database() { }


    inline void open(char const* url) { set_raw_handle(::est_node_new(url)); }
    inline void close() { clear(); }
    inline void set_proxy(char const* host, int port) { ::est_node_set_proxy(this->raw_handle_, host, port); }
    inline void set_timeout(int sec) { ::est_node_set_timeout(this->raw_handle_, sec); }
    inline void set_auth(char const* name, char const* passwd) { ::est_node_set_auth(this->raw_handle_, name, passwd); }

    inline int status() const { return ::est_node_status(raw_handle_); }
    inline char const* label() const { return ::est_node_label(raw_handle_); }

    inline ESTNODERES* search(ESTCOND* cond, int depth) { return ::est_node_search(raw_handle_, cond, depth); }
    inline ESTNODERES* search(condition const& cond, int depth) { return search(cond.raw_handle(), depth); }

    inline bool set_user(char const* name, int mode) { return ::est_node_set_user(raw_handle_, name, mode)? true : false; }
    inline bool set_link(char const* uri, char const* label, int credit) {
      return ::est_node_set_link(raw_handle_, uri, label, credit)? true : false;
    }


    inline void     set_snippet_width(int wwidth, int hwidth, int awidth) {
      ::est_node_set_snippet_width(raw_handle_, wwidth, hwidth, awidth);
    }

    //
    // Override
    //
    virtual bool    put_doc(ESTDOC* doc, int /*options*/) { return ::est_node_put_doc(raw_handle_, doc)? true : false; }
    virtual bool    put_doc(local_document& doc, int /*options*/) { return ::est_node_put_doc(raw_handle_, doc.raw_handle())? true : false; }
    virtual bool    out_doc(int id, int /*options*/) { return ::est_node_out_doc(raw_handle_, id)? true : false; }
    virtual bool    out_doc(char const* uri, int /*options*/) { return ::est_node_out_doc_by_uri(raw_handle_, uri)? true : false; }
    virtual ESTDOC* get_doc(int id, int /*options*/) const { return ::est_node_get_doc(raw_handle_, id); }
    virtual ESTDOC* get_doc(char const* uri, int /*options*/) const { return ::est_node_get_doc_by_uri(raw_handle_, uri); }
    virtual char*   get_doc_attr(int id, char const* name, char const* dft) const {
      char* v = ::est_node_get_doc_attr(raw_handle_, id, name);
      return v? v : (dft? ::strdup(dft) : 0);
    }
    virtual char*   get_doc_attr(char const* uri, char const* name, char const* dft) const {
      char* v = ::est_node_get_doc_attr_by_uri(raw_handle_, uri, name);
      return v? v : (dft? ::strdup(dft) : 0);
    }
    virtual int     uri_to_id(char const* uri) const { return ::est_node_uri_to_id(raw_handle_, uri); }
    virtual char const* name() const { return ::est_node_name(raw_handle_); }
    virtual int     doc_num() const { return ::est_node_doc_num(raw_handle_); }
    virtual int     word_num() const { return ::est_node_word_num(raw_handle_); }
    virtual double  size() const { return ::est_node_size(raw_handle_); }
    virtual bool    iter_init() { return ::est_node_iter_init(raw_handle_)? true : false; }
    virtual ESTDOC* iter_next() { return ::est_node_iter_next(raw_handle_); }
  };
}


#endif
