/** @file
 */
#if !defined(__HYPERESTRAIER__LOCAL__HPP__)
#define __HYPERESTRAIER__LOCAL__HPP__

#include <cstdlib>
#include <boost/function.hpp>
#include <boost/iterator/reverse_iterator.hpp>
#include <boost/assert.hpp>
#include <boost/operators.hpp>
#include <boost/lambda/algorithm.hpp>

#include <estraier.h>
#if defined(HAVE_LIBPTHREAD)
#  include <estmtdb.h>
#endif
#include <estnode.h>

#include "wrapper.hpp"
#include "db.hpp"

namespace hyperestraier {


  /** @brief データベースに発生したイベントを受け取るコールバック型
   *  @param message イベントの内容を表すメッセージ文字列
   */
  typedef void (*informer_type)(char const* message);

  namespace detail {

    /** @brief 任意の型の配列へのポインタとサイズをラップするコレクション
     *
     * コピーを作る事はできないので、必要な場合は明示的にコピーする必要があります。
     * \code
     * arraywrapper<T> y(x.size());
     * std::copy(x.begin(), x.end(), y.begin());
     * \endcode
     */
    template <typename T>
    class arraywrapper :
      public boost::noncopyable,
      public boost::less_than_comparable<arraywrapper<T> >,
      public boost::equality_comparable<arraywrapper<T> >
    {
    public:
      typedef T             value_type;
      typedef T&            reference;
      typedef T const&      const_reference;
      typedef T*            iterator;
      typedef T const*      const_iterator;
      typedef std::size_t   size_type;
      typedef std::ptrdiff_t difference_type;

      typedef boost::reverse_iterator<iterator> reverse_iterator;
      typedef boost::reverse_iterator<const_iterator> const_reverse_iterator;

    protected:
      value_type*  p_;
      size_type    size_;

    public:
      explicit arraywrapper() : p_(0), size_(0) { }
      explicit arraywrapper(size_type size) { p_ = std::malloc(size); size_ = size; }
      explicit arraywrapper(value_type* p, size_type size) : p_(p), size_(size) { }
      explicit arraywrapper(std::pair<value_type*, size_type> const& src) :
        p_(src.first), size_(src.second) { }
      virtual ~arraywrapper() { if (p_ != 0) std::free(p_); }


      inline iterator begin() { return p_; }
      inline const_iterator begin() const { return p_; }
      inline iterator end() { return p_ + size_; }
      inline const_iterator end() const { return p_ + size_; }
      reverse_iterator rbegin() { return reverse_iterator(end()); }
      const_reverse_iterator rbegin() const { return const_reverse_iterator(end()); }
      reverse_iterator rend() { return reverse_iterator(begin()); }
      const_reverse_iterator rend() const { return reverse_iterator(begin()); }

      inline reference operator[](size_type i) {
        BOOST_ASSERT(i >= 0 && i < size_ && "out of range");
        return p_[i];
      }
      inline const_reference operator[](size_type i) const {

        BOOST_ASSERT(i >= 0 && i < size_ && "out of range");
        return p_[i];
      }

      inline reference at(size_type i) { rangecheck(i); return p_[i]; }
      inline const_reference at(size_type i) const { rangecheck(i); return p_[i]; }

      inline reference front() { return p_[0]; }
      inline const_reference front() const { return p_[0]; }
      inline reference back() { return p_[size_ - 1]; }
      inline const_reference back() const { return p_[size_ -1]; }

      inline size_type size() const { return size_; }
      inline bool empty() const { return size_ == 0; }
      inline size_type max_size() const { return size_; }

      /** @brief swapはポインタの交換だけですませる
          @param y 交換対象
       */
      void swap(arraywrapper<T>& y) {
        T* tmpptr = p_;
        size_type tmpsize = size_;
        p_ = y.p_;
        size_ = y.size_;
        y.p_ = tmpptr;
        y.size_ = tmpsize;
      }

      const T* data() const { return p_; }
      T* c_array() const { return p_; }

      arraywrapper<value_type>& operator =(std::pair<value_type*, int> r) {
        if (p_ != 0) std::free(p_);
        p_ = r.first;
        size_ = r.second;
        return *this;
      }

      void assign(T const& value) { std::fill_n(begin(), size(), value); }

      void rangecheck(size_type i) {
        if (i < 0 || i >= size())
          throw std::range_error("arraywrapper<>: index out of range");
      }

      //bool operator <(arraywrapper<T> const& y) {
      //  return boost::lambda::std::lexicographical_compare(begin(), end(), y.begin(), y.end());
      //}
      bool operator ==(arraywrapper<T> const& y) {
        if (y.size() != size()) return false;
        std::equal(begin(), end(), y.begin());
      }
    };


    template <> inline void raw_handle_deleter<ESTDB>(ESTDB* p) { int ecode = ESTENOERR; ::est_db_close(p, &ecode); }

#if defined(HAVE_LIBPTHREAD)
    template <> inline void raw_handle_deleter<ESTMTDB>(ESTMTDB* p) { int ecode = ESTENOERR; ::est_mtdb_close(p, &ecode); }
#endif

  }

  /** @brief 文書のキーワードベクトルを生成する為のコールバック型
   *  @param db データベースハンドル
   *  @param id 文書ID
   *  @param user_data ユーザデータ
   *  @returns etch_docの戻り値の仕様に準拠した新規のマップオブジェクト
   *
   *  @TODO C++向けにラップする事
   */
  typedef CBMAP* (*vectorizer_type)(void* db, int id, void* user_data);


#define RT(NAME, THREADING) return est_ ## THREADING ## _ ## NAME
#define ESTDBAPIS(THREADING) \
    static inline estdb_type* open(const char* name, int omode, int* ecp) { RT(open, THREADING)(name, omode, ecp); } \
    static inline int         close(estdb_type* db, int* ecp) { RT(close, THREADING)(db, ecp); } \
    static inline int         error(estdb_type* db) { RT(error, THREADING)(db); } \
    static inline int         fatal(estdb_type* db) { RT(fatal, THREADING)(db); } \
    static inline int         flush(estdb_type* db, int max) { RT(flush, THREADING)(db, max); } \
    static inline int         sync(estdb_type* db) { RT(sync, THREADING)(db); } \
    static inline int         optimize(estdb_type* db, int options) { RT(optimize, THREADING)(db, options); } \
    static inline int         put_doc(estdb_type* db, ESTDOC* doc, int options) { RT(put_doc, THREADING)(db, doc, options); } \
    static inline int         out_doc(estdb_type* db, int id, int options) { RT(out_doc, THREADING)(db, id, options); } \
    static inline ESTDOC*     get_doc(estdb_type* db, int id, int options) { RT(get_doc, THREADING)(db, id, options); } \
    static inline int         uri_to_id(estdb_type* db, char const* uri) { RT(uri_to_id, THREADING)(db, uri); } \
    static inline CBMAP*      etch_doc(estdb_type* db, ESTDOC* doc, int max) { RT(etch_doc, THREADING)(db, doc, max); } \
    static inline int         iter_init(estdb_type* db, char const* prev) { RT(iter_init, THREADING)(db, prev); } \
    static inline int         iter_next(estdb_type* db) { RT(iter_next, THREADING)(db); } \
    static inline char const* name(estdb_type* db) { RT(name, THREADING)(db); } \
    static inline int         doc_num(estdb_type* db) { RT(doc_num, THREADING)(db); } \
    static inline int         word_num(estdb_type* db) { RT(word_num, THREADING)(db); } \
    static inline double      size(estdb_type* db) { RT(size, THREADING)(db); } \
    static inline int*        search(estdb_type* db, ESTCOND* cond, int* nump, CBMAP* hints) { RT(search, THREADING)(db, cond, nump, hints); } \
    static inline void        set_cache_size(estdb_type* db, size_t size, int anum, int tnum, int rnum) { RT(set_cache_size, THREADING)(db, size, anum, tnum, rnum); } \
    static inline void        set_informer(estdb_type* db, informer_type f) { RT(set_informer, THREADING)(db, f); } \
    static inline void        set_vectorizer(estdb_type* db, vectorizer_type f, void* user_data) { RT(set_vectorizer, THREADING)(db, f, user_data); } \
    static inline int         scan_doc(estdb_type* db, ESTDOC* doc, ESTCOND* cond) { RT(scan_doc, THREADING)(db, doc, cond); }


  struct single_threaded
  {
    struct estdbapi {
      typedef ESTDB   estdb_type;
      ESTDBAPIS(db);
    };
  };
  
#if defined(HAVE_LIBPTHREAD)
  struct multi_threaded {
    struct estdbapi {
      typedef ESTMTDB   estdb_type;
      ESTDBAPIS(mtdb);
    };
  };
#endif


#undef RT
#undef ESTDBAPIS




  template <typename THREADING = single_threaded>
  class local_database : 
    public handle_wrapper<typename THREADING::estdbapi::estdb_type, local_database<THREADING> >,
    public THREADING,
    public database
  {
  public:
    typedef THREADING                                       threading_type;
    typedef typename threading_type::estdbapi               estdbapi;
    typedef typename threading_type::estdbapi::estdb_type   estdb_type;
    typedef handle_wrapper<estdb_type, local_database<THREADING> > handle_wrapper_type;
    
    typedef boost::function<CBMAP* (local_database<THREADING>* db, int id)> vectorizer_type;

    typedef detail::arraywrapper<int>                search_result_type;

  protected:
    int last_error_;
    
  public:
    /** @brief デフォルトコンストラクタ
     * 
     *  構築直後はデータベースはオープンされていないので、別途openする必要があります。
     */
    local_database() :
      handle_wrapper_type(),
      database(),
      last_error_(ESTENOERR)
    {
    }
    


    /** @brief インスタンスを構築すると同時にデータベースをオープンします
     *  @param name データベースの名前
     *  @param error エラーコードを受け取る変数へのポインタ
     *
     *  オープンに成功した場合には*errorがESTENOERRになり、失敗した場合には
     *  エラーコードが格納されます。
     */
    local_database(char const* name, int omode = ESTDBREADER) :
      handle_wrapper_type(),
      database(),
      last_error_(ESTENOERR)
    {
      open(name, omode);
    }
    
    
    /** @brief デストラクタ
     *  開いているデータベースは自動的にクローズします。
     *  @remarks 仮想デストラクタで無いので、このクラスを継承してはいけません。
     */
    ~local_database() { }

    
    /** @name エラー処理関係 */
    /* @{ */

    /** @brief データベースがオープンされているか検査します。
     *  @returns オープンされていればtrue。そうでなければfalse。
     */
    inline bool   is_opened() const { return this->raw_handle_ != 0; }
    

    /** @brief 最後に検出したエラーコードを取得します
     *  @returns 最後に検出したエラーコード
     */
    inline int    get_last_error() const { return last_error_; }


    /** @brief エラーが発生しているか検査します
     *  @returns エラーが発生していればtrue、そうでなければfalse。
     */
    inline bool   has_error() const { return get_last_error() != ESTENOERR; }


    /** @brief エラーコードに対応するネイティブのエラーメッセージを取得します
     *  @param ecode エラーコード
     *  @returns エラーメッセージ
     */
    static inline char const* get_error_message(int ecode) { return ::est_err_msg(ecode); }

    /** @brief 最後に発生したエラーに対応するネイティブのエラーメッセージを取得します
     *  @returns エラーメッセージ
     */
    inline char const* error_message() { return get_error_message(get_last_error()); }


    /** @brief 直前に発生したエラーコードをネイティブで取り出します
     *  @returns ネイティブのエラーコード
     *
     *  データベースがオープンしていない場合には、常にエラー無しとして報告します。
     *  従って、オープンエラーの時にはこのメソッドを使う事はできません。
     */
    inline int get_db_error() const
    {
      return is_opened()? estdbapi::error(this->raw_handle_) : ESTENOERR;
    }

    /** @brief データベースに致命的なエラーが発生しているか検査します
     *  @returns 致命的なエラーが発生していればtrue、そうでなければfalse。
     */
    inline bool is_fatal() const
    {
      return estdbapi::fatal(this->raw_handle_)? true : false;
    }
    /* @} */




    /** @brief データベースを開きます
     *  @param name データベースの名前
     *  @param omode オープンモード
     *     \arg \c ESTDBWRITER    ライタモード
     *     \arg \c ESTDBREADER    リーダモード
     *     \arg \c ESTDBCREAT     データベースが存在しない場合に新しく作成します(ライタモードのみ)
     *     \arg \c ESTDBTRUNC     データベースが存在していたら、新規に作成し直します(ライタモードのみ)
     *     \arg \c ESTDBNOLCK     データベースをロックしません
     *     \arg \c ESTDBLCKNB     データベースをロックしますが、ブロックしません
     *     \arg \c ESTDBPERFNG    ESTDBCREATの時、欧文も完全なN-gramでインデックスするデータベースを作ります
     *  @returns エラーコード。成功すれば ESTENOERR (0)。
     *  @remarks データベースが既にオープンされていた場合には自動的にクローズします。
     *           自動的なクローズに失敗した場合には、そのエラーコードが戻ります。
     */
    int open(const char *name, int omode = ESTDBREADER)
    {
      if (is_opened()) {
        close();
        if (has_error()) return last_error_;
      }

      set_raw_handle(estdbapi::open(name, omode, &last_error_));
      return last_error_;
    }


    /** @brief データベースを閉じます
     *  開いていなかった場合にはなにもせずに成功(ESTENOERR)します。
     *  @returns エラーコード。成功すればESTENOERR (0)。
     */
    int close()
    {
      last_error_ = ESTENOERR;
      if (is_opened()) {
        if (estdbapi::close(this->raw_handle_, &last_error_))
          this->raw_handle_ = 0;
      }
      return last_error_;
    }


    /** @brief ライタとして接続しているデータベースをフラッシュします。
     *  @param max フラッシュする語の最大数。0以下であれば全ての索引語がフラッシュされます。
     *  \retval true   成功
     *  \retval false  データベースに致命的なエラーがあった
     */
    inline bool flush(int max) { return estdbapi::flush(this->raw_handle_, max)? false : true; }



    /** @brief データベースの更新内容を同期します
     *  \retval true 成功
     *  \retval false 失敗
     */
    inline bool sync() { return estdbapi::sync(this->raw_handle_)? true : false; }

 
    /** @brief データベースを最適化します
     *  @param options オプション
     *  \arg \c ESTOPTNOPURGE 削除された文書の情報を削除しません
     *  \arg \c ESTOPTNODBOPT データベースファイルの最適化を行いません
     *  \retval true 成功
     *  \retval false 失敗
     */
    inline bool optimize(int options) {
      return estdbapi::optimize(this->raw_handle_, options)? true : false;
    }


    /* @brief 文書オブジェクトからキーワードを抽出します
     * @param doc 文書オブジェクト
     * @param max 抽出する最大数
     * @returns キーワードとスコア（BCD)のマップ。cbmapcloseで破棄するする必要があります。
     */
    inline CBMAP* etch_doc(ESTDOC* doc, int max) const { return estdbapi::etch_doc(this->raw_handle_, doc, max); }


    /* @brief 文書オブジェクトからキーワードを抽出します
     * @param doc 文書オブジェクト
     * @param max 抽出する最大数
     * @returns キーワードとスコア（BCD)のマップ。cbmapcloseで破棄するする必要があります。
     */
    inline CBMAP* etch_doc(local_document* doc, int max) const { return etch_doc(doc->raw_handle(), max); }



    /** @brief キャッシュメモリの最大サイズを指定します
     *  @param size インデックスキャッシュの最大サイズ(デフォルトは64MB)。
     *  @param anum 文書属性キャッシュの最大レコード数(デフォルトは8192)。
     *  @param tnum 文書テキストキャッシュの最大レコード数（デフォルトは1024)。
     *
     *  いずれの値も、負を指定すると現在の値を変更しません。
     */
    inline void set_cache_size(size_t size, int anum, int tnum, int rnum) { estdbapi::set_cache_size(this->raw_handle_, size, anum, tnum, rnum); }


    /** @brief データベースに起きたイベントを通知する為のコールバック関数を指定します
     *  @param func コールバック関数へのポインタ
     */
    inline void set_informer(informer_type func) { estdbapi::set_informer(this->raw_handle_, func); }


    /** @brief 文書のキーワードのベクトルを生成する為のコールバック関数を指定します。
     *  @param func ベクトル化コールバック関数へのポインタ
     */
    inline void set_vectorizer(vectorizer_type func, void* user_data) {
      vectorizer_ = func;
      estdbapi::set_vectorizer(this->raw_handle_, &local_database<THREADING>::vectorier_cb, (void*)this);
    }



    /** @brief 検索条件に合致する文書の一覧を取得します
     *  @param cond 検索条件オブジェクト
     *  @param nump 戻り値の配列の用素数を受け取る変数へのポインタ
     *  @param hints 各検索後に該当する文書数を格納するマップオブジェクト。NULLなら使用しません
     *  @returns 該当した文書IDの配列。freeで破棄する必要があります。
     *
     *  この関数は決してエラーになりません。該当する文書が無い場合には空の配列が返されます。
     */
    inline int* search(ESTCOND* cond, int* nump, CBMAP* hints) { return estdbapi::search(this->raw_handle_, cond, nump, hints); }

    /** @brief 検索条件に合致する文書の一覧を取得します
     *  @param cond 検索条件オブジェクト
     *  @param hints 各検索後に該当する文書数を格納するマップオブジェクト。NULLなら使用しません
     *  @returns 文書IDの入った配列へのポインタとサイズのpair。
     *
     *  戻されたポインタは明示的にfreeする必要がありますが、search_result_type
     *  に代入すれば自動的に開放されます。
     */
    inline std::pair<int*, std::size_t> search(ESTCOND* cond, CBMAP* hints) {
      int  count;
      int* result = search(cond, &count, hints);
      return std::pair<int*, std::size_t>(result, count);
    }

    /** @brief 検索条件に合致する文書の一覧を取得します
     *  @param cond 検索条件
     *  @param nump 戻り値の配列の用素数を受け取る変数へのポインタ
     *  @param hints 各検索後に該当する文書数を格納するマップオブジェクト。NULLなら使用しません
     *  @returns 該当した文書IDの配列。freeで破棄する必要があります。
     *
     *  この関数は決してエラーになりません。該当する文書が無い場合には空の配列が返されます。
     */
    inline int* search(condition const& cond, int* nump, CBMAP* hints) {
      return estdbapi::search(this->raw_handle_, cond.raw_handle(), nump, hints);
    }


    /** @brief 検索条件に合致する文書の一覧を取得します
     *  @param cond 検索条件オブジェクト
     *  @param hints 各検索後に該当する文書数を格納するマップオブジェクト。NULLなら使用しません
     *  @returns 文書IDの入った配列へのポインタとサイズのpair。
     *
     *  戻されたポインタは明示的にfreeする必要がありますが、search_result_type
     *  に代入すれば自動的に開放されます。
     */
    inline std::pair<int*, std::size_t> search(condition const& cond, CBMAP* hints) {
      int  count;
      int* result = this->search(cond, &count, hints);
      return std::pair<int*, std::size_t>(result, count);
    }


    inline int scan_doc(local_document const& doc, condition const& cond) const {
      return estdbapi::scan_doc(doc.raw_handle(), cond.raw_handle());
    }


    //
    // Override
    //
    virtual bool    put_doc(ESTDOC* doc, int options = 0) { return estdbapi::put_doc(this->raw_handle_, doc, options)? true : false;  }
    virtual bool    put_doc(local_document& doc, int options = 0) { return estdbapi::put_doc(this->raw_handle_, doc.raw_handle(), options)? true : false;  }
    virtual bool    out_doc(int id, int options = 0) { return estdbapi::out_doc(this->raw_handle_, id, options)? true : false; }
    virtual bool    out_doc(char const* uri, int options = 0) {
      bool result = database::out_doc(uri, options);
      iter_init(uri);
      return result;
    }
    virtual ESTDOC* get_doc(int id, int options = 0) const { return estdbapi::get_doc(this->raw_handle_, id, options); }
    virtual ESTDOC* get_doc(char const* uri, int options = 0) const { return database::get_doc(uri, options); }
    virtual int     uri_to_id(char const* uri) const { return estdbapi::uri_to_id(this->raw_handle_, uri); }
    virtual char const* name() const { return estdbapi::name(this->raw_handle_); }
    virtual int     doc_num() const { return estdbapi::doc_num(this->raw_handle_); }
    virtual int     word_num() const { return estdbapi::word_num(this->raw_handle_); }
    virtual double  size() const { return estdbapi::size(this->raw_handle_); }
    virtual bool    iter_init() { return estdbapi::iter_init(this->raw_handle_, 0)? true : false; }
    inline  bool    iter_init(char const* prev) { return estdbapi::iter_init(this->raw_handle_, prev)? true : false; }
    virtual ESTDOC* iter_next() { int id = estdbapi::iter_next(this->raw_handle_); return (id < 0)? 0 : get_doc(id, ESTGDNOTEXT); }




    /** */
  protected:
    vectorizer_type vectorizer_;

    /** @brief ベクタライザコールバックをラップする為のヘルパ
     */
    static CBMAP* vectorizer_cb(void* db, int id, void* user_data)
    {
      local_database<THREADING>* me = dynamic_cast<local_database<THREADING>*>(user_data);
      BOOST_ASSERT(me != 0 && "Dynamic cast failed.");
      return vectorier_(me, id);
    }
  };
}

#endif
