/** @file
 */
#if !defined(__HYPERESTRAIER__DOC__HPP__)
#define __HYPERESTRAIER__DOC__HPP__

#include <string>
#include <boost/noncopyable.hpp>

#include <estraier.h>
#include "wrapper.hpp"
#include "cond.hpp"

namespace hyperestraier {


  /** @brief 抽象的な文書オブジェクト
   */
  class document : public boost::noncopyable
  {
  protected:
    document() { }

  public:
    virtual ~document() { }

    /** @brief 文書のURIを取得します
     *  @returns URI文字列。破棄する必要はありません。
     */
    virtual char const* get_uri() const = 0;

    /** @brief 文書に付与されている属性名の一覧を取得します
     *  @return 属性名のリスト。呼出側で::cblistcloseで破棄する必要がります。
     */
    virtual ::CBLIST*   attr_names() const = 0;

    /** @brief 文書の属性名に対応する属性値を取得します
     *  @param name  属性名
     *  @param dft   属性が登録されていなかった時に使うデフォルト値
     *  @returns 属性値。属性が登録されていなかった場合にはdftを戻します。
     *           取り出した属性値を破棄する必要はありません。
     */
    virtual char const* get_attr(char const* name, char const* dft = 0) const = 0;


  };


  namespace detail {
    template <> inline void raw_handle_deleter<ESTDOC>(ESTDOC* p) { ::est_doc_delete(p); }
  }


  /** @brief 文書オブジェクト
   */
  class local_document :
    public document,
    public handle_wrapper<ESTDOC, local_document>
  {
  public:
    /** @brief デフォルトコンストラクタ
     *  空の状態でインスタンスを構築します
     */
    local_document() :
      handle_wrapper<ESTDOC, local_document>()
    { }

    /** @brief コンストラクタ
     *  @param raw_handle ネイティブの文書オブジェクトハンドル
     */
    local_document(ESTDOC* raw_handle) :
      handle_wrapper<ESTDOC, local_document>(raw_handle)
    { }



    /** @brief デストラクタ
     *  関連付いたネイティブの文書オブジェクトは自動的に破棄されます
     */
    virtual ~local_document() { }


    /** @brief 新しい空の文書オブジェクトを生成します
     */
    inline void create() {
      clear();
      raw_handle_ = ::est_doc_new();
    }


    /** @brief 文書ドラフトから新しい文書オブジェクトを生成します
     *  @param draft 文書ドラフト
     */
    inline void create(char const* draft) {
      clear();
      raw_handle_ = ::est_doc_new_from_draft(draft);
    }

    



    /** @brief 文書に属性を追加します
     *  @param name 属性の名前
     *  @param value 属性の値。NULLを指定すると、属性が削除されます。
     */
    inline void set_attr(char const* name, char const* value) { ::est_doc_add_attr(raw_handle_, name, value); }

    /** @brief 文書に属性を追加します
     *  @param name 属性の名前
     *  @param value 属性の値。
     *
     *  このメソッドでは、属性の削除はできません。
     */
    inline void set_attr(char const* name, std::string const& value) { ::est_doc_add_attr(raw_handle_, name, value.c_str()); }


    /** @brief 文書にテキストを追加します
     *  @param text 追加するテキスト
     *  @param hidden 隠しテキストとして追加する場合はtrue
     */
    inline void add_text(char const* text, bool hidden = false) {
      if (hidden) ::est_doc_add_hidden_text(raw_handle_, text);
      else        ::est_doc_add_text(raw_handle_, text);
    }

    inline void add_text(std::string const& text, bool hidden = false) { return add_text(text.c_str(), hidden); }
    

    /** @brief 文書のIDを取得します
     *  @returns 文書ID。文書がまだ登録されていない場合には-1。
     */
    inline int id() const { return ::est_doc_id(raw_handle_); }




    /** @brief 文書のテキストを取得します
     *  @returns 文書のテキストの入ったリスト。
     */
    inline CBLIST const* get_texts() const { return ::est_doc_texts(raw_handle_); }

    /** @brief 文書ドラフトへダンプします
     *  @returns ダンプされた文書ドラフト。freeで破棄する必要があります
     */
    inline char* dump_draft() const { return ::est_doc_dump_draft(raw_handle_); }


    //
    // Override
    //
    virtual char const* get_uri() const { return get_attr(ESTDATTRURI); }
    virtual CBLIST* attr_names() const { return ::est_doc_attr_names(raw_handle_); }
    virtual char const* get_attr(char const* name, char const* dft = 0) const {
      char const* v = ::est_doc_attr(raw_handle_, name);
      return (v == 0)? dft : v;
    }



    /* @brief 文書テキストの要約を作成します
     * @param word ハイライトする語を指定するlistオブジェクト
     * @param wwidth 結果全体の幅
     * @param hwidth テキストの先頭からピックアップする幅
     * @param asidth ハイライトされた語の周囲からピックアップする幅
     * @returns 要約文字列。freeで破棄する必要があります。
     *
     * 要約文字列はタブで分割された文字列です。各々の行は表示する為の語で、ほとんどは1フィールドのみですが、2つの
     * フィールドを持っている場合があります。
     * 2つ目のフィールドが存在した場合、最初のフィールドはハイライト表示する為の語で、2番目は正規化されている語です。
     */
    inline char* make_snippet(CBLIST const* words, int wwidth, int hwidth, int awidth) const {
      return ::est_doc_make_snippet(raw_handle_, words, wwidth, hwidth, awidth);
    }
  };

}

#endif

