/** @file
 */
#if !defined(__HYPERESTRAIER__DB__HPP__)
#define __HYPERESTRAIER__DB__HPP__

#define GNU_SOURCE
#include <cstring>
#include <boost/noncopyable.hpp>

#include <estraier.h>

#include "doc.hpp"
#include "cond.hpp"

namespace hyperestraier {


  class database : 
    public boost::noncopyable
  {
  protected:
    database() { };

  public:
    virtual ~database() { };

    /** @brief データベースに文書を追加します
     *  @param doc  登録する文書
     *  @param options オプション
     *  \arg \c ESTPDCLEAN 上書きされた文書の領域を整理します。（ローカルデータベースのみ）
     *  \retval true 成功
     *  \retval false 失敗
     */
    virtual bool     put_doc(ESTDOC* doc, int options = 0) = 0;



    /** @brief データベースに文書を追加します
     *  @param doc  登録する文書
     *  @param options オプション
     *  \arg \c ESTPDCLEAN 上書きされた文書の領域を整理します。（ローカルデータベースのみ）
     *  \retval true 成功
     *  \retval false 失敗
     */
    virtual  bool     put_doc(local_document& doc, int options = 0) { return put_doc(doc.raw_handle(), options); }



    /** @brief データベースから文書を削除します
     *  @param id 文書ID
     *  @param options オプション
     *  \arg \c ESTODCLEAN 削除された文書の領域を整理します。(ローカルデータベースのみ)
     *  \retval true 成功
     *  \retval false 失敗
     */
    virtual bool     out_doc(int id, int options = 0) = 0;


    /** @brief データベースから文書を削除します
     *  @param uri 文書URI
     *  @param options オプション
     *  \arg \c ESTODCLEAN 削除された文書の領域を整理します。(ローカルデータベースのみ)
     *  \retval true 成功
     *  \retval false 失敗
     */
    virtual bool     out_doc(char const* uri, int options = 0) {
      int id = uri_to_id(uri);
      return (id == 0)? false : out_doc(id, options);
    }


    /** @brief データベースから文書を取得します
     *  @param id 文書ID
     *  @param options オプション
     *  \arg ESTGDNOATTR 属性を取得しません。(ローカルデータベースのみ)
     *  \arg ESTGDNOTEXT 本文を取得しません。(ローカルデータベースのみ)
     *  @returns エラーの場合にはNULL。成功した場合には文書オブジェクト。
     *  @remarks APIマニュアルに記述は無いが、estseek.cgiを見るかぎり、
     *           est_db_get_docで取り出した文書オブジェクトも破棄する必要があるらしい。
     */
    virtual ESTDOC*  get_doc(int id, int options = 0) const = 0;


    /** @brief データベースから文書を取得します
     *  @param uri  文書URI
     *  @param options オプション
     *  \arg ESTGDNOATTR 属性を取得しません。(ローカルデータベースのみ)
     *  \arg ESTGDNOTEXT 本文を取得しません。(ローカルデータベースのみ)
     *  @returns エラーの場合にはNULL。成功した場合には文書オブジェクト。
     *  @remarks APIマニュアルに記述は無いが、estseek.cgiを見るかぎり、
     *           est_db_get_docで取り出した文書オブジェクトも破棄する必要があるらしい。
     */
    virtual ESTDOC*  get_doc(char const* uri, int options = 0) const {
      int id = uri_to_id(uri);
      return (id == 0)? 0 : get_doc(id, options);
    }


    /** @brief 文書の属性を取得します
     *  @param id 文書ID
     *  @param name 属性の名前
     *  @param dft  デフォルト値
     *  @return 属性の値、またはdft。
     */
    virtual char* get_doc_attr(int id, char const* name, char const* dft = 0) const {
      local_document doc(get_doc(id, ESTGDNOTEXT));
      char const* v = 0;
      if (!! doc) v = doc.get_attr(name, dft);
      return v? ::strdup(v) : (dft? ::strdup(dft) : 0);
    }

    /** @brief 文書の属性を取得します
     *  @param uri 文書URI
     *  @param name 属性の名前
     *  @param dft  デフォルト値
     *  @return 属性の値、またはdft。
     */
    virtual char* get_doc_attr(char const* uri, char const* name, char const* dft = 0) const {
      local_document doc(get_doc(uri, ESTGDNOTEXT));
      char const* v = 0;
      if (!! doc) v = doc.get_attr(name, dft);
      return v? ::strdup(v) : (dft? ::strdup(dft) : 0);
    };



    /** @brief URIに対応する文書のIDを取得します
     *  @param uri  文書URI
     *  @returns エラーの場合には-1を、成功すれば文書ID
     */
    virtual int     uri_to_id(char const* uri) const = 0;


    /** @brief データベースの名前を取得します
     *  @returns データベースの名前
     */
    virtual char const* name() const = 0;


    /** @brief データベースに登録されている文書の数を取得します
     *  @returns 登録されている文書の数
     */
    virtual int     doc_num() const = 0;

    /** @brief データベースに登録されているユニークな語の数を取得します
     *  @returns 登録されているユニークな語の数
     */
    virtual int     word_num() const = 0;


    /** @brief データベースのサイズを取得します
     *  @returns データベースのサイズ
     */
    virtual double  size() const = 0;


    /* @{ @name 文書の列挙 */

    /** 文書イテレータを初期化します
     * @return 成功すれば true。失敗すれば false。
     */
    virtual bool    iter_init() = 0;

    /** 次の文書を取得します
     * @return 文書ハンドル。もう列挙すべき文書が無い場合には0。
     */
    virtual ESTDOC* iter_next() = 0;
    /* @} */

  };



}

#endif

