/** @file
 */
#if !defined(__GDESTRAIER__BUILDER__FILTER__HPP__)
#define __GDESTRAIER__BUILDER__FILTER__HPP__

#include <string>
#include <map>
#include <set>
#include <boost/noncopyable.hpp>

#include <libgnomevfs/gnome-vfs-file-info.h>
#include <libgnomevfs/gnome-vfs-uri.h>

#include "hyperestraier/doc.hpp"

#include "model/index.hpp"
#include "model/encodings.hpp"

namespace gdestraier {
  namespace builder {
    namespace filter {

      /** @brief 文書フィルタの抽象基底クラス
       *
       */
      class abstract_filter : public boost::noncopyable {
      public:
        abstract_filter();
        virtual ~abstract_filter();

        virtual bool operator()(hyperestraier::local_document* doc,
                                gdestraier::model::index_type const& index,
                                ::GnomeVFSURI* uri,
                                char const* text_uri,
                                ::GnomeVFSFileInfo* info,
                                char const* mime_type
                                ) const = 0;

      protected:
        enum { MAX_CONTENT_SIZE = 10 * 1024 * 1024 };
        static void* load_file_content(::GnomeVFSURI* uri, ::GnomeVFSFileSize size, ::GnomeVFSFileSize* bytes_read);

        static char* get_local_path_from_uri(::GnomeVFSURI* uri, char const* text_uri = 0);
      };


      /** @brief 文書フィルタのファクトリ
       */
      class factory : public boost::noncopyable {
      public:
        struct extention_map_type {
          char const* ext_;
          char const* mime_type_;
        };

        factory(char const* filter_name,
                abstract_filter* (*create_proc)(),
                extention_map_type const* extmap);
        virtual ~factory();

      protected:
        char const* filter_name_;
        abstract_filter* (*create_proc_)();
        extention_map_type const* ext_;

      public:
        char const* get_filter_name() const { return filter_name_; }
        extention_map_type const* get_extentions() const { return ext_; }

        abstract_filter* create_filter() const { return (*create_proc_)(); }
      };



      /** システムにインストールされている全てのフィルタのファクトリを管理します。
       */
      class available_factories : 
        public std::map<std::string, factory const*>,
        public boost::noncopyable
      {
      protected:
        available_factories();

        void insert_factory(factory const& f);

      public:
        virtual ~available_factories();

        factory const* get_factory_by_name(char const* name) const;

        static available_factories const& instance();
      };
    }
  }
}

#endif

