/** @file
 */
#if defined(HAVE_CONFIG_H)
#  include "../../config.h"
#endif
#include <boost/assert.hpp>
#include <utility>

#include <libgnomevfs/gnome-vfs-ops.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <libgnomevfs/gnome-vfs-utils.h>

#include "filter.hpp"

#include "document_draft_filter.hpp"
#include "text_filter.hpp"
#include "html_filter.hpp"
#if defined(HAVE_LIBID3)
#include "id3_filter.hpp"
#endif
#include "pdf_filter.hpp"
#include "excel_filter.hpp"
#include "powerpoint_filter.hpp"
#include "unknown_filter.hpp"

namespace gdestraier {
  namespace builder {
    namespace filter {

      /** コンストラクタ */
      abstract_filter::abstract_filter() { }

      /** デストラクタ */
      abstract_filter::~abstract_filter() { }



      /**
       * URIで示されるリソースの中身を読み込みます
       * @param uri  読み込むリソースのURI
       * @param size リソースのサイズ
       * @param bytes_read 実際に読み込んだサイズを戻す変数へのポインタ
       * @returns 読み込んだバッファへのポインタ。このバッファはstd::freeで破棄する必要があります。読み込みに失敗した場合には 0。
       *
       *  バッファの末尾には'\0'が付加されるので、そのまま文字列として使用できます。
       */
      void*
      abstract_filter::load_file_content(::GnomeVFSURI* uri,
                                         ::GnomeVFSFileSize size,
                                         ::GnomeVFSFileSize* bytes_read) {
        if (size > MAX_CONTENT_SIZE) size = MAX_CONTENT_SIZE;
        if (size == 0) return 0;

        void* content = std::malloc(size + 1);
        if (content == 0) return 0;

        ::GnomeVFSHandle* fh = 0;
        ::GnomeVFSResult res = ::gnome_vfs_open_uri(&fh, uri, ::GNOME_VFS_OPEN_READ);
        if (res != GNOME_VFS_OK || fh == 0) {
          std::free(content);
          if (fh != 0) ::gnome_vfs_close(fh);
          return 0;
        }


        *bytes_read = 0;
        res = ::gnome_vfs_read(fh, content, size, bytes_read);
        if (res != GNOME_VFS_OK) {
          ::gnome_vfs_close(fh);
          std::free(content);
          return 0;
        }

        ::gnome_vfs_close(fh);
        *((char*)content + *bytes_read) = '\0';

        return content;
      }


      /** @brief URIオブジェクトとテキスト形式からローカルパスを取得します。
       *  @param uri  URIオブジェクト
       *  @param text_uri テキスト形式。0 の場合は uriオブジェクトから生成されます。
       *  @returns 取得できた場合には変換されたローカルパス。変換できないか、uriがローカルで無い場合には0。
       *           戻された文字列はg_freeで開放する必要があります。
       */
      char* abstract_filter::get_local_path_from_uri(::GnomeVFSURI* uri, char const* text_uri)
      {
        char* p = 0;
        if (text_uri == 0) {
          text_uri = p =
            ::gnome_vfs_uri_to_string(uri,
                                      ::GnomeVFSURIHideOptions(::GNOME_VFS_URI_HIDE_USER_NAME |
                                                               ::GNOME_VFS_URI_HIDE_PASSWORD));
          if (text_uri == 0) return 0;
        }

        char* local_path = 0;
        if (! ::gnome_vfs_uri_is_local(uri))
          local_path = ::gnome_vfs_get_local_path_from_uri(text_uri);

        if (p != 0) ::g_free(p);
        return local_path;
      }



      /** @brief コンストラクタ
       *  @param filter_name   このファクトリが作成するフィルタの名前
       *  @param create_proc   実際にフィルタを作成する関数
       *  @param exe           フィルタがサポートするファイル拡張子の配列
       */
      factory::factory(char const* filter_name,
                       abstract_filter* (create_proc)(),
                       extention_map_type const* ext) :
        filter_name_(filter_name),
        create_proc_(create_proc),
        ext_(ext)
      { }

      /** デストラクタ */
      factory::~factory() { }



      /** コンストラクタ
       *  システムに存在する全てのフィルタの為のファクトリを登録します。
       */
      available_factories::available_factories() {
        insert_factory(from_document_draft::get_factory());
        insert_factory(from_text::get_factory());
        insert_factory(from_html::get_factory());
#if defined(HAVE_LIBID3)
        insert_factory(from_id3::get_factory());
#endif
        insert_factory(from_PDF::get_factory());

        insert_factory(from_excel::get_factory());
        insert_factory(from_PowerPoint::get_factory());

        insert_factory(from_unknown::get_factory());
      }

      /** デストラクタ */
      available_factories::~available_factories() { }



      void
      available_factories::insert_factory(factory const& f) {
        insert(std::pair<std::string, factory const*>(std::string(f.get_filter_name()), &f));
      }


      available_factories const& available_factories::instance() {
        static available_factories inst;
        return inst;
      }


      factory const*
      available_factories::get_factory_by_name(char const* name) const {
        const_iterator i = this->find(std::string(name));
        if (i == this->end()) return 0;
        return i->second;
      }



    }
  }
}
