#include <ruby.h>
#include <node.h>

#include "yarvcore.h"
#include "vm.h"

static void
control_frame_dump(yarv_thread_t *th, yarv_control_frame_t *cfp)
{
    int pc = -1, bp = -1;
    int lfp = cfp->lfp - th->stack;
    int dfp = cfp->dfp - th->stack;

    char *magic, *iseq_name = "-", *selfstr = "-", *biseq_name = "-";
    VALUE tmp;

    if (cfp->block_iseq != 0 && BUILTIN_TYPE(cfp->block_iseq) != T_NODE) {
	biseq_name = "";	//RSTRING(cfp->block_iseq->name)->ptr;
    }

    if (lfp < 0 || lfp > th->stack_size)
	lfp = lfp % 1000;
    if (dfp < 0 || dfp > th->stack_size)
	dfp = dfp % 1000;
    if (cfp->bp) {
	bp = cfp->bp - th->stack;
    }

    switch (cfp->magic) {
    case FRAME_MAGIC_TOP:
	magic = "TOP";
	break;
    case FRAME_MAGIC_METHOD:
	magic = "METHOD";
	break;
    case FRAME_MAGIC_CLASS:
	magic = "CLASS";
	break;
    case FRAME_MAGIC_BLOCK:
	magic = "BLOCK";
	break;
    case FRAME_MAGIC_FINISH:
	magic = "FINISH";
	break;
    case FRAME_MAGIC_CFUNC:
	magic = "CFUNC";
	break;
    case FRAME_MAGIC_PROC:
	magic = "PROC";
	break;
      case FRAME_MAGIC_LAMBDA:
	magic = "LAMBDA";
	break;
    case FRAME_MAGIC_IFUNC:
	magic = "IFUNC";
	break;
    case FRAME_MAGIC_EVAL:
	magic = "EVAL";
	break;
    case 0:
	magic = "------";
	break;
    default:
	magic = "(none)";
	break;
    }

    if (0) {
	tmp = rb_inspect(cfp->self);
	selfstr = StringValueCStr(tmp);
    }
    else {
	selfstr = "";
    }

    if (cfp->iseq != 0) {
	if (YARV_CMETHOD_INFO_P(cfp->iseq)) {
	    struct yarv_cmethod_info *cmi = (void *)cfp->iseq;
	    iseq_name = rb_id2name(cmi->id);
	}
	else if (YARV_IFUNC_P(cfp->iseq)) {
	    iseq_name = "<ifunc>";
	}
	else {
	    pc = cfp->pc - cfp->iseq->iseq_encoded;
	    iseq_name = RSTRING(cfp->iseq->name)->ptr;
	}
    }

    printf("c:%04ld ",
	   (yarv_control_frame_t *)(th->stack + th->stack_size) - cfp);
    printf("p:%04d s:%04ld b:%04d ", pc, cfp->sp - th->stack, bp);
    printf("l:%04d d:%04d ", lfp, dfp);
    printf("%-6s ", magic);
    printf("i:%-8s ", iseq_name);
    printf("s:%-8s ", selfstr);
    printf("%-2s ", biseq_name);
    printf("\n");
}

void
vm_stack_dump_raw(yarv_thread_t *th, yarv_control_frame_t *cfp)
{
    VALUE *sp = cfp->sp, *bp = cfp->bp;
    VALUE *lfp = cfp->lfp;
    VALUE *dfp = cfp->dfp;
    VALUE *p, *st, *t;

    printf("-- stack frame ------------\n");
    for (p = st = th->stack; p < sp; p++) {
	printf("%p (%04ld): %08lx", p, p - st, *p);

	t = (VALUE *)*p;
	if (th->stack <= t && t < sp) {
	    printf(" (= %ld)", (VALUE *)GC_GUARDED_PTR_REF(t) - th->stack);
	}

	if (p == lfp)
	    printf(" <- lfp");
	if (p == dfp)
	    printf(" <- dfp");
	if (p == bp)
	    printf(" <- bp");	/* should not be */

	printf("\n");
    }
    printf("-- control frame ----------\n");
    while ((void *)cfp < (void *)(th->stack + th->stack_size)) {
	control_frame_dump(th, cfp);
	cfp++;
    }
    printf("---------------------------\n");
}

void
env_dump_raw(yarv_env_t *env, VALUE *lfp, VALUE *dfp)
{
    int i;
    printf("-- env --------------------\n");

    while (env) {
	printf("--\n");
	for (i = 0; i < env->env_size; i++) {
	    printf("%04d: %08lx (%p)", -env->local_size + i, env->env[i],
		   &env->env[i]);
	    if (&env->env[i] == lfp)
		printf(" <- lfp");
	    if (&env->env[i] == dfp)
		printf(" <- dfp");
	    printf("\n");
	}

	if (env->prev_envval != 0) {
	    GetEnvVal(env->prev_envval, env);
	}
	else {
	    env = 0;
	}
    }
    printf("---------------------------\n");
}

void
proc_dump_raw(yarv_proc_t *proc)
{
    yarv_env_t *env;
    char *selfstr;
    VALUE val = rb_inspect(proc->block.self);
    selfstr = StringValueCStr(val);


    printf("-- proc -------------------\n");
    printf("self: %s\n", selfstr);
    GetEnvVal(proc->envval, env);
    env_dump_raw(env, proc->block.lfp, proc->block.dfp);
}

void
stack_dump_th(VALUE thval)
{
    yarv_thread_t *th;
    GetThreadVal(thval, th);
    vm_stack_dump_raw(th, th->cfp);
}

void
stack_dump_each(yarv_thread_t *th, yarv_control_frame_t *cfp)
{
    int i;

    VALUE rstr;
    VALUE *sp = cfp->sp;
    VALUE *lfp = cfp->lfp;
    VALUE *dfp = cfp->dfp;

    int argc, local_size;
    char *name;
    yarv_iseq_t *iseq = cfp->iseq;

    if (iseq == 0) {
	name = "?";
	local_size = 0;
    }
    else if (YARV_CMETHOD_INFO_P(iseq)) {
	struct yarv_cmethod_info *cmi = (void *)iseq;
	argc = 0;
	local_size = 0;
	name = rb_id2name(cmi->id);
    }
    else if (YARV_IFUNC_P(iseq)) {
	argc = 0;
	local_size = 0;
	name = "<ifunc>";
    }
    else {
	argc = iseq->argc;
	local_size = iseq->local_size;
	name = RSTRING(iseq->name)->ptr;
    }

    /* stack trace header */

    if (cfp->magic == FRAME_MAGIC_METHOD ||
	cfp->magic == FRAME_MAGIC_TOP ||
	cfp->magic == FRAME_MAGIC_BLOCK ||
	cfp->magic == FRAME_MAGIC_CLASS ||
	cfp->magic == FRAME_MAGIC_PROC ||
	cfp->magic == FRAME_MAGIC_LAMBDA ||
	cfp->magic == FRAME_MAGIC_CFUNC ||
	cfp->magic == FRAME_MAGIC_IFUNC ||
	cfp->magic == FRAME_MAGIC_EVAL) {

	VALUE *ptr = dfp - local_size;

	stack_dump_each(th, cfp + 1);
	control_frame_dump(th, cfp);

	if (lfp != dfp) {
	    local_size++;
	}
	for (i = 0; i < argc; i++) {
	    rstr = rb_inspect(*ptr);
	    printf("  arg   %2d: %8s (%p)\n", i, StringValueCStr(rstr),
		   ptr++);
	}
	for (; i < local_size - 1; i++) {
	    rstr = rb_inspect(*ptr);
	    printf("  local %2d: %8s (%p)\n", i, StringValueCStr(rstr),
		   ptr++);
	}

	ptr = cfp->bp;
	for (; ptr < sp; ptr++, i++) {
	    if (*ptr == Qundef) {
		rstr = rb_str_new2("undef");
	    }
	    else {
		rstr = rb_inspect(*ptr);
	    }
	    printf("  stack %2d: %8s (%ld)\n", i, StringValueCStr(rstr),
		   ptr - th->stack);
	}
    }
    else if (cfp->magic == FRAME_MAGIC_FINISH) {
	if ((th)->stack + (th)->stack_size > (VALUE *)(cfp + 2)) {
	    stack_dump_each(th, cfp + 1);
	}
	else {
	    // SDR();
	}
    }
    else {
	rb_bug("unsupport frame type: %08lx", cfp->magic);
    }
}


void
debug_print_register(yarv_thread_t *th)
{
    yarv_control_frame_t *cfp = th->cfp;
    int pc = -1;
    int lfp = cfp->lfp - th->stack;
    int dfp = cfp->dfp - th->stack;
    int cfpi;

    if (YARV_NORMAL_ISEQ_P(cfp->iseq)) {
	pc = cfp->pc - cfp->iseq->iseq_encoded;
    }

    if (lfp < 0 || lfp > th->stack_size)
	lfp = -1;
    if (dfp < 0 || dfp > th->stack_size)
	dfp = -1;

    cfpi = ((yarv_control_frame_t *)(th->stack + th->stack_size)) - cfp;
    printf("  [PC] %04d, [SP] %04ld, [LFP] %04d, [DFP] %04d, [CFP] %04d\n",
	   pc, cfp->sp - th->stack, lfp, dfp, cfpi);
}

void
thread_dump_regs(VALUE thval)
{
    yarv_thread_t *th;
    GetThreadVal(thval, th);
    debug_print_register(th);
}

void
debug_print_pre(yarv_thread_t *th, yarv_control_frame_t *cfp)
{
    yarv_iseq_t *iseq = cfp->iseq;

    if (iseq != 0 && cfp->magic != FRAME_MAGIC_FINISH) {
	VALUE *seq = iseq->iseq;
	int pc = cfp->pc - iseq->iseq_encoded;

	iseq_disasm_insn(0, seq, pc, iseq, 0);
    }

#if VMDEBUG > 3
    printf("        (1)");
    debug_print_register(th);
#endif
}

void
debug_print_post(yarv_thread_t *th, yarv_control_frame_t *cfp
#if OPT_STACK_CACHING
		 , VALUE reg_a, VALUE reg_b
#endif
    )
{
#if VMDEBUG > 9
    SDR2(cfp);
#endif

#if VMDEBUG > 3
    printf("        (2)");
    debug_print_register(th);
#endif
    // stack_dump_raw(th, cfp);

#if VMDEBUG > 2
    // stack_dump_thobj(th);
    stack_dump_each(th, th->cfp);
#if OPT_STACK_CACHING
    {
	VALUE rstr;
	rstr = rb_inspect(reg_a);
	printf("  sc reg A: %s\n", StringValueCStr(rstr));
	rstr = rb_inspect(reg_b);
	printf("  sc reg B: %s\n", StringValueCStr(rstr));
    }
#endif
    printf
	("--------------------------------------------------------------\n");
#endif

#if VMDEBUG > 9
    GC_CHECK();
#endif
}

#ifdef COLLECT_USAGE_ANALYSIS
/* uh = {
 *   insn(Fixnum) => ihash(Hash)
 * }
 * ihash = {
 *   -1(Fixnum) => count,      # insn usage
 *    0(Fixnum) => ophash,     # operand usage
 * }
 * ophash = {
 *   val(interned string) => count(Fixnum)
 * }
 */
void
vm_analysis_insn(int insn)
{
    static ID usage_hash;
    static ID bigram_hash;
    static int prev_insn = -1;

    VALUE uh;
    VALUE ihash;
    VALUE cv;

    SET_YARV_STOP();

    if (usage_hash == 0) {
	usage_hash = rb_intern("USAGE_ANALISYS_INSN");
	bigram_hash = rb_intern("USAGE_ANALISYS_INSN_BIGRAM");
    }
    uh = rb_const_get(mYarvCore, usage_hash);
    if ((ihash = rb_hash_aref(uh, INT2FIX(insn))) == Qnil) {
	ihash = rb_hash_new();
	rb_hash_aset(uh, INT2FIX(insn), ihash);
    }
    if ((cv = rb_hash_aref(ihash, INT2FIX(-1))) == Qnil) {
	cv = INT2FIX(0);
    }
    rb_hash_aset(ihash, INT2FIX(-1), INT2FIX(FIX2INT(cv) + 1));

    /* calc bigram */
    if (prev_insn != -1) {
	VALUE bi;
	VALUE ary[2];
	VALUE cv;

	ary[0] = INT2FIX(prev_insn);
	ary[1] = INT2FIX(insn);
	bi = rb_ary_new4(2, &ary[0]);

	uh = rb_const_get(mYarvCore, bigram_hash);
	if ((cv = rb_hash_aref(uh, bi)) == Qnil) {
	    cv = INT2FIX(0);
	}
	rb_hash_aset(uh, bi, INT2FIX(FIX2INT(cv) + 1));
    }
    prev_insn = insn;

    SET_YARV_START();
}

/* from disasm.c */
extern VALUE insn_operand_intern(int insn, int op_no, VALUE op,
				 int len, int pos, VALUE child);

void
vm_analysis_operand(int insn, int n, VALUE op)
{
    static ID usage_hash;

    VALUE uh;
    VALUE ihash;
    VALUE ophash;
    VALUE valstr;
    VALUE cv;

    SET_YARV_STOP();

    if (usage_hash == 0) {
	usage_hash = rb_intern("USAGE_ANALISYS_INSN");
    }

    uh = rb_const_get(mYarvCore, usage_hash);
    if ((ihash = rb_hash_aref(uh, INT2FIX(insn))) == Qnil) {
	ihash = rb_hash_new();
	rb_hash_aset(uh, INT2FIX(insn), ihash);
    }
    if ((ophash = rb_hash_aref(ihash, INT2FIX(n))) == Qnil) {
	ophash = rb_hash_new();
	rb_hash_aset(ihash, INT2FIX(n), ophash);
    }
    /* intern */
    valstr = insn_operand_intern(insn, n, op, 0, 0, 0);

    /* set count */
    if ((cv = rb_hash_aref(ophash, valstr)) == Qnil) {
	cv = INT2FIX(0);
    }
    rb_hash_aset(ophash, valstr, INT2FIX(FIX2INT(cv) + 1));

    SET_YARV_START();
}

void
vm_analysis_register(int reg, int isset)
{
    static ID usage_hash;
    VALUE uh;
    VALUE rhash;
    VALUE valstr;
    char *regstrs[] = {
	"pc",			// 0
	"sp",			// 1
	"cfp",			// 2
	"lfp",			// 3
	"dfp",			// 4
	"self",			// 5
	"iseq",			// 6
    };
    char *getsetstr[] = {
	"get",
	"set",
    };
    static VALUE syms[sizeof(regstrs) / sizeof(regstrs[0])][2];

    VALUE cv;

    SET_YARV_STOP();

    if (usage_hash == 0) {
	char buff[0x10];
	int i;

	usage_hash = rb_intern("USAGE_ANALISYS_REGS");

	for (i = 0; i < sizeof(regstrs) / sizeof(regstrs[0]); i++) {
	    int j;
	    for (j = 0; j < 2; j++) {
		snprintf(buff, 0x10, "%d %s %-4s", i, getsetstr[j],
			 regstrs[i]);
		syms[i][j] = ID2SYM(rb_intern(buff));
	    }
	}
    }
    valstr = syms[reg][isset];

    uh = rb_const_get(mYarvCore, usage_hash);
    if ((cv = rb_hash_aref(uh, valstr)) == Qnil) {
	cv = INT2FIX(0);
    }
    rb_hash_aset(uh, valstr, INT2FIX(FIX2INT(cv) + 1));

    SET_YARV_START();
}


#endif


VALUE
thread_dump_state(VALUE self)
{
    yarv_thread_t *th;
    yarv_control_frame_t *cfp;
    GetThreadVal(self, th);
    cfp = th->cfp;

    printf("Thread state dump:\n");
    printf("pc : %p, sp : %p\n", cfp->pc, cfp->sp);
    printf("cfp: %p, lfp: %p, dfp: %p\n", cfp, cfp->lfp, cfp->dfp);

    return Qnil;
}



void
yarv_bug()
{
    yarv_thread_t *th = GET_THREAD();
    VALUE bt;

    if (GET_THREAD()->vm) {
	int i;
	SDR();
	
	bt = th_backtrace(th, 0);
	for (i = 0; i < RARRAY(bt)->len; i++) {
	    dp(RARRAY(bt)->ptr[i]);
	}
    }
}
