﻿using System;
using System.Collections.Generic;
using System.Text;
using MinorShift.Emuera.GameData.Expression;
using MinorShift.Emuera.GameData;
using System.Text.RegularExpressions;

namespace MinorShift.Emuera.Sub
{
	enum LexEndWith
	{
		//いずれにせよEoLで強制終了
		None = 0,
		EoL,//常に最後まで解析
		Operator,//演算子を見つけたら終了。代入式の左辺
		Question,//三項演算子?により終了。\@～～?～～#～～\@
		Percent,//%により終了。%～～%
		RightCurlyBrace,//}により終了。{～～}
		Comma,//,により終了。TIMES第一引数
		Single,//Identifier一つで終了
	}

	enum FormStrEndWith
	{
		//いずれにせよEoLで強制終了
		None = 0,
		EoL,//常に最後まで解析
		DoubleQuotation,//"で終了。@"～～"
		Sharp,//#で終了。\@～～?～～#～～\@　の一つ目
		YenAt,//\@で終了。\@～～?～～#～～\@　の二つ目
		Comma,//,により終了。ANY_FORM引数
		LeftParenthesis_Bracket_Comma,//[または(または,により終了。CALLFORM系の関数名部分。
	}

	enum StrEndWith
	{
		//いずれにせよEoLで強制終了
		None = 0,
		EoL,//常に最後まで解析
		DoubleQuotation,//"で終了。"～～"
		Comma,//,により終了。PRINTV'～～,
		LeftParenthesis_Bracket_Comma,//[または(または,により終了。関数名部分。
	}

	/// <summary>
	/// 1756 TokenReaderより改名
	/// Lexicalといいつつ構文解析を含む
	/// </summary>
	internal static class LexicalAnalyzer
	{


		//readonly static IList<char> operators = new char[] { '+', '-', '*', '/', '%', '=', '!', '<', '>', '|', '&', '^', '~', '?', '#' };
		//readonly static IList<char> whiteSpaces = new char[] { ' ', '　', '\t' };
		//readonly static IList<char> endOfExpression = new char[] { ')', '}', ']', ',', ':' };
		//readonly static IList<char> startOfExpression = new char[] { '(' };
		//readonly static IList<char> stringToken = new char[] { '\"', };
		//readonly static IList<char> stringFormToken = new char[] { '@', };
		//readonly static IList<char> etcSymbol = new char[] { '[', '{', '$', '\\', };
		//readonly static IList<char> decimalDigits = new char[] { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', };
		readonly static IList<char> hexadecimalDigits = new char[] { 'a', 'b', 'c', 'd', 'e', 'f', 'A', 'B', 'C', 'D', 'E', 'F' };

		#region read
		public static Int64 ReadInt64(StringStream st, bool retZero)
		{
			Int64 significand = 0;
			int expBase = 0;
			int exponent = 0;
			int stStartPos = st.CurrentPosition;
			int stEndPos = st.CurrentPosition;
			int fromBase = 10;
			if (st.Current == '0')
			{
				char c = st.Next;
				if ((c == 'x') || (c == 'X'))
				{
					fromBase = 16;
					st.ShiftNext();
					st.ShiftNext();
				}
				else if ((c == 'b') || (c == 'B'))
				{
					fromBase = 2;
					st.ShiftNext();
					st.ShiftNext();
				}
				//8進法は互換性の問題から採用しない。
				//else if (dchar.IsDigit(c))
				//{
				//    fromBase = 8;
				//    st.ShiftNext();
				//}
			}
			if (retZero && st.Current != '+' && st.Current != '-' && !char.IsDigit(st.Current))
			{
				if (fromBase != 16)
					return 0;
				else if (!hexadecimalDigits.Contains(st.Current))
					return 0;
			}
			significand = readDigits(st, fromBase);
			if ((st.Current == 'p') || (st.Current == 'P'))
				expBase = 2;
			else if ((st.Current == 'e') || (st.Current == 'E'))
				expBase = 10;
			if (expBase != 0)
			{
				st.ShiftNext();
				unchecked { exponent = (int)readDigits(st, fromBase); }
			}
			stEndPos = st.CurrentPosition;
			if ((expBase != 0) && (exponent != 0))
			{

				double d = significand * Math.Pow(expBase, exponent);
				if ((double.IsNaN(d)) || (double.IsInfinity(d)) || (d > Int64.MaxValue) || (d < Int64.MinValue))
					throw new CodeEE("\"" + st.Substring(stStartPos, stEndPos) + "\"は64ビット符号付整数の範囲を超えています");
				significand = (Int64)d;
			}
			return significand;
		}
		//static Regex reg = new Regex(@"[0-9A-Fa-f]+", RegexOptions.Compiled);
		private static Int64 readDigits(StringStream st, int fromBase)
		{
			int start = st.CurrentPosition;
			//1756 正規表現を使ってみたがほぼ変わらなかったので没
			//Match m = reg.Match(st.RowString, st.CurrentPosition);
			//st.Jump(m.Length);
			char c = st.Current;
			if ((c == '-') || (c == '+'))
			{
				st.ShiftNext();
			}
			if (fromBase == 10)
			{
				while (!st.EOS)
				{
					c = st.Current;
					if (char.IsDigit(c))
					{
						st.ShiftNext();
						continue;
					}
					break;
				}
			}
			else if (fromBase == 16)
			{
				while (!st.EOS)
				{
					c = st.Current;
					if (char.IsDigit(c) || hexadecimalDigits.Contains(c))
					{
						st.ShiftNext();
						continue;
					}
					break;
				}
			}
			else if (fromBase == 2)
			{
				while (!st.EOS)
				{
					c = st.Current;
					if (char.IsDigit(c))
					{
						if ((c != '0') && (c != '1'))
							throw new CodeEE("二進法表記の中で使用できない文字が使われています");
						st.ShiftNext();
						continue;
					}
					break;
				}
			}
			string strInt = st.Substring(start, st.CurrentPosition - start);
			try
			{
				return Convert.ToInt64(strInt, fromBase);
			}
			catch (FormatException)
			{
				throw new CodeEE("\"" + strInt + "\"は整数値に変換できません");
			}
			catch (OverflowException)
			{
				throw new CodeEE("\"" + strInt + "\"は64ビット符号付き整数の範囲を超えています");
			}
			catch (ArgumentOutOfRangeException)
			{
				throw new CodeEE("文字列\"" + strInt + "\"は数値として認識できません");
			}
		}

		/// <summary>
		/// TIMES第二引数のみが使用する。
		/// Convertクラスが発行する例外をそのまま投げるので適切に処理すること。
		/// </summary>
		/// <param name="st"></param>
		/// <returns></returns>
		public static double ReadDouble(StringStream st)
		{
			int start = st.CurrentPosition;
			//大雑把に読み込んでエラー処理はConvertクラスに任せる。
			//仮数小数部

			if (st.Current == '-')
			{
				st.ShiftNext();
			}
			while (!st.EOS)
			{//仮数部
				char c = st.Current;
				if (char.IsDigit(c) || (c == '.'))
				{
					st.ShiftNext();
					continue;
				}
				break;
			}
			if ((st.Current == 'e') || (st.Current == 'E'))
			{
				st.ShiftNext();
				if (st.Current == '-')
				{
					st.ShiftNext();
				}
				while (!st.EOS)
				{//指数部
					char c = st.Current;
					if (char.IsDigit(c) || (c == '.'))
					{
						st.ShiftNext();
						continue;
					}
					break;
				}
			}
			return Convert.ToDouble(st.Substring(start, st.CurrentPosition - start));
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="st"></param>
		/// <returns></returns>
		public static string ReadSingleIdentifer(StringStream st)
		{
			int start = st.CurrentPosition;
			while (!st.EOS)
			{
				switch (st.Current)
				{
					case ' ':
					case '\t':
					case '+':
					case '-':
					case '*':
					case '/':
					case '%':
					case '=':
					case '!':
					case '<':
					case '>':
					case '|':
					case '&':
					case '^':
					case '~':
					case '?':
					case '#':
					case ')':
					case '}':
					case ']':
					case ',':
					case ':':
					case '(':
					case '{':
					case '[':
					case '$':
					case '\\':
					case ';':
					case '\'':
					case '\"':
					case '@':
					case '.':
						goto end;
					case '　':
						if (!Config.SystemAllowFullSpace)
							throw new CodeEE("予期しない全角スペースを発見しました");
						goto end;
				}
				st.ShiftNext();
			}
		end:
			return st.Substring(start, st.CurrentPosition - start);
		}

		/// <summary>
		/// endWithが見つかるまで読み込む。始点と終端のチェックは呼び出し側で行うこと。
		/// エスケープあり。
		/// </summary>
		/// <param name="st"></param>
		/// <returns></returns>
		public static string ReadString(StringStream st, StrEndWith endWith)
		{
			StringBuilder buffer = new StringBuilder(100);
			while (true)
			{
				switch (st.Current)
				{
					case '\0':
						goto end;
					case '\"':
						if (endWith == StrEndWith.DoubleQuotation)
							goto end;
						break;
					case ',':
						if ((endWith == StrEndWith.Comma) || (endWith == StrEndWith.LeftParenthesis_Bracket_Comma))
							goto end;
						break;
					case '(':
					case '[':
						if (endWith == StrEndWith.LeftParenthesis_Bracket_Comma)
							goto end;
						break;
					case '\\'://エスケープ処理
						st.ShiftNext();//\を読み飛ばす
						switch (st.Current)
						{
							case StringStream.EndOfString:
								throw new CodeEE("エスケープ文字\\の後に文字がありません");
							case '\n': break;
							case 's': buffer.Append(' '); break;
							case 'S': buffer.Append('　'); break;
							case 't': buffer.Append('\t'); break;
							case 'n': buffer.Append('\n'); break;
							default: buffer.Append(st.Current); break;
						}
						st.ShiftNext();//\の次の文字を読み飛ばす
						continue;
				}
				buffer.Append(st.Current);
				st.ShiftNext();
			}
		end:
			return buffer.ToString();
		}

		/// <summary>
		/// 失敗したらCodeEE。OperatorManagerには頼らない
		/// OperatorCode.Assignmentを返すことがある。
		/// </summary>
		/// <param name="st"></param>
		/// <returns></returns>
		public static OperatorCode ReadOperator(StringStream st, bool allowAssignment)
		{
			char cur = st.Current;
			st.ShiftNext();
			char next = st.Current;
			switch (cur)
			{
				case '+':
					if (next == '+')
					{
						st.ShiftNext();
						return OperatorCode.Increment;
					}
					return OperatorCode.Plus;
				case '-':
					if (next == '-')
					{
						st.ShiftNext();
						return OperatorCode.Decrement;
					}
					return OperatorCode.Minus;
				case '*':
					return OperatorCode.Mult;
				case '/':
					return OperatorCode.Div;
				case '%':
					return OperatorCode.Mod;
				case '=':
					if (next == '=')
					{
						st.ShiftNext();
						return OperatorCode.Equal;
					}
					if (allowAssignment)
						return OperatorCode.Assignment;
					throw new CodeEE("予期しない代入演算子'='を発見しました(等価比較には'=='を使用してください)");
				case '!':
					if (next == '=')
					{
						st.ShiftNext();
						return OperatorCode.NotEqual;
					}
					else if (next == '&')
					{
						st.ShiftNext();
						return OperatorCode.Nand;
					}
					else if (next == '|')
					{
						st.ShiftNext();
						return OperatorCode.Nor;
					}
					return OperatorCode.Not;
				case '<':
					if (next == '=')
					{
						st.ShiftNext();
						return OperatorCode.LessEqual;
					}
					else if (next == '<')
					{
						st.ShiftNext();
						return OperatorCode.LeftShift;
					}
					return OperatorCode.Less;
				case '>':
					if (next == '=')
					{
						st.ShiftNext();
						return OperatorCode.GreaterEqual;
					}
					else if (next == '>')
					{
						st.ShiftNext();
						return OperatorCode.RightShift;
					}
					return OperatorCode.Greater;
				case '|':
					if (next == '|')
					{
						st.ShiftNext();
						return OperatorCode.Or;
					}
					return OperatorCode.BitOr;
				case '&':
					if (next == '&')
					{
						st.ShiftNext();
						return OperatorCode.And;
					}
					return OperatorCode.BitAnd;
				case '^':
					if (next == '^')
					{
						st.ShiftNext();
						return OperatorCode.Xor;
					}
					return OperatorCode.BitXor;
				case '~':
					return OperatorCode.BitNot;
				case '?':
					return OperatorCode.Ternary_a;
				case '#':
					return OperatorCode.Ternary_b;

			}
			throw new CodeEE("'" + cur + "'は演算子として認識できません");
		}

		/// <summary>
		/// 失敗したらCodeEE。OperatorManagerには頼らない
		/// "="の時、OperatorCode.Assignmentを返す。"=="の時はEqual
		/// </summary>
		/// <param name="st"></param>
		/// <returns></returns>
		public static OperatorCode ReadAssignmentOperator(StringStream st)
		{
			OperatorCode ret = OperatorCode.NULL;
			char cur = st.Current;
			st.ShiftNext();
			char next = st.Current;
			switch (cur)
			{
				case '+':
					if (next == '+')
						ret = OperatorCode.Increment;
					else if (next == '=')
						ret = OperatorCode.Plus;
					break;
				case '-':
					if (next == '-')
						ret = OperatorCode.Decrement;
					else if (next == '=')
						ret = OperatorCode.Minus;
					break;
				case '*':
					if (next == '=')
						ret = OperatorCode.Mult;
					break;
				case '/':
					if (next == '=')
						ret = OperatorCode.Div;
					break;
				case '%':
					if (next == '=')
						ret = OperatorCode.Mod;
					break;
				case '=':
					if (next == '=')
					{
						ret = OperatorCode.Equal;
						break;
					}
					return OperatorCode.Assignment;
				case '<':
					if (next == '<')
					{
						st.ShiftNext();
						if (st.Current == '=')
						{
							ret = OperatorCode.LeftShift;
							break;
						}
						throw new CodeEE("'<'は代入演算子として認識できません");
					}
					break;
				case '>':
					if (next == '>')
					{
						st.ShiftNext();
						if (st.Current == '=')
						{
							ret = OperatorCode.RightShift;
							break;
						}
						throw new CodeEE("'>'は代入演算子として認識できません");
					}
					break;
				case '|':
					if (next == '=')
						ret = OperatorCode.BitOr;
					break;
				case '&':
					if (next == '=')
						ret = OperatorCode.BitAnd;
					break;
				case '^':
					if (next == '=')
						ret = OperatorCode.BitXor;
					break;
			}
			if (ret == OperatorCode.NULL)
				throw new CodeEE("'" + cur + "'は代入演算子として認識できません");
			st.ShiftNext();
			return ret;
		}


		public static int SkipAllSpace(StringStream st)
		{
			int count = 0;
			while (true)
			{
				switch (st.Current)
				{
					case ' ':
					case '\t':
					case '　':
						count++;
						st.ShiftNext();
						continue;
				}
				return count;
			}
		}

		public static bool IsWhiteSpace(char c)
		{
			return c == ' ' || c == '\t' || c == '　';
		}

		public static int SkipWhiteSpace(StringStream st)
		{
			int count = 0;
			while (true)
			{
				switch (st.Current)
				{
					case ' ':
					case '\t':
						count++;
						st.ShiftNext();
						continue;
					case '　':
						if (!Config.SystemAllowFullSpace)
							return count;
							
							//throw new CodeEE("予期しない全角スペースを発見しました");
						goto case ' ';
				}
				return count;
			}
		}

		public static int SkipHalfSpace(StringStream st)
		{
			int count = 0;
			while (st.Current == ' ')
			{
				count++;
				st.ShiftNext();
			}
			return count;
		}
		#endregion

		#region analyse

		/// <summary>
		/// 解析できるものは関数宣言や式のみ。FORM文字列や普通の文字列を送ってはいけない
		/// return時にはendWithの文字がCurrentになっているはず。終端の適切さの検証は呼び出し元が行う。
		/// </summary>
		/// <returns></returns>
		public static WordCollection Analyse(StringStream st, LexEndWith endWith, bool isPrintV, bool allowAssignment)
		{
			WordCollection ret = new WordCollection();
			int nestBracketS = 0;
			//int nestBracketM = 0;
			int nestBracketL = 0;
			while (true)
			{
				switch (st.Current)
				{
					case '\n':
					case '\0':
						goto end;
					case ' ':
					case '\t':
						st.ShiftNext();
						continue;
					case '　':
						if (!Config.SystemAllowFullSpace)
							throw new CodeEE("字句解析中に予期しない全角スペースを発見しました");
						st.ShiftNext();
						continue;
					case '0':
					case '1':
					case '2':
					case '3':
					case '4':
					case '5':
					case '6':
					case '7':
					case '8':
					case '9':
						ret.Add(new LiteralIntegerWord(ReadInt64(st, false)));
						break;
					case '+':
					case '-':
					case '*':
					case '/':
					case '%':
					case '=':
					case '!':
					case '<':
					case '>':
					case '|':
					case '&':
					case '^':
					case '~':
					case '?':
					case '#':
						if ((nestBracketS == 0) && (nestBracketL == 0))
						{
							if (endWith == LexEndWith.Operator)
								goto end;//代入演算子のはずである。呼び出し元がチェックするはず
							else if ((endWith == LexEndWith.Percent) && (st.Current == '%'))
								goto end;
							else if ((endWith == LexEndWith.Question) && (st.Current == '?'))
								goto end;
						}
						ret.Add(new OperatorWord(ReadOperator(st, allowAssignment)));
						break;
					case ')': ret.Add(new SymbolWord(')')); nestBracketS--; st.ShiftNext(); continue;
					case ']': ret.Add(new SymbolWord(']')); nestBracketL--; st.ShiftNext(); continue;
					case '(': ret.Add(new SymbolWord('(')); nestBracketS++; st.ShiftNext(); continue;
					case '[':
						if (st.Next == '[')
						{
							if (ParserMediator.RenameDic == null)
								throw new CodeEE("字句解析中に予期しない文字\"[[\"を発見しました");
							int start = st.CurrentPosition;
							int find = st.Find("]]");
							if (find <= 2)
							{
								if (find == 2)
									throw new CodeEE("空の[[]]です");
								else
									throw new CodeEE("対応する\"]]\"のない\"[[\"です");
							}
							string key = st.Substring(start, find + 2);
							string value = null;
							if (!ParserMediator.RenameDic.TryGetValue(key, out value))
								throw new CodeEE("字句解析中に置換できない符号" + key + "を発見しました");
							st.Replace(start, find + 2, value);
							continue;//その場から再度解析スタート
						}
						ret.Add(new SymbolWord('[')); nestBracketL++; st.ShiftNext(); continue;
					case ':': ret.Add(new SymbolWord(':')); st.ShiftNext(); continue;
					case ',':
						if ((endWith == LexEndWith.Comma) && (nestBracketS == 0) && (nestBracketL == 0))
							goto end;
						ret.Add(new SymbolWord(',')); st.ShiftNext(); continue;
					//case '}': ret.Add(new SymbolWT('}')); nestBracketM--; break;
					//case '{': ret.Add(new SymbolWT('{')); nestBracketM++; break;
					case '\'':
						if (!isPrintV)
							throw new CodeEE("字句解析中に予期しない文字'" + st.Current + "'を発見しました");
						st.ShiftNext();
						ret.Add(new LiteralStringWord(ReadString(st, StrEndWith.Comma)));
						if (st.Current == ',')
							goto case ',';//続きがあるなら,の処理へ。それ以外は行終端のはず
						goto end;
					case '}':
						if (endWith == LexEndWith.RightCurlyBrace)
							goto end;
						throw new CodeEE("字句解析中に予期しない文字'" + st.Current + "'を発見しました");
					case '\"':
						st.ShiftNext();
						ret.Add(new LiteralStringWord(ReadString(st, StrEndWith.DoubleQuotation)));
						if (st.Current != '\"')
							throw new CodeEE("\"が閉じられていません");
						st.ShiftNext();
						break;
					case '@':
						if (st.Next != '\"')
						{
							ret.Add(new SymbolWord('@'));
							st.ShiftNext();
							continue;
						}
						st.ShiftNext();
						st.ShiftNext();
						ret.Add(AnalyseFormattedString(st, FormStrEndWith.DoubleQuotation, false));
						if (st.Current != '\"')
							throw new CodeEE("\"が閉じられていません");
						st.ShiftNext();
						break;

					case '\\':
						if (st.Next != '@')
							throw new CodeEE("字句解析中に予期しない文字'" + st.Current + "'を発見しました");
						{
							st.Jump(2);
							ret.Add(new StrFormWord(new string[] { "", "" }, new SubWord[] { AnalyseYenAt(st) }));
						}
						break;
					case '{':
					case '$':
					case ';':
					case '.':
						throw new CodeEE("字句解析中に予期しない文字'" + st.Current + "'を発見しました");
					default:
						ret.Add(new IdentifierWord(ReadSingleIdentifer(st))); break;
				}
				if (endWith == LexEndWith.Single)
					return ret;//Singleなら()の対応チェックは呼び出し元で行う
			}
		end:
			if ((nestBracketS != 0) || (nestBracketL != 0))
			{
				if (nestBracketS < 0)
					throw new CodeEE("対応する'('のない')'です");
				else if (nestBracketS > 0)
					throw new CodeEE("対応する')'のない'('です");
				if (nestBracketL < 0)
					throw new CodeEE("対応する'['のない']'です");
				else if (nestBracketL > 0)
					throw new CodeEE("対応する']'のない'['です");
			}
			return ret;

		}

		/// <summary>
		/// @"などの直後からの開始
		/// return時にはendWithの文字がCurrentになっているはず。終端の適切さの検証は呼び出し元が行う。
		/// </summary>
		/// <returns></returns>
		public static StrFormWord AnalyseFormattedString(StringStream st, FormStrEndWith endWith, bool trim)
		{
			List<string> strs = new List<string>();
			List<SubWord> SWTs = new List<SubWord>();
			StringBuilder buffer = new StringBuilder(100);
			while (true)
			{
				char cur = st.Current;
				switch (cur)
				{
					case '\n':
					case '\0':
						goto end;
					case '\"':
						if (endWith == FormStrEndWith.DoubleQuotation)
							goto end;
						buffer.Append(cur);
						break;
					case '#':
						if (endWith == FormStrEndWith.Sharp)
							goto end;
						buffer.Append(cur);
						break;
					case ',':
						if ((endWith == FormStrEndWith.Comma) || (endWith == FormStrEndWith.LeftParenthesis_Bracket_Comma))
							goto end;
						buffer.Append(cur);
						break;
					case '(':
					case '[':
						if (endWith == FormStrEndWith.LeftParenthesis_Bracket_Comma)
							goto end;
						buffer.Append(cur);
						break;
					case '%':
						strs.Add(buffer.ToString());
						buffer.Remove(0, buffer.Length);
						st.ShiftNext();
						SWTs.Add(new PercentSubWord(Analyse(st, LexEndWith.Percent, false, false)));
						if (st.Current != '%')
							throw new CodeEE("\'%\'が使われましたが対応する\'%\'が見つかりません");
						break;
					case '{':
						strs.Add(buffer.ToString());
						buffer.Remove(0, buffer.Length);
						st.ShiftNext();
						SWTs.Add(new CurlyBraceSubWord(Analyse(st, LexEndWith.RightCurlyBrace, false, false)));
						if (st.Current != '}')
							throw new CodeEE("\'{\'が使われましたが対応する\'}\'が見つかりません");
						break;
					case '*':
					case '+':
					case '=':
					case '/':
					case '$':
						if (st.TripleSymbol())
						{
							strs.Add(buffer.ToString());
							buffer.Remove(0, buffer.Length);
							st.Jump(3);
							SWTs.Add(new TripleSymbolSubWord(cur));
							continue;
						}
						else
							buffer.Append(cur);
						break;
					case '\\'://エスケープ文字の使用

						st.ShiftNext();
						cur = st.Current;
						switch (cur)
						{
							case '\0':
								throw new CodeEE("エスケープ文字\\の後に文字がありません");
							case '\n': break;
							case 's': buffer.Append(' '); break;
							case 'S': buffer.Append('　'); break;
							case 't': buffer.Append('\t'); break;
							case 'n': buffer.Append('\n'); break;
							case '@'://\@～～?～～#～～\@
								{
									if ((endWith == FormStrEndWith.YenAt) || (endWith == FormStrEndWith.Sharp))
										goto end;
									strs.Add(buffer.ToString());
									buffer.Remove(0, buffer.Length);
									st.ShiftNext();
									SWTs.Add(AnalyseYenAt(st));
									continue;
								}
							default:
								buffer.Append(cur);
								st.ShiftNext();
								continue;
						}
						break;
					default:
						buffer.Append(cur);
						break;
				}
				st.ShiftNext();
			}
		end:
			strs.Add(buffer.ToString());

			string[] retStr = new string[strs.Count];
			SubWord[] retSWTs = new SubWord[SWTs.Count];
			strs.CopyTo(retStr);
			SWTs.CopyTo(retSWTs);
			if (trim && retStr.Length > 0)
			{
				retStr[0] = retStr[0].TrimStart(new char[] { ' ', '\t' });
				retStr[retStr.Length - 1] = retStr[retStr.Length - 1].TrimEnd(new char[] { ' ', '\t' });
			}
			return new StrFormWord(retStr, retSWTs);
		}



		/// <summary>
		/// \@直後からの開始、\@の直後がCurrentになる
		/// </summary>
		/// <param name="st"></param>
		/// <returns></returns>
		public static YenAtSubWord AnalyseYenAt(StringStream st)
		{
			WordCollection w = Analyse(st, LexEndWith.Question, false, false);
			if (st.Current != '?')
				throw new CodeEE("\'\\@\'が使われましたが対応する\'?\'が見つかりません");
			st.ShiftNext();
			StrFormWord left = AnalyseFormattedString(st, FormStrEndWith.Sharp, true);
			if (st.Current != '#')
			{
				if (st.Current != '@')
					throw new CodeEE("\'\\@\',\'?\'が使われましたが対応する\'#\'が見つかりません");
				ParserMediator.Warn("\'\\@\',\'?\'が使われましたが対応する\'#\'が見つかりません", GlobalStatic.Process.GetScaningLine(), 1, false, false);
				return new YenAtSubWord(w, left, null);
			}
			st.ShiftNext();
			StrFormWord right = AnalyseFormattedString(st, FormStrEndWith.YenAt, true);
			if (st.Current != '@')
				throw new CodeEE("\'\\@\',\'?\',\'#\'が使われましたが対応する\'\\@\'が見つかりません");
			st.ShiftNext();
			return new YenAtSubWord(w, left, right);
		}

		#endregion

	}
}