﻿using System;
using System.Collections.Generic;
using System.Text;
using MinorShift.Emuera.Sub;
using MinorShift.Emuera.GameData.Variable;
using MinorShift.Emuera.GameData.Function;
using MinorShift.Emuera.GameProc;
using MinorShift.Emuera.GameView;
using System.IO;
using System.Text.RegularExpressions;
using MinorShift.Emuera.GameProc.Function;
using MinorShift.Emuera.GameData.Expression;

namespace MinorShift.Emuera
{
	//1756 新設。
	//また、使用されている名前を記憶し衝突を検出する。
	//作成途中
	internal sealed class IdentifierDictionary
	{
		private enum SystemNameType
		{
			None = 0,
			Reserved,
			SystemVariable,
			SystemMethod,
			SystemInstrument,
		}
		private enum DefineNameType
		{
			None = 0,
			UserIdentifier,
			UserGlobalVariable,
			UserMacro,
		}
		readonly static char[] badSymbolAsIdentifier = new char[]
		{
			'+', '-', '*', '/', '%', '=', '!', '<', '>', '|', '&', '^', '~',
			' ', '　', '\t' ,
			'\"','(', ')', '{', '}', '[', ']', ',', '.', ':',
			'\\', '@', '$', '#', '?', ';', '\'',
			//'_'はOK
		};
		readonly static Regex regexCom = new Regex("^COM[0-9]+$");
		readonly static Regex regexComAble = new Regex("^COM_ABLE[0-9]+$");
		readonly static Regex regexAblup = new Regex("^ABLUP[0-9]+$");
		#region static
		
		public static bool IsEventLabelName(string labelName)
		{
			switch (labelName)
			{
				case "EVENTFIRST":
				case "EVENTTRAIN":
				case "EVENTSHOP":
				case "EVENTBUY":
				case "EVENTCOM":
				case "EVENTTURNEND":
				case "EVENTCOMEND":
				case "EVENTEND":
				case "EVENTLOAD":
					return true;
			}
			return false;
		}
		public static bool IsSystemLabelName(string labelName)
		{
			switch (labelName)
			{
				case "EVENTFIRST":
				case "EVENTTRAIN":
				case "EVENTSHOP":
				case "EVENTBUY":
				case "EVENTCOM":
				case "EVENTTURNEND":
				case "EVENTCOMEND":
				case "EVENTEND":
				case "SHOW_STATUS":
				case "SHOW_USERCOM":
				case "USERCOM":
				case "SOURCE_CHECK":
				case "CALLTRAINTEND":
				case "SHOW_JUEL":
				case "SHOW_ABLUP_SELECT":
				case "USERABLUP":
				case "SHOW_SHOP":
				case "SAVEINFO":
				case "USERSHOP":

				case "EVENTLOAD":
				case "TITLE_LOADGAME":
				case "SYSTEM_AUTOSAVE":
				case "SYSTEM_TITLE":
					return true;
			}

			if (labelName.StartsWith("COM"))
			{
				if (regexCom.IsMatch(labelName))
					return true;
				if (regexComAble.IsMatch(labelName))
					return true;
			}
			if (labelName.StartsWith("ABLUP"))
				if (regexAblup.IsMatch(labelName))
					return true;
			return false;
		}
		#endregion


		Dictionary<string, SystemNameType> nameDic = new Dictionary<string, SystemNameType>();
		Dictionary<string, DefineNameType> defineDic = new Dictionary<string, DefineNameType>();

		List<string> privateDimList = new List<string>();
		List<string> disableList = new List<string>();
		Dictionary<string, VariableToken> userDefinedVarDic = new Dictionary<string, VariableToken>();

		VariableData varData;
		Dictionary<string, VariableToken> varTokenDic;
		Dictionary<string, VariableLocal> localvarTokenDic;
		Dictionary<string, FunctionIdentifier> instructionDic;
		Dictionary<string, FunctionMethod> methodDic;
		#region initialize
		public IdentifierDictionary(VariableData varData)
		{
			this.varData = varData;
			nameDic.Clear();
			//予約語を登録。式中に登場すると構文解析が崩壊する名前群。
			//ただしeramaker用スクリプトなら特に気にすることはない。式中に出てこない単語も同様。
			nameDic.Add("IS", SystemNameType.Reserved);
			nameDic.Add("TO", SystemNameType.Reserved);
			nameDic.Add("STATIC", SystemNameType.Reserved);
			nameDic.Add("DYNAMIC", SystemNameType.Reserved);
			instructionDic = FunctionIdentifier.GetInstructionNameDic();

			varTokenDic = varData.GetVarTokenDic();
			localvarTokenDic = varData.GetLocalvarTokenDic();
			methodDic = FunctionMethodCreator.GetMethodList();

			foreach(KeyValuePair<string, FunctionMethod> pair in methodDic)
			{
				nameDic.Add(pair.Key, SystemNameType.SystemMethod);
			}

			foreach (KeyValuePair<string, VariableToken> pair in varTokenDic)
			{
				//RANDが衝突している
				if (!nameDic.ContainsKey(pair.Key)) 
					nameDic.Add(pair.Key, SystemNameType.SystemVariable);
			}

			foreach (KeyValuePair<string, VariableLocal> pair in localvarTokenDic)
			{
				nameDic.Add(pair.Key, SystemNameType.SystemVariable);
			}

			foreach (KeyValuePair<string, FunctionIdentifier> pair in instructionDic)
			{
				//Methodと被る
				if (!nameDic.ContainsKey(pair.Key))
					nameDic.Add(pair.Key, SystemNameType.SystemInstrument);
			}
		}
		
		public void SetSystemInstrumentName(List<string> names)
		{
		}
		
		public void CheckUserLabelName(ref string errMes, ref int warnLevel, bool isFunction, string labelName)
		{
			if (labelName.Length == 0)
			{
				errMes = "ラベル名がありません";
				warnLevel = 2;
				return;
			}
			//1.721 記号をサポートしない方向に変更
			if (labelName.IndexOfAny(badSymbolAsIdentifier) >= 0)
			{
				errMes = "ラベル名" + labelName + "に\"_\"以外の記号が含まれています";
				warnLevel = 1;
				return;
			}
			if (char.IsDigit(labelName[0]))
			{
				errMes = "ラベル名" + labelName + "が半角数字から始まっています";
				warnLevel = 0;
				return;
			}
			if (!isFunction || !Config.WarnFunctionOverloading)
				return;
			if (!nameDic.ContainsKey(labelName))
				return;

			if (nameDic.ContainsKey(labelName))
			{
				switch (nameDic[labelName])
				{
					case SystemNameType.Reserved:
						if (Config.AllowFunctionOverloading)
						{
							errMes = "関数名" + labelName + "はEmueraの予約語と衝突しています。Emuera専用構文の構文解析に支障をきたす恐れがあります";
							warnLevel = 1;
						}
						else
						{
							errMes = "関数名" + labelName + "はEmueraの予約語です";
							warnLevel = 2;
						}
						break;
					case SystemNameType.SystemMethod:
						if (Config.AllowFunctionOverloading)
						{
							errMes = "関数名" + labelName + "はEmueraの式中関数を上書きします";
							warnLevel = 1;
						}
						else
						{
							errMes = "関数名" + labelName + "はEmueraの式中関数名として使われています";
							warnLevel = 2;
						}
						break;
					case SystemNameType.SystemVariable:
						errMes = "関数名" + labelName + "はEmueraの変数で使われています";
						warnLevel = 1;
						break;
					case SystemNameType.SystemInstrument:
						errMes = "関数名" + labelName + "はEmueraの変数もしくは命令で使われています";
						warnLevel = 1;
						break;
				}
			}
		}
		
		public void CheckUserVarName(ref string errMes, ref int warnLevel, string varName)
		{
			if (varName.Length == 0)
			{
				errMes = "変数名がありません";
				warnLevel = 2;
				return;
			}
			//1.721 記号をサポートしない方向に変更
			if (varName.IndexOfAny(badSymbolAsIdentifier) >= 0)
			{
				errMes = "変数名" + varName + "に\"_\"以外の記号が含まれています";
				warnLevel = 2;
				return;
			}
			if (char.IsDigit(varName[0]))
			{
				errMes = "変数名" + varName + "が半角数字から始まっています";
				warnLevel = 2;
				return;
			}

			if (nameDic.ContainsKey(varName))
			{
				switch (nameDic[varName])
				{
					case SystemNameType.Reserved:
						errMes = "変数名" + varName + "はEmueraの予約語です";
						warnLevel = 2;
						break;
					case SystemNameType.SystemInstrument:
					case SystemNameType.SystemMethod:
						//代入文が使えなくなるために命令名との衝突は致命的。
						errMes = "変数名" + varName + "はEmueraの命令名として使われています";
						warnLevel = 2;
						break;
					case SystemNameType.SystemVariable:
						errMes = "変数名" + varName + "はEmueraの変数名として使われています";
						warnLevel = 2;
						break;
				}
			}
		}
		
		public void CheckUserPrivateVarName(ref string errMes, ref int warnLevel, string varName)
		{
			if (varName.Length == 0)
			{
				errMes = "変数名がありません";
				warnLevel = 2;
				return;
			}
			//1.721 記号をサポートしない方向に変更
			if (varName.IndexOfAny(badSymbolAsIdentifier) >= 0)
			{
				errMes = "変数名" + varName + "に\"_\"以外の記号が含まれています";
				warnLevel = 2;
				return;
			}
			if (char.IsDigit(varName[0]))
			{
				errMes = "変数名" + varName + "が半角数字から始まっています";
				warnLevel = 2;
				return;
			}
			if(nameDic.ContainsKey(varName))
			{
				switch(nameDic[varName])
				{
					case SystemNameType.Reserved:
						errMes = "変数名" + varName + "はEmueraの予約語です";
						warnLevel = 2;
						return;
					case SystemNameType.SystemInstrument:
					case SystemNameType.SystemMethod:
						//代入文が使えなくなるために命令名との衝突は致命的。
						errMes = "変数名" + varName + "はEmueraの命令名として使われています";
						warnLevel = 2;
						return;
					case SystemNameType.SystemVariable:
					//Private なら変数の上書きを許す。
						//errMes = "変数名" + varName + "はEmueraの変数で使われています";
						//warnLevel = 1;
						break;
				}
			}
			privateDimList.Add(varName);
		}
		#endregion

		#region header.erb
		//#define FOO ()     id to wc
		//#define BAR($1) ()     idwithargs to wc(replaced)
		//#define HOGE PUGE      id to id @不可
		//#diseble FOOBAR             
		//#dim piyo, i
		//#sdim puyo, j
		static List<string> keywordsList = new List<string>();
		private void analyzeSharpDefine(StringStream st, ScriptPosition position)
		{
			bool hasArg = false;
			bool isWord = false;
			List<string> argID = new List<string>();
			LexicalAnalyzer.SkipWhiteSpace(st);
			string srcID = LexicalAnalyzer.ReadSingleIdentifer(st);
			Word destWord = null;
			WordCollection destWc = null;
			if (string.IsNullOrEmpty(srcID))
				throw new CodeEE("置換元の識別子がありません");
			if (Config.ICVariable)
				srcID.ToUpper();
			if (keywordsList.Contains(srcID))
				throw new CodeEE("識別子" + srcID + "は既に使われています");
			keywordsList.Add(srcID);
			if (st.Current == '(')
				hasArg = true;
			else
			{
				LexicalAnalyzer.SkipWhiteSpace(st);
				if (st.Current != '(')
					isWord = true;
			}
			WordCollection wc = LexicalAnalyzer.Analyse(st, LexEndWith.EoL, false, false);
			if (hasArg)
			{
				wc.ShiftNext();//'('を読み飛ばす
				while (!wc.EOL && (wc.Current.Type != ')'))
				{
					IdentifierWord word = wc.Current as IdentifierWord;
					if (word == null)
						throw new CodeEE("置換元の引数指定の書式が間違っています");
					word.SetIsMacro();
					string id = word.Code;
					if (Config.ICVariable)
						id = id.ToUpper();
					if (argID.Contains(id))
						throw new CodeEE("置換元の引数に同じ文字が2回以上使われています");
					argID.Add(id);
					wc.ShiftNext();
					if (wc.Current.Type == ',')
					{
						wc.ShiftNext();
						continue;
					}
					throw new CodeEE("置換元の引数指定の書式が間違っています");
				}
				if (wc.EOL)
					throw new CodeEE("')'が閉じられていません");
				wc.ShiftNext();
			}
			if (wc.EOL)
				throw new CodeEE("置換先の識別子又は式がありません");
			if (isWord)
			{
				destWord = wc.Current;
				if (destWord is OperatorWord)
					throw new CodeEE("置換先を演算子にできません");
				else if (destWord is SymbolWord)
					throw new CodeEE("置換先を記号にできません");
				wc.ShiftNext();
				if (wc.EOL)
					throw new CodeEE("DEFINEの置換先は一単語であるか()でくくられていなければなりません");
				destWord.SetIsMacro();
			}
			else
			{
				destWc = new WordCollection();
				destWc.Collection.AddRange(wc.Collection.GetRange(wc.Pointer, wc.Collection.Count - wc.Pointer));
				if (destWc.Collection[destWc.Collection.Count - 1].Type != ')')
					throw new CodeEE("置換先の式が')'で終わっていません");
				destWc.SetIsMacro();
			}
			defineDic.Add(srcID, DefineNameType.UserMacro);
			throw new NotImplementedException();
		}

		private void analyzeSharpDisable(StringStream st, ScriptPosition position)
		{
			LexicalAnalyzer.SkipWhiteSpace(st);
			string srcID = LexicalAnalyzer.ReadSingleIdentifer(st);
			LexicalAnalyzer.SkipWhiteSpace(st);
			if (srcID == null)
				throw new CodeEE("対象となる識別子がありません");
			if (st.EOS)
				throw new CodeEE("DISABLE宣言の後に余分な文字があります");
			if (keywordsList.Contains(srcID))
				throw new CodeEE("識別子" + srcID + "は既にDISABLEが宣言されています");
			disableList.Add(srcID);
		}

		private void analyzeSharpDim(StringStream st, ScriptPosition position, bool dims)
		{
			LexicalAnalyzer.SkipWhiteSpace(st);
			string srcID = null;
			WordCollection wc = LexicalAnalyzer.Analyse(st, LexEndWith.EoL, false, false);
			IdentifierWord word = wc.Current as IdentifierWord;
			if (word == null)
				throw new CodeEE("置換元の識別子がありません");
			srcID = word.Code;
			if (Config.ICVariable)
				srcID.ToUpper();
			if (keywordsList.Contains(srcID))
				throw new CodeEE("識別子" + srcID + "は既に使われています");
			keywordsList.Add(srcID);
			if (wc.Current.Type != ',')
				throw new CodeEE("書式が間違っています");
			wc.ShiftNext();
			LiteralIntegerWord iWord = wc.Current as LiteralIntegerWord;
			if (iWord == null)
				throw new CodeEE("要素数の指定がありません");
			if ((iWord.Int <= 0) || (iWord.Int > 1000000))
				throw new CodeEE("要素数は1以上、100万以下で指定してください");
			wc.ShiftNext();
			if (!wc.EOL)
				throw new CodeEE("要素数の指定の後に余分な文字があります");
			VariableToken var = varData.CreateUserDefVariable(srcID, (int)iWord.Int, dims);
			userDefinedVarDic.Add(srcID, var);
			defineDic.Add(srcID, DefineNameType.UserGlobalVariable);
		}


		public bool LoadDefineFile(string filepath)
		{
			StringStream st = null;
			ScriptPosition position = null;
			EraStreamReader eReader = new EraStreamReader();
			if ((!File.Exists(filepath)) || (!eReader.Open(filepath)))
				return true;
			try
			{
				while ((st = eReader.ReadEnabledLine()) != null)
				{
					position = new ScriptPosition("_HEADER.ERB", eReader.LineNo, st.RowString);
					if (st.Current != '#')
						throw new CodeEE("_HEADER.ERBの中に#で始まらない行があります", position);
					st.ShiftNext();
					string sharpID = LexicalAnalyzer.ReadSingleIdentifer(st);
					if (string.IsNullOrEmpty(sharpID))
						throw new CodeEE("#の後に文字がありません");
					if (Config.ICFunction)
						sharpID.ToUpper();
					switch (sharpID)
					{
						case "DEFINE":
							analyzeSharpDefine(st, position);
							break;
						case "DISABLE":
							analyzeSharpDisable(st, position);
							break;
						case "DIM":
						case "DIMS":
							analyzeSharpDim(st, position, sharpID == "DIMS");
							break;
					}

				}
			}
			catch (SystemException e)
			{
				if (position != null)
					throw new CodeEE(e.Message, position);
				else
					throw new CodeEE(e.Message);
			}
			finally
			{
				eReader.Close();
			}
			return true;
		}
		#endregion

		#region get
		public VariableToken GetVariableToken(string key, string subKey, bool allowPrivate)
		{
			VariableToken ret = null;
			if (allowPrivate)
			{
				LogicalLine line = GlobalStatic.Process.GetScaningLine();
				if ((line != null) && (line.ParentLabelLine != null))
				{
					if (Config.ICVariable)
						key = key.ToUpper();
					ret = line.ParentLabelLine.GetPrivateVariable(key);
					if(ret != null)
					{
						if (subKey != null)
							throw new CodeEE("プライベート変数" + key + "に対して@が使われました");
						return ret;
					}
				}
			}
			if (localvarTokenDic.ContainsKey(key))
			{
				LogicalLine line = GlobalStatic.Process.GetScaningLine();
                if (string.IsNullOrEmpty(subKey))
				{
					//システムの入力待ち中にデバッグコマンドからLOCALを呼んだとき。
					if((line == null)||(line.ParentLabelLine == null))
						throw new CodeEE("実行中の関数が存在しないため" + key + "を取得又は変更できませんでした");
					subKey = line.ParentLabelLine.LabelName;
				}
				else if (Config.ICFunction)
				{
					subKey = subKey.ToUpper();
					ParserMediator.Warn("コード中でローカル変数を@付きで呼ぶことは推奨されません", line, 1, false, false);
				}
                LocalVariableToken retLocal = localvarTokenDic[key].GetExistLocalVariableToken(subKey);
                if (retLocal == null)
                    retLocal = localvarTokenDic[key].GetNewLocalVariableToken(subKey, line.ParentLabelLine);
                return retLocal;
			}
			if (varTokenDic.TryGetValue(key, out ret))
			{
				if (subKey != null)
					throw new CodeEE("ローカル変数でない変数" + key + "に対して@が使われました");
                return ret;
            }
			if (subKey != null)
				throw new CodeEE("@の使い方が不正です");
			return null;
		}

		public FunctionIdentifier GetFunctionIdentifier(string str)
		{
			string key = str;
			FunctionIdentifier ret = null;
			if (string.IsNullOrEmpty(key))
				return null;
			if (Config.ICFunction)
				key = key.ToUpper();
			instructionDic.TryGetValue(key, out ret);
			return ret;
		}

		public List<string> GetOverloadedList(LabelDictionary labelDic)
		{
			List<string> list = new List<string>();
			foreach (KeyValuePair<string, FunctionMethod> pair in methodDic)
			{
				FunctionLabelLine func = labelDic.GetNonEventLabel(pair.Key);
				if (func == null)
					continue;
				if (!func.IsMethod)
					continue;
				list.Add(pair.Key);
			}
			return list;
		}

		public IOperandTerm GetFunctionMethod(LabelDictionary labelDic, string codeStr, IOperandTerm[] arguments)
		{
			if (Config.ICFunction)
				codeStr = codeStr.ToUpper();
			if ((labelDic != null) && (labelDic.Initialized))
			{
				FunctionLabelLine func = labelDic.GetNonEventLabel(codeStr);
				if (func != null)
				{
					if (func.IsMethod)
					{
						CalledFunction call = CalledFunction.CreateCalledFunctionMethod(func, codeStr);
						return new UserDefinedMethodTerm(arguments, func.MethodType, call);
					}
					//1.721 #FUNCTIONが定義されていない関数は組み込み関数を上書きしない方向に。 PANCTION.ERBのRANDとか。
					if (!methodDic.ContainsKey(codeStr))
						throw new CodeEE("#FUNCTIONが定義されていない関数(" + func.Position.Filename + ":" + func.Position.LineNo + "行目)を式中で呼び出そうとしました");
				}
			}
			FunctionMethod method = null;
			if (!methodDic.TryGetValue(codeStr, out method))
				return null;
			string errmes = method.CheckArgumentType(codeStr, arguments);
			if (errmes != null)
				throw new CodeEE(errmes);
			return new FunctionMethodTerm(method, arguments);
		}

		//1756 作成中途
		//名前リストを元に何がやりたかったのかを推定してCodeEEを投げる
		public void ThrowException(string idStr, bool isFunc)
		{
			if (disableList.Contains(idStr))
				throw new CodeEE("\"" + idStr + "\"は#DISABLEが宣言されています");
			if (!isFunc && privateDimList.Contains(idStr))
				throw new CodeEE("変数\"" + idStr + "\"はこの関数中では定義されていません");
			throw new CodeEE("\"" + idStr + "\"は解釈できない識別子です");
		}
		#endregion

        #region util
        public void resizeLocalVars(string key, string subKey, int newSize)
        {
            localvarTokenDic[key].ResizeLocalVariableToken(subKey, newSize);
        }
        #endregion
    }
}