package org.j69.eewiki.wiki.transformer;

import org.j69.eewiki.util.StringUtil;

/**
 * キーワード「:」を処理する
 *
 * , : <dl><dt>[通常文字列]</dt><dd>[通常文字列]</dd></dl>
 *
 * @see DataLineCreator
 */
public class DefinitionDataLine implements DataLine {
    private String keyword_ = ":",          //キーワード（置き換え対象）
                    lineData_,                  //データ
                    dataHoldTag_ = "",          //データ自体をはさむタグ
                    outerHoldTag_ = "dl";       //データの外側をはさむタグ

    /**
     * 初期化
     *
     * @param lineData 行データ（キーワード分解前）
     **/
    public void init(String lineData) {
        // :[用語]:[説明文]となっているのでコロンで分割
        String[] cells = StringUtil.split(lineData.substring(1), ":");
        StringBuffer buf = new StringBuffer();
        if(cells.length > 0) {
            buf.append("<dt>" + cells[0] + "</dt>");
        }
        if(cells.length > 1) {
            buf.append("<dd>" + cells[1] + "</dd>");
        }
        lineData_ = buf.toString();
    }

    /**
     * 行頭キーワード取得
     *
     * @return 行頭キーワード
     **/
    public String getKeyword() {
        return keyword_;
    }

    /**
     * データの外側を挟む開始タグ（<dl>）取得
     *
     * @param oldLine 前の行データ
     * @return <dl>
     **/
    public String openTag(DataLine oldLine) {
        if(oldLine != null && oldLine instanceof DefinitionDataLine) {
            //前の行も説明文データの場合は<dl>はいらない
            return "";
        } else {
            return "<" + outerHoldTag_ + ">\n";
        }
    }

    /**
     * データの外側を挟む終了タグ（</dl>）取得
     *
     * @param newLine 次の行データ
     * @return </dl>
     **/
    public String closeTag(DataLine newLine) {
        if(newLine != null && !(newLine instanceof DefinitionDataLine)) {
            return "</" + outerHoldTag_ + ">\n";
        } else {
            //次の行も説明文データの場合は</dl>はいらない
            return "";
        }
    }

    /**
     * 行データ取得
     *
     * @return 行データ
     **/
    public String get(int row) {
        return lineData_ + "\n";
    }
}

