#　状態クラス

module DarkHall


	class State < StorableObject
		include GetText
		LT_NONE = 0     # 重ならない（同じ状態異常は失敗する）
		LT_STACK = 1    # レベル加算式
		LT_OVERRIDE = 2 # レベル上書き式
		
		AL_CANNOT_CONCENTRATE = 2 # 攻撃・呪文詠唱不可
		AL_CANNOT_ATTACK = 1      # 攻撃できない
		AL_ATTACK_ONLY = 3        # 勝手に攻撃し続ける
		AL_CANNOT_ACT = 4         # いっさい行動できない
		
	
		attr_accessor :name, :caption
		bool_attr_accessor :standing, :deactive, :unconcentratable, :uncommandable, :battle_only
		attr_accessor :level, :level_max, :level_min, :round, :lapping_type
		attr_reader :str_bonus, :tec_bonus, :agl_bonus, :psy_bonus, :wil_bonus
		attr_reader :attack_bonus, :hitting_bonus, :avoid_bonus, :speed_bonus, :damaged_rate
		attr_reader :protect_targets, :extra_defense
		
	
		
		def initialize(level = 1)
			@name = nil
			@str_bonus = 0
			@tec_bonus = 0
			@agl_bonus = 0
			@wil_bonus = 0
			@psy_bonus = 0
			@attack_bonus = 0
			@hitting_bonus = 0
			@avoid_bonus = 0
			@speed_bonus = 0
			@damaged_rate = 1.0
			
			@level = level
			@level_max = 999
			@level_min = 0
			@round = 1
			@recover_round = nil
			@lapping_type = LT_NONE
			
			@deactive = false
			@battle_only = true
			@standing = true
			@unconcentratable = false
			@uncommandable = false
			
			@cannot_attack = false
			@cannot_magic = false
			@cannot_defense = false
			
			@protect_targets = []
			@extra_defense = 0
		end
		
		def good?
			false
		end
		
		def visible?
			true
		end
		
		
		
		def caption
			if self.name then
				color = (good? ? 'GoodColor' : 'BadColor')
				if @recover_round then
					"$[#{color}]#{self.name}(#{self.rest_round})"
				else
					"$[#{color}]#{self.name}"
				end
			end
		end
		
		# クリーンヒット率補正
		def clean_hitting_bonus; 0; end
		
		# 生存補正
		def surviving_bonus; 0; end

		# 初手逃走補正
		def fast_escaping_bonus; 0; end

		# 敵逃走率補正
		def enemy_ousting_bonus; 0; end

		def rest_round
			if @recover_round then
				@recover_round - @round + 1
			else
				nil
			end
		end
		
		def check_damage_recover(target, damage)
			return false
		end
		
		def get_added_message(target)
			if target.kind_of?(Enemy) then
				get_enemy_added_message(target)
			else
				get_member_added_message(target)
			end
		end
		
		def get_member_added_message(target)
		end
		
		def get_enemy_added_message(target)
		end
		
		def get_recover_message(target)
		end
		
		def get_group_added_message(targets)
			if targets.first.kind_of?(Enemy) then
				get_enemy_group_added_message(targets)
			else
				get_member_group_added_message(targets)
			end

		end
		
		def get_enemy_group_added_message(targets)
		end

		def get_member_group_added_message(targets)
		end

		
		def get_some_of_enemy_group_added_message(targets, suc)
			get_enemy_group_added_message(targets)
		end

		
		def get_time_recover_message(target)
			get_recover_message(target)
		end
		
		def get_damage_recover_message(target)
			get_recover_message(target)
		end
		
		def get_guard_message(target)
			
		end
		
		def get_some_of_enemy_group_caption(targets, added)
			if targets.size == 1 then
				return targets.first.name
			elsif targets.size == added then
				msg = n_("%{all}体の%{target}全員", "%{all}体の%{target}全員", targets.size)
				return msg.evaluate(:all => targets.size, :target => targets.first.name)
			else
				msg = n_("%{all}体のうち、%{added}体の%{target}", "%{all}体のうち、%{added}体の%{target}", targets.size)
				return msg.evaluate(:all => targets.size, :added => added, :target => targets.first.name)
			end
		end

		
		
		def delete_self(target)
			target.added_status.delete(self)
		end

		
		def on_round_start(target)
			
		end
		
		
		def on_round_end(target)
			if @recover_round and @round >= @recover_round then
				delete_self(target)
				PARTY_WINDOW.update
				msg = get_time_recover_message(target)
				Game.message(msg) if msg
			end
			@round += 1
		end
		
		def on_damaged(target, damage)
			if check_damage_recover(target, damage) then
				delete_self(target)
				PARTY_WINDOW.update
				return get_damage_recover_message(target)
			end
		end
		
		def on_use_mp(target, mp_cost)
		end
		
		
		
		def on_walk(target)
		end
		
		def get_damage_resist_bonus(damage_type)
			return RES_WEAKEST
		end

		
		def dump
			hash = super
			hash['level'] = @level
			hash['round'] = @round
			return hash
		end


		
		def self.create_from_mpac_model(hash)
			obj = new(hash['level'])
			obj.round = hash['round']
			return obj
		end
		

	end
	
	
	
	class ParameterBonusState < State
		def visible?
			false
		end
		
		def name
			"#{parameter_name}#{Util.bonus_to_s(@level)}"
		end
		
		def good?
			@level >= 1
		end
		
		
	end

	class CleanHittingBonusState < ParameterBonusState
		def parameter_name
			'クリーンヒット率'
		end

		def clean_hitting_bonus
			@level
		end
	end

	class SurvivingBonusState < ParameterBonusState
		def parameter_name
			'生存率'
		end

		def surviving_bonus
			@level
		end
	end

	class FastEscapingBonusState < ParameterBonusState
		def parameter_name
			'初手逃走率'
		end

		def fast_escaping_bonus
			@level
		end
	end
	
	class EnemyOustingBonusState < ParameterBonusState
		def parameter_name
			'敵逃走率'
		end

		def enemy_ousting_bonus
			@level
		end
	end

	
	
	
	class ChargingState < State
		attr_reader :attack_targets
		RELEASE_ROUND_TABLE = [1, 2, 3]
		DAMAGE_RATE_BONUS_TABLE = [1.4, 1.8, 2.0]
		HITTING_BONUS_TABLE = [+8, +15, +25]
		def initialize(level, attack_targets)
			super(level)
			@deactive = true
			@avoiding_bonus = -999
			@attack_targets = attack_targets
		end
		
		def name
			@name = _("チャージ%{rest}") % {:rest => RELEASE_ROUND_TABLE[@level - 1] + 1 - @round}
		end
		
		def on_round_start(target)
			if at_release_round? then
				self.delete_self(target)
			end
			
			return nil
		end
		
		def at_release_round?
			# @round は1から始まることに注意
			(@round >= RELEASE_ROUND_TABLE[@level - 1] + 1)
		end
		
	end
	
	# 呪い
	# 解除するまで全能力値が徐々に減少していく
	class CurseState < State
		def initialize(level = 5)
			super
			@battle_only = false
			@lapping_type = LT_OVERRIDE
		end
		
		def name
			_('呪い') + "(#{@level})"
		end
		
		def caption
			"$[BadColor]" + name
		end
		
		def on_round_start(target)
			if rand(100) < 20 then
				@level += 1
			end
			
			return nil
		end
		
		def on_walk(target)
			if rand(100) == 0 then
				@level += 1
			end
		end
		
		def str_bonus; @level * -1; end
		def tec_bonus; @level * -1; end
		def agl_bonus; @level * -1; end
		def wil_bonus; @level * -1; end
		def psy_bonus; @level * -1; end
		
		def get_added_message(target)
			_("%{target}は呪われた！").evaluate(:target => target.name)
		end

		
		def get_recover_message(target)
			_("%{target}にかかった呪いが解けた").evaluate(:target => target.name)
		end
	end
	
	
	class PoisonState < State
		def initialize(level = 1)
			super
			@lapping_type = LT_OVERRIDE
			@battle_only = false
			@str_bonus = -10
			@wil_bonus = -10
		end
		
		def name
			case @level
			when 1
				_('毒')
			when 2
				_('強毒')
			when 3
				_('猛毒')
			end
		end
		
		def caption
			"$[badcolor]#{name}"
		end
		
		def get_added_message(target)
			s_("PoisonState|%{target}は%{state}に侵された").evaluate(:target => target.name, :state => state.name)
		end
		
		
		def get_recover_message(target)
			s_("PoisonState|%{target}の体から%{state}が消えた").evaluate(:target => target.name, :state => state.name)
		end
		
		def on_walk(target)
			if target.alive? then
				target.life_damage(1) unless Roll.check_3d33(70 - @level * 5)
			end
		end
	
		def on_round_start(target)
			if target.alive? then
				effect = StateDamageEffect.new
				effect.fixed_damage = @level * 10
				effect.messages << _("毒が体力を奪っていく……")
				return effect.operate(target, [target])
			end
		end
	end
	
	class SleepState < State
		def initialize(level = 1)
			super
			@name = _('睡眠')
			@deactive = true
			@avoid_bonus = -999
			@recover_round = 3 + rand(4)
		end

		def get_member_added_message(target)
			_("%{target}は眠ってしまった！").evaluate(:target => target.name)
		end

		def get_enemy_added_message(target)
			_("%{target}を眠らせた！").evaluate(:target => target.name)
		end
		
		
		def get_some_of_enemy_group_added_message(targets, suc)
			_("%{target}を眠らせた！").evaluate(:target => get_some_of_enemy_group_caption(targets, suc))
		end
		


		
		def check_damage_recover(target, damage)
			return true
		end

		def get_recover_message(target)
			_("%{target}は目を覚ました").evaluate(:target => target.name)
		end

	end
	
	
	class VineState < State
		def initialize(level = 1)
			super
			@level_max = 3
			@lapping_type = LT_STACK
			@recover_round = 1
		end
		
		def name
			_("ツタ")
		end
		
		def agl_bonus
			@level * -10
		end

		def get_added_message(target)
			case @level
			when 1
				s_("VineState|%{target}は足を取られて動きが鈍った").evaluate(:target => target.name)
			when 2
				s_("VineState|%{target}は足を取られて転倒した").evaluate(:target => target.name)
			when 3
				s_("VineState|%{target}は完全に動きを封じられた").evaluate(:target => target.name)
			end
		end		


	end

	
	
	# 恐怖（恐慌状態）
	# 器用さ＆集中力-20、50%の確率で動けない
	class TerrorState < State
		def initialize(level = 1)
			super
			@name = _('恐怖')
			@tec_bonus = -20
			@wil_bonus = -20
			@recover_round = 4 + rand(3)
		end
		
		
		def get_member_added_message(target)
			_("%{target}は恐怖にとらわれた！").evaluate(:target => target.name)
		end
		
		def get_enemy_added_message(target)
			_("%{target}を恐怖させた！").evaluate(:target => target.name)
		end
		
		
		def get_some_of_enemy_group_added_message(targets, suc)
			_("%{target}を恐怖させた！").evaluate(:target => get_some_of_enemy_group_caption(targets, suc))
		end

		
		def on_round_start(target)
			if rand(100) < 50 then
				target.clear_action
				_("%{target}は恐怖で立ちすくんでいる……").evaluate(:target => target.name)
			end
		end
	end
	
	class BerserkState < State
		def initialize(level = 1)
			super
			@name = _('暴走')
			@tec_bonus = -20
			@avoid_bonus = -20
			@uncommandable = true
			@recover_round = 4 + rand(3)
		end
		
		
		def get_member_added_message(target)
			_("%{target}は我を忘れて暴走しはじめた！").evaluate(:target => target.name)
		end
		
		def get_enemy_added_message(target)
			_("%{target}が暴走しはじめた！").evaluate(:target => target.name)
		end
		
		
		def get_some_of_enemy_group_added_message(targets, suc)
			_("%{target}が暴走しはじめた！").evaluate(:target => get_some_of_enemy_group_caption(targets, suc))
		end
		
		def on_round_start(target)
			LOGGER.puts('暴走中のため攻撃しかできない')
			forward = Util.random_pick(GS.party.forward_members)
			if forward then
				target.set_action(AttackAction.new(target, [forward]))
			else
				target.set_action(AttackAction.new(target, []))
			end
			return nil
		end
		
	end

	
	# 精神支配
	# 同士討ちを始める
	class MindDominatedState < State
		def initialize(level = 1)
			super
			@name = _('精神支配')
			@avoid_bonus = -999
			@recover_round = 5 + rand(3)
			@uncommandable = true
		end
		
		
		def get_enemy_added_message(target)
			_("%{target}の精神を掌握した！").evaluate(:target => target.name)
		end
		
		
		def get_some_of_enemy_group_added_message(targets, suc)
			_("%{target}の精神を掌握した！").evaluate(:target => get_some_of_enemy_group_caption(targets, suc))
		end

		
		def get_member_added_message(target)
			_("%{target}の精神が掌握された！").evaluate(:target => target.name)
		end
		
		def get_recover_message(target)
			_("%{target}は支配から解放された").evaluate(:target => target.name)
		end
		
		def on_round_start(target)
			target.set_action(AttackAction.new(target, Util.random_pick($troop.groups)))
			return nil
		end
		
		def check_damage_recover(target, damage)
			if rand(100) < 25 then
				target.clear_action
			end
		end
	end

	
	# 幻霧
	# 命中率を大幅に減少させる
	class IllusionMistState < State
		def initialize(level = 1)
			super
			@name = "幻霧"
			@avoid_bonus = -30
			@recover_round = 4 + rand(3)
		end
		
		
		def get_enemy_added_message(target)
			"#{target.name}を幻惑した！"
		end
			
		def get_member_added_message(target)
			"#{target.name}は幻影に取り込まれた！"
		end
	end


	# 火傷＆炎上
	class BurningState < State
		def initialize(level = 1)
			super
			@level_max = 2
			@lapping_type = LT_STACK
			@recover_round = 3 + rand(3)
		end
		
		def name
			case @level
			when 1
				_('発火')
			when 2
				_('炎上')
			end
		end

		def get_added_message(target)
			case @level
			when 1
				"#{target.name}の体に火がついた！"
			when 2
				"#{target.name}の体が炎に包まれた！"
			end
		end
		
		def wil_bonus
			@level * -10
		end
		
		def unconcentratable
			@level >= 2
		end
		
		def get_recover_message(target)
			"#{target.name}の火が消えた"
		end
		
		def on_round_start(target)
			if target.alive? then
				effect = StateDamageEffect.new
				effect.fixed_damage = @level * 6
				effect.messages << "火が身体を焼き焦がす！"
				return effect.operate(nil, [target])
			end
		end
		
	end


	class FreezingState < State
		def initialize(level = 1)
			super
			@level_max = 2
			@lapping_type = LT_STACK
			@recover_round = 4 + rand(4)

		end
		
		def deactive
			@level >= 2
		end
		
		def tec_bonus
			@level * -10
		end

		def agl_bonus
			@level * -10
		end

		
		def name
			case @level
			when 1
				_('凍結')
			when 2
				_('氷漬け')
			end
		end


		def get_added_message(target)
			if target.kind_of?(Enemy) then
				"#{target.name}の身体を凍り付かせた！"
			else
				"#{target.name}の身体が凍り付いてしまった！"
			end
		end
		
		def get_recover_message(target)
			"#{target.name}の氷が割れた"
		end
		
	end
	

	
	class ParalyzedState < State
		def initialize(level = 1)
			super
			@deactive = true
			@level_max = 2
			@lapping_type = LT_STACK
			@recover_round = 2 + rand(3)
		end
		
		def name
			case @level
			when 1
				_('痺れ')
			when 2
				_('麻痺')
			end
		end


		def get_added_message(target)
			if target.kind_of?(Enemy) then
				_("%{target}を痺れさせた").evaluate(:target => target.name)
			else
				_("%{target}の体が痺れた").evaluate(:target => target.name)
			end
		end
		
		def get_recover_message(target)
			_("%{target}は痺れから回復した").evaluate(:target => target.name)
		end
	end
	
	class MistWallState < State
		def initialize(level = 0)
			super
			@name = _("霧の壁")
			@avoid_bonus = +10
			@lapping_type = LT_OVERRIDE
			@recover_round = 3
		end
		
		def caption
			"$[goodcolor]#{name}(#{rest_round})"
		end
		
		def get_added_message(target)
			_("濃霧が%{target}の姿を覆い隠した").evaluate(:target => target.name)
		end

		def get_group_added_message(targets)
			_("濃霧が%{target}の姿を覆い隠した").evaluate(:target => Util.targets_caption(targets))
		end

		
	end

	
	
	class SmashedState < State
		def initialize(level = 1)
			super
			@name = _("吹き飛び")
			@standing = false
			@deactive = true
			@recover_round = 2
		end
		
		def get_added_message(target)
			if target.kind_of?(Enemy) then
				_("%{target}を衝撃で吹き飛ばした！").evaluate(:target => target.name)
			else
				_("%{target}を衝撃で吹き飛んだ！").evaluate(:target => target.name)
			end
		end
		
		
		

	end
	
	class FeintState < State
		def initialize(level = 1)
			super
			@name = _("フェイント")
			@lapping_type = LT_STACK
			@recover_round = 1
		end
		
		def get_added_message(target)
			_("%{target}の防御に隙ができた").evaluate(:target => target.name)
		end
		
		def avoid_bonus
			@level * -10
		end
		
		
		

	end
	
	class MagicBoostState < State
		def initialize(dummy = nil)
			super
			@name = _("ブースト")
		end
		
		def caption
			_("ブースト")
		end
	end
	
	
	class HeatBarrierState < State
		def initialize(level)
			super
			@name = _("熱障壁")
			@lapping_type = LT_OVERRIDE
			@protect_targets = [DT_HEAT, DT_COLD]
		end
		
		def get_added_message(target)
			_("高熱の障壁が%{target}を包み込んだ").evaluate(:target => target.name)
		end
		
		def extra_defense
			(@level * 0.5).to_i
		end
		
		def protection_damage(value)
			@level -= (value * 0.5).to_i
		end
		
		
		def get_guard_message(target)
			_("熱障壁が攻撃を受け止めた")
		end

		def caption
			"$[goodcolor]#{name} (+#{@level})"
		end
	end


	
	class BurningCageState < State
		def initialize(level)
			super
			@name = _("火の檻")
			@caption = "$[badcolor]#{name}"
			@deactive = true
			@recover_round = 3 + rand(3)
		end
		
		def get_enemy_added_message(target)
			_("%{target}を火の檻に閉じ込めた！").evaluate(:target => target.name)
		end
		
		def get_member_added_message(target)
			_("%{target}は火の檻に閉じ込められた！").evaluate(:target => target.name)
		end
		
		def on_round_start(target)
			effect = AttackEffect.new
			effect.messages << _("%{target}は火柱に包まれている！")
			effect.damage_types = [DT_HEAT]
			effect.fixed_damage = (@level / 2).to_i
			effect.through_defense = true
			
			Game.message(effect.operate(target, [target]))
			return nil
		end

	end
	
	class BlindState < State
	end


	class AbilityBonusState < State
		def initialize(level = +10)
			super
			@name = ''
			@lapping_type = LT_STACK
			@level_max = 30
		end
		
		def caption
			Util.bonus_to_s(@level)
		end
		
		def get_added_message(target)
			"#{target.name}の#{ability_name}が#{@level}上昇した"
		end
	
		def on_round_end(target)
			if @level > 0 then
				@level -= ((@level / 5) + 1)
			elsif @level < 0 then
				@level += ((@level.abs / 5) + 1)
			end
		
			if @level == 0 then
				delete_self(target)
			end
		end
	end

	class StrBonusState < AbilityBonusState
		def ability_name
			_('筋力')
		end
		
		def str_bonus
			@level
		end
	end
	
	class TecBonusState < AbilityBonusState
		def ability_name
			_('器用さ')
		end
		def tec_bonus
			@level
		end
	end

	class AglBonusState < AbilityBonusState
		def ability_name
			_('素早さ')
		end
		
		def agl_bonus
			@level
		end
	end

	class WilBonusState < AbilityBonusState
		def ability_name
			_('集中力')
		end
		
		def wil_bonus
			@level
		end
	end

	class PsyBonusState < AbilityBonusState
		def ability_name
			_('霊感')
		end
		
		def psy_bonus
			@level
		end
	end
	


	
	


	class AwakeningState < State
		def initialize
			super
			@name = "覚醒"
			@level = 4
			@lapping_type = LT_OVERRIDE
			@str_bonus = 20
			@tec_bonus = 20
			@agl_bonus = 20
			@wil_bonus = 20
			@psy_bonus = 20
		end
		
		def caption
			"$[goodcolor]覚醒 (#{@level})"
		end
		
		
		
		def on_round_end(target)
			@level -= 1
			if @level <= 0 then
				target.added_status.delete(self)
				target.added_status << AwakeningReboundState.new(5)
				return "#{target.name}の覚醒が切れた！\n強烈な反動が#{target.name}を襲う！"
			end
		end

	end
	
	class AwakeningReboundState < State
		def initialize
			super
			@name = "覚醒反動"
			@level = 3
			@lapping_type = LT_OVERRIDE
			@deactive = true
			@str_bonus = -10
			@tec_bonus = -10
			@agl_bonus = -10
			@wil_bonus = -10
			@psy_bonus = -10
		end
		
		def caption
			"$[badcolor]覚醒の反動 (#{@level})"
		end
		
		
		
		def on_round_end(target)
			@level -= 1
			if @level <= 0 then
				target.added_status.delete(self)
				return nil
			end
		end

	end
	
	class RegeneratingState < State
		def initialize(level)
			super
			@name = "自己再生"
			@lapping_type = LT_OVERRIDE
		end
		
		def caption
			"$[goodcolor]自己再生"
		end
		
		
		
		def on_round_start(target)
			cure = (target.hp_max / 20).to_i
			target.recover_hp(cure)
			return "#{target.name}のHPが#{cure}回復"
		end
	end
	
	class LifeStormState < State
		def initialize(level)
			super
			@name = _('命の渦')
			@lapping_type = LT_OVERRIDE
			@recover_round = 5
		end
		
		def caption
			"$[goodcolor]#{name}"
		end
		
		
		def on_round_start(target)
			if target.alive? then
				effect = RecoverEffect.new
				effect.hp_fixed_recovery = 10 + @level * 0.5
				effect.messages << _("命の渦が活力を与える！")
				return effect.operate(target, [target])
			end
		end
	end
	
	class BodyAccelerationState < State
		def initialize(level)
			super
			@name = _('身体加速')
			@recover_round = level
			@agl_bonus = +10
			@tec_bonus = +10
		end
		
		def caption
			"$[goodcolor]#{name}(#{rest_round})"
		end
		
		
		def get_added_message(target)
			_("%{target}の動きが素早くなった！").evaluate(:target => target.name)
		end
	end
	
	class CastingAccelerationState < State
		def initialize(level)
			super
			@name = _('詠唱加速')
			@recover_round = 5
		end
		
		def caption
			"$[goodcolor]#{@name}(#{rest_round})"
		end
		
		
		def get_added_message(target)
			_("%{target}の呪文詠唱が早くなった！").evaluate(:target => target.name)
		end
	end
	
	class SlowState < State
		def initialize(level)
			super
			@name = _('低速化')
			@recover_round = 3
			@agl_bonus = -10
			@tec_bonus = -10
		end
		
		def caption
			"$[goodcolor]#{@name}(#{rest_round})"
		end
		
		def get_added_message(target)
			_("%{target}の動きが遅くなった！").evaluate(:target => target.name)
		end
		
		def get_group_added_message(targets)
			_("%{target}の動きが遅くなった！").evaluate(:target => Util.targets_caption(targets))
		end
	end
	




	


	
	
	
	class IceSpikeState < State
		def initialize(level)
			super
			@name = _("アイススパイク")
			@lapping_type = LT_OVERRIDE
		end
		
		def attack_bonus
			(@level * 0.5).to_i
		end
		
		def get_added_message(target)
			_("%{target}の武器が冷気を纏った").evaluate(:target => target.name)
		end

		def caption
			"$[goodcolor]#{name} (+#{attack_bonus})"
		end
	end
	
	class InvisibleState < State
		def initialize(level = 1)
			super
			@name = "不可視"
			@lapping_type = LT_OVERRIDE
			@standing = false
			@hitting_bonus = +10
		end
		
		def clean_hitting_bonus
			+10
		end

		
		def caption
			"$[GoodColor]不可視"
		end
		
		def get_added_message(target)
			if target.kind_of?(Enemy) then
				"#{target.name}の姿が見えなくなった"
			else
				"#{target.name}は姿を消した"
			end
		end
	end
	
	class HidingState < State
		def initialize(level)
			super
			@name = _("潜伏")
			@standing = false
			@hitting_bonus = +10
		end
		
		def caption
			"$[GoodColor]#{name}"
		end
		
		def clean_hitting_bonus
			+10
		end
	end

	
	
	class PrevisionState < State
		def initialize(level)
			super
			@name = "予知"
			@recover_round = 5
		end
		
		def caption
			"$[GoodColor]予知"
		end
		
		def get_added_message(target)
			"#{target.name}は相手の動きを予知できるようになった"
		end
	end
	
	class RingOfDarkState < State
		def initialize(level = 1)
			super
			@name = _("暗黒の指輪")
		end
		
		def on_use_mp(target, mp_cost)
			dmg = (target.hp_max * (0.03 + mp_cost * 0.01)).to_i + 1
			target.life_damage(dmg)
			LOGGER.puts "\t暗黒の指輪:ライフダメージ#{dmg}点"
		end

	end


	class ImperiumState < State
		def initialize(level)
			super
			@name = "攻撃命令"
			@recover_round = 1
			@speed_bonus = +999
			@hitting_bonus = level
		end
		
		def good?
			true
		end
	end
	

	
	class ElementalProtectionState < State
		def initialize(level)
			super
			@name = _('保護')
			@recover_round = 1
		end
		
		def visible?
			true
		end
		
		def good?
			true
		end
	end
	
	

	MAIN_STATE_CLASSES = [
		SmashedState, SleepState, ParalyzedState, TerrorState,
		BerserkState, MindDominatedState, CurseState, PoisonState,
	]


end
