/*
 * Copyright(C) 2010 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.lunchorder;

import java.util.Calendar;
import java.util.jar.Attributes;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.eip.cayenne.om.portlet.AvzTLunchOrder;
import com.aimluck.eip.cayenne.om.portlet.AvzTLunchOrderItem;
import com.aimluck.eip.cayenne.om.portlet.AvzTLunchUserOrder;
import com.aimluck.eip.common.ALAbstractSelectData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALData;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.ResultList;
import com.aimluck.eip.orm.query.SelectQuery;

/**
 * 注文表検索データ(初期画面)を管理するクラスです。 <BR>
 * 
 */
public class LunchOrderSelectData extends
    ALAbstractSelectData<AvzTLunchOrder, Object> implements ALData {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(LunchOrderSelectData.class.getName());

  /** 現在の日付と時刻をカレンダークラスから取得 */
  private final Calendar cal = Calendar.getInstance();

  /** 時だけ */
  private final String nowhour = String.valueOf(cal.get(Calendar.HOUR_OF_DAY));

  /** 分だけ */
  private String nowminute = String.valueOf(cal.get(Calendar.MINUTE));

  /** 注文表の総数 */
  private int LunchOrderMapSum;

  /**
   * 
   * @param action
   * @param rundata
   * @param context
   * @see com.aimluck.eip.common.ALAbstractSelectData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)

  throws ALPageNotFoundException, ALDBErrorException {

    super.init(action, rundata, context);

  }

  /**
   * 一覧データを取得します。 <BR>
   * 
   * @param rundata
   * @param context
   * @return
   * @see com.aimluck.eip.common.ALAbstractListData#selectData(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  public ResultList<AvzTLunchOrder> selectList(RunData rundata, Context context) {// 検索条件を設定した
    // SelectQuery
    // を元に検索するメソッド
    try {

      SelectQuery<AvzTLunchOrder> query = getSelectQuery(rundata, context);
      buildSelectQueryForListView(query);
      query.orderAscending("storeName");
      ResultList<AvzTLunchOrder> list = query.getResultList();
      LunchOrderMapSum = list.size(); // 注文表の総数 をセットする．
      return list;

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }

  }

  /**
   * 検索条件を設定した SelectQuery を返します。 <BR>
   * 
   * @param rundata
   * @param context
   * @return
   */
  private SelectQuery<AvzTLunchOrder> getSelectQuery(RunData rundata,
      Context context) {
    SelectQuery<AvzTLunchOrder> query = Database.query(AvzTLunchOrder.class);
    return query;
  }

  /**
   * ResultData に値を格納して返します。（一覧データ） <BR>
   * 
   * @param obj
   * @return
   * @see com.aimluck.eip.common.ALAbstractSelectData#getListData(java.lang.Object)
   */
  @Override
  protected Object getResultData(AvzTLunchOrder obj) {// 結果を保存
    try {
      AvzTLunchOrder record = obj;

      LunchOrderResultData rd = new LunchOrderResultData();// インスタンス生成

      rd.initField(); // オブジェクトのフィールドを初期化

      if ((record.getMinStatus().equals("P"))) {// 最低注文”価格”の設定がある場合

        int mapId = record.getOrderMapId();// 弁当屋のID
        int min_price = record.getMinPlice();// 最低注文価格
        int order_price = 0;// 実際注文されている合計金額用の変数

        Expression exp01 =
          ExpressionFactory.matchDbExp(// 弁当屋ごとの条件
            AvzTLunchUserOrder.TO_AVZ_TLUNCH_ORDER_ITEM_PROPERTY
              + "."
              + AvzTLunchOrderItem.TO_AVZ_TLUNCH_ORDER_PROPERTY
              + "."
              + AvzTLunchOrder.ORDER_MAP_ID_PK_COLUMN,
            mapId);

        SelectQuery<AvzTLunchUserOrder> query01 =
          Database.query(AvzTLunchUserOrder.class, exp01);

        ResultList<AvzTLunchUserOrder> orderlist = query01.getResultList();

        // ↓弁当屋ごとに登録されているアイテム（弁当）をとりだして、アイテムの注文状況を検索
        for (int i = 0; i < orderlist.size(); i++) {

          AvzTLunchUserOrder avzTLunchUserOrder = orderlist.get(i);

          int itemPrice =
            avzTLunchUserOrder.getToAvzTLunchOrderItem().getPrice();

          order_price =
            order_price + (avzTLunchUserOrder.getOrderAmount() * itemPrice);
        }

        if (min_price > order_price) { // 最低注文金額未達の場合

          rd.setStatus("price_NotAchievement");// 備考表示内容フラグ設定（velocityで表示振り分けに使用）

        } else { // 最低注文金額達成の場合

          rd.setStatus("price_Achievement");// 備考表示内容フラグ設定（velocityで表示振り分けに使用）
        }

      } else if (record.getMinStatus().equals("N")) {// 最低注文”数”の設定がある場合

        int mapId = record.getOrderMapId();// 弁当屋のID
        int min_num = record.getMinNum();// 最低注文数
        int order_amount = 0;// 実際注文されている合計数用の変数

        Expression exp01 =
          ExpressionFactory.matchDbExp(// 弁当屋ごとの条件
            AvzTLunchUserOrder.TO_AVZ_TLUNCH_ORDER_ITEM_PROPERTY
              + "."
              + AvzTLunchOrderItem.TO_AVZ_TLUNCH_ORDER_PROPERTY
              + "."
              + AvzTLunchOrder.ORDER_MAP_ID_PK_COLUMN,
            mapId);

        Expression exp02 =
          ExpressionFactory.matchExp(// 注文数にカウントするアイテムだけの条件（※"1"はカウントする"2"はカウントしない）
            AvzTLunchUserOrder.TO_AVZ_TLUNCH_ORDER_ITEM_PROPERTY
              + "."
              + AvzTLunchOrderItem.MIN_ORDER_FLAG_PROPERTY,
            "1");

        SelectQuery<AvzTLunchUserOrder> query =
          Database.query(AvzTLunchUserOrder.class, exp01.andExp(exp02));

        ResultList<AvzTLunchUserOrder> orderlist = query.getResultList();

        for (int i = 0; i < orderlist.size(); i++) {// 注文数だけを取り出していく

          AvzTLunchUserOrder avzTLunchUserOrder = orderlist.get(i);

          order_amount = avzTLunchUserOrder.getOrderAmount() + order_amount;

        }

        if (min_num > order_amount) {// 最低注文数未達の場合

          rd.setStatus("amount_NotAchievement");// 備考表示内容フラグ設定（velocityで表示振り分けに使用）

        } else {// 最低注文数達成の場合

          rd.setStatus("amount_Achievement");// 備考表示内容フラグ設定（velocityで表示振り分けに使用）

        }

      } else {// 特に条件なしの場合

        rd.setStatus("condition_none");// 備考表示内容フラグ設定（velocityで表示振り分けに使用）

      }

      int nowtime = 0;// 現在時刻用の変数

      if (nowminute.length() == 1) {// 分が一桁の場合は0を付け加えて2桁にする

        nowminute = ("0" + nowminute);
      }

      nowtime = Integer.parseInt(nowhour + nowminute);

      if (record.getDeadlineDate() != null) {// 締め切り時刻がある場合

        if (record.getDeadlineDate() < nowtime) {// 締め切り時刻を過ぎてる場合

          rd.setDeadLineFlag("F");// 注文ボタン表示内容フラグ設定（velocityで表示振り分けに使用）

        } else {// 締め切り時刻を過ぎていない場合

          rd.setDeadLineFlag("T");// 注文ボタン表示内容フラグ設定（velocityで表示振り分けに使用）

        }

      } else {// 締め切り時刻がない場合

        rd.setDeadLineFlag("T");// 注文ボタン表示内容フラグ設定（velocityで表示振り分けに使用）

      }

      rd.setStoreName(record.getStoreName());// 弁当屋名をセット

      rd.setOrderMapId(record.getOrderMapId());// 弁当屋IDセット

      return rd;

    } catch (Exception ex) {
      logger.error("Exception", ex);
      return null;
    }
  }

  /**
   * 詳細データを取得します。 <BR>
   * 
   * @param rundata
   * @param context
   * @return
   * @see com.aimluck.eip.common.ALAbstractSelectData#selectDetail(org.apache.turbine.util.RunData,
   *      org.apache.velocity.context.Context)
   */
  @Override
  public Object selectDetail(RunData rundata, Context context)
      throws ALPageNotFoundException {
    return null;
  }

  /**
   * ResultData に値を格納して返します。（詳細データ） <BR>
   * 
   * @param obj
   * @return
   * @see com.aimluck.eip.common.ALAbstractSelectData#getResultDataDetail(java.lang.Object)
   */
  @Override
  protected Object getResultDataDetail(Object obj) {
    return null;
  }

  /**
   * 注文表の総数を返す． <BR>
   * 
   * @return
   */
  public int getLunchOrderMapSum() {
    return LunchOrderMapSum;
  }

  /**
   * @return
   * @see com.aimluck.eip.common.ALAbstractSelectData#getColumnMap()
   */
  @Override
  protected Attributes getColumnMap() {
    Attributes map = new Attributes();
    return map;
  }
}
