/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.portlet;

import java.util.Enumeration;
import java.util.Locale;

/**
 ** The <CODE>PortletConfig</CODE> interface provides the portlet with
 ** its configuration. The configuration holds information about
 ** the portlet that is valid for all users, and is
 ** maintained by the administrator. The portlet can therefore only
 ** read the configuration. Only when the portlet is in CONFIGURE mode,
 ** it has write access to the configuartion data (the rest of the configuration
 ** is managed by the portlet container directly).
 **
 ** @see   Portlet
 **
 ** @author   <A HREF="mailto:tboehme@us.ibm.com">Thomas F. Boehme</A>
 **/

public interface PortletConfig
{
    /**
     ** Returns the name of the portlet. The portlet container needs the
     ** portlet name for internal and administration purposes.
     **
     ** @return   the portlet name
     **/

    public String getName ();

    /**
     ** Sets the attribute with the given name and value.
     **
     ** @param   name
     **          the attribute name
     ** @param   value
     **          the attribute value
     **
     ** @exception   AccessDeniedException
     **              if the caller isn't authorized to access
     **              this data object
     **/

    public void setAttribute (String name, String value) throws AccessDeniedException;

    /**
     ** Returns the value of the attribute with the given name,
     ** or <CODE>null</CODE> if no such attribute exists.
     **
     ** @param   name
     **          the attribute name
     **
     ** @return   the attribute value
     **
     ** @exception   AccessDeniedException
     **              if the caller isn't authorized to access
     **              this data object
     **/

    public String getAttribute (String name) throws AccessDeniedException;

    /**
     ** Returns an enumeration of all available attributes names.
     **
     ** @return   an enumeration of attribute names
     **
     ** @exception   AccessDeniedException
     **              if the caller isn't authorized to access
     **              this data object
     **
     ** @see   #getAttribute(String)
     **/

    public Enumeration getAttributeNames () throws AccessDeniedException;

    /**
     ** Removes the attribute with the given name. If no such
     ** attribute exists, this method does nothing.
     **
     ** @param   name
     **          the attribute name
     **
     ** @exception   AccessDeniedException
     **              if the caller isn't authorized to access
     **              this data object
     **/

    public void removeAttribute (String name) throws AccessDeniedException;

    /**
     ** Returns the title of this window. This method returns
     ** only a static title from the portlet configuration.
     **
     ** @param   locale
     **          the locale
     ** @param   client
     **          the client
     **
     ** @return   the text of the portlet title
     **/

    public String getTitle (Locale locale,
                            Client client);

    /**
     * Returns whether the portlet window supports the given state
     * 
     * @param state  the portlet window state to be supported
     * @return <CODE>true</CODE> if the portlet window provides support, <BR>
     *         <CODE>false</CODE> otherwise
     * @see PortletWindow.State
     */
    public boolean supports (PortletWindow.State state);

    /**
     ** Returns whether the portlet supports the given locale.
     **
     ** @param   locale
     **          the locale to be supported
     **
     ** @return  <CODE>true</CODE> if the portlet provides support, <BR>
     **          <CODE>false</CODE> otherwise
     **/

    public boolean supports (Locale locale);

    /**
     * Returns whether the portlet supports the given mode for
     * the given client.
     * 
     * @param mode
     *               the portlet mode to be supported
     * @param client
     *               the client to be supported
     * @return <CODE>true</CODE> if the portlet provides support, <BR>
     *         <CODE>false</CODE> otherwise
     * @see Portlet.Mode
     */
    public boolean supports (Portlet.Mode mode, Client client);

    /**
     ** Returns the portlet context.
     **
     ** @return   the portlet context
     **/

    public PortletContext getContext ();
}
