/* $Id$ */
package saccubus.converter.filegetter;

import java.io.IOException;
import java.text.Normalizer;
import org.apache.commons.lang.StringUtils;
import saccubus.ConvertStopFlag;
import saccubus.converter.profile.Proxy;
import saccubus.net.CommentInfo;
import saccubus.net.NicoClient;
import saccubus.net.VideoInfo;
import saccubus.util.WayBackTimeParser;
import yukihane.inqubus.saccubus_adapter.NicoBrowserAdapter;

/**
 * 動画ファイル, コメントファイルなど必要なファイルのうち, 1つでもダウンロード処理を必要とする場合のインスタンス化クラス.
 * @author yuki
 */
public class WebFileInstanciator extends FileInstanciator {

    private final NicoClient client;
    private final VideoInfo videoInfo;
    private final CommentInfo commentInfo;

    WebFileInstanciator(
            ConvertStopFlag stopFlag,
            InstanciationType videoType,
            CommentInstanciationType commentType,
            InstanciationType tcommType,
            LoginInfo li,
            String tag,
            String time) throws IOException {
        super(videoType, commentType, tcommType, tag);

        if (li.getMail() == null || li.getPass() == null || li.getMail().equals("") || li.getPass().equals("")) {
            throw new IllegalArgumentException("メールアドレスかパスワードが空白です。");
        }

        String host;
        int port;
        if (li.getProxy() != Proxy.NO_PROXY) {
            host = li.getProxy().getHost();
            port = li.getProxy().getPort();
        } else {
            host = null;
            port = -1;
        }
        client = new NicoBrowserAdapter(li.getMail(), li.getPass(), stopFlag, host, port) {
        };

        if (!client.isLoggedIn()) {
            throw new IOException("ログインに失敗");
        }

        try {
            videoInfo = client.getVideoInfo(tag);
            if (StringUtils.isNotBlank(time)) {
                System.out.print("Setting wayback time...");
                final long waybacktime = WayBackTimeParser.parse(time);
                String waybackkey = client.getWayBackKey(videoInfo);
                commentInfo = new CommentInfo(waybackkey, waybacktime);
            } else {
                commentInfo = null;
            }

        } catch (IOException ex) {
            final String text = tag + "の情報の取得に失敗";
            final String cause = (ex.getMessage() != null) ? "(" + ex.getMessage() + ")" : "";

            throw new IOException(text + cause, ex);
        }

        if (videoType.isDownload()) {
            setVideoFileGetter(new VideoFileWebGetter(client, videoInfo));
        }

        if (commentType.isDownload()) {
            setCommentFileGetter(new CommentFileWebGetter(client, videoInfo, commentInfo, commentType.isAutoCommentNum(),
                    commentType.getBackComment(), commentType.isReduceComment()));
        }

        if (tcommType.isDownload()) {
            setTcommFileGetter(new TcommFileWebGetter(client, videoInfo));
        }
    }

    /**
     * 動画のタイトルを取得する。
     * 実際の動画タイトルはファイル名に用いることが出来ない文字を含んである場合があるため、ファイル名に適合するように編集した値が返る。
     * @return
     */
    @Override
    public String getVideoTitle() {
        String name = videoInfo.getVideoTitle();
        name = Normalizer.normalize(name, Normalizer.Form.NFKC);
        name = name.replaceAll("[\\\\/:*?\"<>|.]", "_");
        name = name.replace('\u2212', '\uff0d'); // － U+2212(MINUS SIGN) -> U+FF0D(FULLWIDTH HYPHEN-MINUS)
        name = name.replace('\u301c', '\uff5e'); // ～ U+301C(WAVE DASH) -> U+FF5E(FULLWIDTH TILDE)
        name = name.replace('\u223c', '\uff5e'); // ～ U+223C(TILDE OPERATOR) -> U+FF5E(FULLWIDTH TILDE)
        name = name.replace('\u00a2', '\uffe0'); // ￠ U+00A2(CENT SIGN) -> U+FFE0(FULLWIDTH CENT SIGN)
        name = name.replace('\u00a3', '\uffe1'); // ￡ U+00A3(POUND SIGN) -> U+FFE1(FULLWIDTH POUND SIGN)
        name = name.replace('\u00ac', '\uffe2'); // ￢ U+00AC(NOT SIGN) -> U+FFE2(FULLWIDHT NOT SIGN)
        return name;
    }
}
