/*$Id$*/
package nicobrowser.main;

import java.io.File;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.EntityTransaction;
import javax.persistence.Persistence;
import javax.persistence.Query;
import javax.swing.JFrame;
import javax.swing.SwingUtilities;
import nicobrowser.config.Config;
import nicobrowser.GetFlvResult;
import nicobrowser.NamePattern;
import nicobrowser.NicoHttpClient;
import nicobrowser.ProgressListener;
import nicobrowser.VideoInfo;
import nicobrowser.config.NicoFeed;
import nicobrowser.entity.NicoContent;
import nicobrowser.entity.NicoContent.Status;
import nicobrowser.gui.config.ConfigFrame;
import nicobrowser.update.DBUpdater;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class Main {

    public static final String VERSION = "0.9.5";
    private static Log log = LogFactory.getLog(Main.class);

    public static void main(String[] args) throws Exception {
        final boolean res = Config.createNewConfigFiles();

        CommandLineParser parser = new PosixParser();
        Options options = createOptions();

        try {
            // parse the command line arguments
            CommandLine line = parser.parse(options, args);

            if (line.hasOption("help")) {
                new HelpFormatter().printHelp("java -jar NicoBrowser.jar [option]", options);
                return;
            } else if (line.hasOption("version")) {
                System.out.println(VERSION);
                return;
            } else if (line.hasOption("property")) {
                propertyOpen();
                return;
            }
        } catch (ParseException exp) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp("java -jar NicoBrowser.jar [option]", options);
            return;
        }

        if (res) {
            propertyOpen();
            return;
        }

        DBUpdater updater = new DBUpdater();
        if (args.length > 0 && "sync".equals(args[0])) {
            updater.sync();
            return;
        } else if (args.length > 0 && "sync4".equals(args[0])) {
            updater.sync_for_4();
            return;
        }
        updater.update();

        new Main().start();
    }

    private static Options createOptions() throws IllegalArgumentException {
        Options options = new Options();
        options.addOption("h", "help", false, "このメッセージを表示します");
        options.addOption("p", "property", false, "コンフィグ設定画面を起動します");
        options.addOption("v", "version", false, "バージョン情報を表示します");
        return options;
    }

    private static void propertyOpen() {
        SwingUtilities.invokeLater(new Runnable() {

            public void run() {
                ConfigFrame.setLookAndFeel();
                JFrame frame = new ConfigFrame();
                frame.setLocationByPlatform(true);
                frame.setVisible(true);
            }
        });
    }

    public void start() {
        log.info("program start");

        final Config config = Config.getInstance();
        List<NicoContent> dailyList = new ArrayList<NicoContent>();
        ArrayList<List<NicoContent>> myLists = new ArrayList<List<NicoContent>>();
        NicoHttpClient instance = null;
        try {
            log.info("リストを取得します");
            instance = new NicoHttpClient();
            List<NicoFeed> feeds = config.getNicoFeeds();
            for (NicoFeed f : feeds) {
                List<NicoContent> list = instance.getContentsFromRss(f.getUrl());
                int count = 0;
                for (NicoContent l : list) {
                    if (count >= f.getNumber()) {
                        break;
                    }
                    dailyList.add(l);
                    count++;
                }
            }
            List<String> mylists = config.getDownLoadMyList();
            for (String l : mylists) {
                List<NicoContent> list = instance.loadMyList(l);
                myLists.add(list);
            }
        } catch (Exception e) {
            e.printStackTrace();
            return;
        }

        log.info("今回取得したデータを過去の取得データと比較します");

        EntityManagerFactory factory;
        EntityManager manager;

        HashMap<String, String> map = new HashMap<String, String>();
        map.put("toplink.jdbc.url", "jdbc:h2:" + config.getDbFile());
        factory = Persistence.createEntityManagerFactory("NicoBrowserPU", map);
        manager = factory.createEntityManager();

        EntityTransaction transaction = manager.getTransaction();

        transaction.begin();
        try {
            // ランキング上位のコンテンツ保存
            int num = 0;
            for (NicoContent c : dailyList) {
                save(manager, c);
                num++;
            }

            // マイリストに登録したコンテンツ保存
            for (List<NicoContent> l : myLists) {
                for (NicoContent c : l) {
                    save(manager, c);
                }
            }

            transaction.commit();

            Query query = manager.createQuery("SELECT cont FROM NicoContent AS cont " + "WHERE ?1 <> cont.status").
                    setParameter(1, NicoContent.Status.GET_FILE);
            @SuppressWarnings("unchecked")
            List<NicoContent> results = query.getResultList();
            instance.login(config.getNicoMail(), config.getNicoPassword());
            Date prevDate = null;
            for (NicoContent c : results) {
                if (c.getFailTimes() >= config.getMaxRetry()) {
                    continue;
                }
                if (prevDate != null) {
                    Date nowDate = Calendar.getInstance().getTime();
                    long sleep = nowDate.getTime() - prevDate.getTime();
                    sleep = config.getWaitTime() * 1000 - sleep;
                    if (sleep > 0) {
                        log.info("" + sleep / 1000 + "秒待機します。");
                        try {
                            Thread.sleep(sleep);
                        } catch (InterruptedException e) {
                            log.info("スリープ中断", e);
                        }
                    }
                }
                prevDate = Calendar.getInstance().getTime();
                File saveDir = new File(config.getSrcSaveDir());
                NamePattern np = new NamePattern(config.getFileNamePattern(), config.getFileNameReplaceFrom(), config.
                        getFileNameReplaceTo(), c.getTitle());
                log.info("ファイルを取得します: " + c.getNicoId() + " " + c.getTitle());
                try {
                    VideoInfo vi = instance.getVideoInfo(c.getNicoId());
                    GetFlvResult result = instance.getFlvFile(vi, saveDir, np, c.getStatus(), config.needLowFile(),
                            ProgressListener.EMPTY_LISTENER);

                    c.setFileName(FilenameUtils.getBaseName(result.getFile().toString()));
                    c.setAuthor(result.getAuthor());
                    Status status = result.getStatus();
                    c.setStatus(status);
                    if (status == Status.GET_INFO) {
                        c.setFailTimes(c.getFailTimes() + 1);
                    }
                } catch (Exception ex) {
                    c.setFailTimes(c.getFailTimes() + 1);
                    log.error("ファイル取得に失敗しました: " + c.getNicoId() + ", 通算失敗回数: " + c.getFailTimes(), ex);
                }
                transaction.begin();
                manager.persist(c);
                transaction.commit();
                log.info("完了しました");
            }
        } catch (Exception ex) {
            ex.printStackTrace();
            transaction.rollback();
        } finally {
            manager.close();
            factory.close();
        }

    }

    private void save(EntityManager manager, NicoContent c) {
        Query query = manager.createQuery("SELECT cont FROM NicoContent AS cont " + "WHERE ?1 = cont.nicoId").
                setParameter(1, c.getNicoId());
        @SuppressWarnings("unchecked")
        List<NicoContent> resList = query.getResultList();
        if (resList.isEmpty()) {
            log.info("NEW! " + c.getNicoId() + " : " + c.getTitle());
            manager.persist(c);
        }
    }
}
