/* $Id$ */
package saccubus.converter.filegetter;

import java.io.File;
import java.io.FilenameFilter;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 * @author yuki
 */
class FileLocator {

    private final boolean autoFileName;
    private final File initFile;
    private final String prefix;
    private final String title;
    private final String[] suffixes;

    FileLocator(boolean autoFileName, File initFile, String prefix, String title, String... suffixes) {
        this.autoFileName = autoFileName;
        this.initFile = initFile;
        this.prefix = prefix;
        this.title = title;
        this.suffixes = suffixes;
    }

    /**
     * コンストラクタで指定した条件でファイルを検索する。
     * autoFileNameがfalseの場合、直接名前指定であるため指定されたファイルをそのまま返す。
     * autoFileNameがtrueの場合、initFileディレクトリにprefixで始まりsuffixで終わるファイルが存在するか検索し
     * あればそれを返す。
     * このファイルも無い場合、指定されたパラメータに従ったファイル名を返す。
     * このとき拡張子(suffixes)が複数指定されていた場合は最初のものを返す。
     * @return 所望のファイル。ファイルは存在する場合もあり、存在しない場合もあることに注意。
     */
    File getFile() {
        if (!isAutoNaming()) {
            return initFile;
        } else {
            File res = searchFile();
            if (res != null) {
                return res;
            }
            return new File(initFile, prefix + title + suffixes[0]);
        }
    }

    private File searchFile() {
        FilenameFilter filter = new FilenameFilter() {

            @Override
            public boolean accept(File dir, String name) {
                if (name.startsWith(prefix)) {
                    for (String suffix : suffixes) {
                        if (name.endsWith(suffix)) {
                            return true;
                        }
                    }
                }
                return false;
            }
        };
        File[] res = initFile.listFiles(filter);
        if (res == null || res.length == 0) {
            return null;
        }
        if (res.length > 1) {
            final String message = "複数の候補ファイルがヒットしました. 意図しないファイルが選択されている可能性があります:"
                    + res[0];
            Logger.getLogger(FileLocator.class.getName()).log(Level.WARNING, message);
        }
        return res[0];
    }

    protected boolean isAutoNaming() {
        return autoFileName;
    }

    protected final boolean getAutoFileName() {
        return autoFileName;
    }
}
