/* $Id$ */
package saccubus.converter;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.lang.StringUtils;
import saccubus.ConvertStopFlag;
import saccubus.conv.ConvertToVideoHook;
import saccubus.converter.profile.Ffmpeg;
import saccubus.converter.profile.FfmpegOption;
import saccubus.converter.profile.GeneralSetting;
import saccubus.converter.profile.NgSetting;
import saccubus.net.TextProgressListener;
import yukihane.mediainfowrapper.Info;
import yukihane.mediainfowrapper.MediaInfo;
import yukihane.mediainfowrapper.Size;
import yukihane.swf.Cws2Fws;

/**
 *
 * @author yuki
 */
public class FfmpegCommand extends AbstractCommand {
    private static final Logger logger = Logger.getLogger(FfmpegCommand.class.getName());
    private final File commentMiddleFile;
    private final File tcommMiddleFile;
    private final File TMP_CWS;
    private final File commentFile;
    private final File tcommFile;
    private final File videoFile;
    private final File convertedVideoFile;
    private final Ffmpeg ffmpeg;

    FfmpegCommand(TextProgressListener listener, ConvertStopFlag flag, File commentFile, File tcommFile,
            File videoFile, File convertedVideoFile, Ffmpeg ffmpeg, GeneralSetting general) throws IOException {
        super(listener, flag);
        this.commentFile = commentFile;
        this.tcommFile = tcommFile;
        this.videoFile = videoFile;
        this.convertedVideoFile = convertedVideoFile;
        this.ffmpeg = ffmpeg;

        File tmpDir = general.getTempDir();
        commentMiddleFile = File.createTempFile("vhk", ".tmp", tmpDir);
        tcommMiddleFile = File.createTempFile("tcom", ".tmp", tmpDir);
        TMP_CWS = File.createTempFile("cws", ".swf", tmpDir);
    }

    public boolean execute() throws InterruptedException, IOException {
        try {
            return exec();
        } finally {
            if (commentMiddleFile.exists()) {
                commentMiddleFile.delete();
            }
            if (tcommMiddleFile.exists()) {
                tcommMiddleFile.delete();
            }
            if (TMP_CWS.exists()) {
                TMP_CWS.delete();
            }
        }
    }

    private boolean exec() throws InterruptedException, IOException {
        final NgSetting ngSetting = getFfmpeg().getNgSetting();
        if (commentFile != null) {
            sendText("コメントの中間ファイルへの変換中");
            boolean conv = ConvertToVideoHook.convert(commentFile, commentMiddleFile, ngSetting.getId(), ngSetting.
                    getWord());
            if (!conv) {
                sendText("コメント変換に失敗。ファイル名に使用できない文字が含まれているか正規表現の間違い？");
                return false;
            }
        }
        stopFlagReturn();
        if (tcommFile != null) {
            sendText("投稿者コメントの中間ファイルへの変換中");
            boolean conv = ConvertToVideoHook.convert(tcommFile, tcommMiddleFile, ngSetting.getId(), ngSetting.getWord());
            if (!conv) {
                sendText("コメント変換に失敗。ファイル名に使用できない文字が含まれているか正規表現の間違い？");
                return false;
            }
        }
        stopFlagReturn();
        sendText("動画の変換を開始");
        int code;
        if ((code = converting_video(videoFile, convertedVideoFile, (commentFile != null), commentMiddleFile.getPath(),
                (tcommFile
                != null), tcommMiddleFile.getPath(), getFfmpeg().getFfmpegOption())) == 0) {
            sendText("変換が正常に終了しました。");
            return true;
        } else {
            sendText("変換エラー:" + convertedVideoFile.getPath());
        }
        return false;
    }

    private int converting_video(File videoFile, File convertedVideoFile, boolean addComment, String commPath,
            boolean addTcomment, String tcommPath, FfmpegOption ov) throws InterruptedException, IOException {
        File fwsFile = Cws2Fws.createFws(videoFile, TMP_CWS);

        List<String> cmdList = new ArrayList<String>();
        cmdList.add(getFfmpeg().getFfmpeg().getPath());
        cmdList.add("-y");
        String[] mainOptions = ov.getMainOption().split(" +");
        for (String opt : mainOptions) {
            if (StringUtils.isNotBlank(opt)) {
                cmdList.add(opt);
            }
        }
        String[] inOptions = ov.getInOption().split(" +");
        for (String opt : inOptions) {
            if (StringUtils.isNotBlank(opt)) {
                cmdList.add(opt);
            }
        }
        cmdList.add("-i");

        if (fwsFile == null) {
            cmdList.add(videoFile.getPath());
        } else {
            cmdList.add(fwsFile.getPath());
        }
        String[] outOptions = ov.getOutOption().split(" +");
        for (String opt : outOptions) {
            if (StringUtils.isNotBlank(opt)) {
                cmdList.add(opt);
            }
        }

        final Info info = MediaInfo.getInfo(new File("bin", "MediaInfo"), videoFile);
        // 4:3 なら1.33, 16:9 なら1.76
        boolean isHD = ((double)info.getWidth()/(double)info.getHeight() > 1.5);
        if (ov.isResize()) {
            final Size scaled = (ov.isAdjustRatio()) ? MediaInfo.adjustSize(info, ov.getResizeWidth(), ov.
                    getResizeHeight()) : new Size(info.getWidth(), info.getHeight());
            cmdList.add("-s");
            cmdList.add(scaled.getWidth() + "x" + scaled.getHeight());
        }

        List<String> avfilterArgs = getAvfilterOptions(ov.getAvfilterOption());
        if (!getFfmpeg().isVhookDisabled()) {
            String vhookArg = getVhookArg(addComment, commPath, addTcomment, tcommPath, isHD);
            if(StringUtils.isNotBlank(vhookArg)){
                avfilterArgs.add(vhookArg);
            }
        }

        if (!avfilterArgs.isEmpty()) {
            cmdList.add("-vfilters");
            final String args = "\"" + StringUtils.join(avfilterArgs, ", ") + "\"";
            cmdList.add(args);
        }

        cmdList.add(convertedVideoFile.getPath());

        final StringBuilder argMsg = new StringBuilder();
        argMsg.append("arg:");
        for (String s : cmdList) {
            argMsg.append(" ").append(s);
        }
        logger.log(Level.INFO, argMsg.toString());

        try {
            logger.log(Level.INFO, "\n\n----\nProcessing FFmpeg...\n----\n\n");
            Process process = Runtime.getRuntime().exec(cmdList.toArray(new String[0]));
            BufferedReader ebr = new BufferedReader(new InputStreamReader(
                    process.getErrorStream()));
            String e;
            while ((e = ebr.readLine()) != null) {
                String state = e;
                if (state.startsWith("frame=")) {
                    sendText(state);
                } else if (!state.endsWith("No accelerated colorspace conversion found")) {
                    logger.log(Level.INFO, e);
                }

                try {
                    stopFlagReturn();
                } catch (InterruptedException ex) {
                    process.destroy();
                    throw ex;
                }

            }
            process.waitFor();
            return process.exitValue();
        } finally {
            if (fwsFile != null) {
                fwsFile.delete();
            }
        }
    }

    private List<String> getAvfilterOptions(String avfilterOption) {
        final List<String> avfilterArgs = new ArrayList<String>();
        if (StringUtils.isNotBlank(avfilterOption)) {
            avfilterArgs.add(avfilterOption);
        }
        return avfilterArgs;
    }

    private String getVhookArg(boolean addComment, String commPath, boolean addTcomment,
            String tcommPath, boolean isHD) throws UnsupportedEncodingException {
        StringBuilder sb = new StringBuilder();
        sb.append("vhext=");
        sb.append(getFfmpeg().getVhook().getPath().replace("\\", "/"));
        if (addComment) {
            sb.append("|");
            sb.append("--data-user:");
            sb.append(URLEncoder.encode(commPath.replace("\\", "/"), "Shift_JIS"));
        }
        if (addTcomment) {
            sb.append("|");
            sb.append("--data-owner:");
            sb.append(URLEncoder.encode(tcommPath.replace("\\", "/"), "Shift_JIS"));
        }
        sb.append("|");
        sb.append("--font:");
        sb.append(URLEncoder.encode(
                getFfmpeg().getFont().getPath().replace("\\", "/"), "Shift_JIS"));
        sb.append("|");
        sb.append("--font-index:");
        sb.append(getFfmpeg().getFontIndex());
        sb.append("|");
        sb.append("--show-user:");
        sb.append(getFfmpeg().getMaxNumOfComment());
        sb.append("|");
        sb.append("--shadow:");
        sb.append(getFfmpeg().getShadowIndex());
        sb.append("|");
        if (getFfmpeg().isShowConverting()) {
            sb.append("--enable-show-video");
            sb.append("|");
        }
        if (getFfmpeg().isSelfAdjustFontSize()) {
            sb.append("--enable-fix-font-size");
            sb.append("|");
        }
        if (getFfmpeg().isCommentOpaque()) {
            sb.append("--enable-opaque-comment");
            sb.append("|");
        }
        if(isHD){
            sb.append("--aspect-mode:1");
            sb.append("|");
        }
        return sb.toString();
    }

    private Ffmpeg getFfmpeg() {
        return ffmpeg;
    }
}
