package charactermanaj.clipboardSupport;

import java.awt.Color;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.image.BufferedImage;
import java.io.IOException;

import charactermanaj.graphics.io.ImageSaveHelper;

/**
 * クリップボードに画像をコピーするためのセレクション.<br>
 * @author seraphy
 */
public class ImageSelection implements Transferable {

	/**
	 * 対象となるイメージ
	 */
	private BufferedImage img;
	
	/**
	 * 背景色(jpeg画像変換時)
	 */
	private Color bgColor;
	
	/**
	 * JAVA内の転送用.
	 */
	public static final DataFlavor bufferedImageDataFlavor = new DataFlavor(BufferedImage.class, null);
	
	/**
	 * セレクションを構築する.
	 * @param img 対象となるイメージ
	 */
	public ImageSelection(BufferedImage img, Color bgColor) {
		if (img == null) {
			throw new IllegalArgumentException();
		}
		this.img = img;
		this.bgColor = (bgColor == null) ? Color.white : bgColor;
	}
	
	public Object getTransferData(DataFlavor flavor)
			throws UnsupportedFlavorException, IOException {
		if (flavor != null && flavor.equals(DataFlavor.imageFlavor)) {
			// 背景色で塗りつぶしたJPEG互換画像に変換して返す. 
			// JDK5/6のシステムクリップボードへのコピーでは透過画像をサポートしておらず透過部分が黒色になるため.
			ImageSaveHelper imageSaveHelper = new ImageSaveHelper();
			return imageSaveHelper.createJpegFormatPicture(img, bgColor);
		}

		if (flavor != null && flavor.equals(bufferedImageDataFlavor)) {
			// JAVA内データ交換用
			return img;
		}

		throw new UnsupportedFlavorException(flavor);
	}

	public DataFlavor[] getTransferDataFlavors() {
		return new DataFlavor[] {DataFlavor.imageFlavor, bufferedImageDataFlavor};
	}

	public boolean isDataFlavorSupported(DataFlavor flavor) {
		return flavor != null && (flavor.equals(DataFlavor.imageFlavor) || flavor.equals(bufferedImageDataFlavor));
	}
}
