package charactermanaj.graphics;

import java.awt.image.BufferedImage;
import java.lang.ref.SoftReference;
import java.util.HashMap;

/**
 * 画像のキャッシュ.<br>
 * キャッシュは自動的にガベージコレクタにより回収されます.<br>
 * ただし、{@link #unlockImages()}が呼び出されるまで、{@link #set(Object, BufferedImage)}されたイメージは
 * ガベージコレクトの対象にはなりません。 
 * @author seraphy
 *
 * @param <K>
 */
public class ImageCache<K> {

	private HashMap<K, BufferedImage> lockedImages
		= new HashMap<K, BufferedImage>();
	
	private HashMap<K, SoftReference<BufferedImage>> caches
		= new HashMap<K, SoftReference<BufferedImage>>();
	
	public BufferedImage get(K key) {
		if (key == null) {
			return null;
		}
		synchronized (caches) {
			SoftReference<BufferedImage> ref = caches.get(key);
			BufferedImage img = null;
			if (ref != null) {
				img = ref.get();
				if (img == null) {
					caches.remove(key);
				}
			}
			return img;
		}
	}

	public void set(K key, BufferedImage img) {
		if (key == null) {
			return;
		}
		synchronized (caches) {
			if (img == null) {
				caches.remove(key);
			} else {
				lockedImages.put(key, img);
				caches.put(key, new SoftReference<BufferedImage>(img));
			}
		}
	}
	
	public void unlockImages() {
		synchronized (caches) {
			lockedImages.clear();
		}
	}

}
