/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type.builtin;

import jp.bitmeister.asn1.annotation.ASN1Alternative;
import jp.bitmeister.asn1.annotation.ASN1BuiltIn;
import jp.bitmeister.asn1.processor.ASN1Visitor;
import jp.bitmeister.asn1.type.ASN1TagClass;
import jp.bitmeister.asn1.type.ASN1Type;
import jp.bitmeister.asn1.type.SelectiveType;

/**
 * Represents ASN.1 'CHOICE' type.
 * 
 * <p>
 * This is the base class used for defining 'CHOICE' types. A sub-class of
 * {@code CHOICE} shall have one or more fields annotated as
 * {@code @ASN1Alternative}. When a value is assigned to one of the
 * {@code @ASN1Alternative} fields of an instance, the field becomes selected
 * field. If an instance has more than two fields which assigned value, the
 * instance can't be encoded.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see ASN1Alternative
 */
@ASN1BuiltIn
public abstract class CHOICE extends SelectiveType {

	/**
	 * Instantiates an empty {@code CHOICE}.
	 */
	public CHOICE() {
	}

	/**
	 * Instantiates a {@code CHOICE} and initialize it with the parameter. The
	 * ASN.1 tag that assigned to the type of the data is used for select an
	 * alternative.
	 * 
	 * @param data
	 *            The data to be assigned.
	 */
	public CHOICE(ASN1Type data) {
		super(data);
	}

	/**
	 * Instantiates a {@code CHOICE} and initialize it with parameters.
	 * 
	 * @param tagClass
	 *            The tag class used for select an alternative.
	 * @param tagNumber
	 *            The tag number used for select an alternative.
	 * @param data
	 *            The data to be assigned.
	 */
	public CHOICE(ASN1TagClass tagClass, int tagNumber, ASN1Type data) {
		super(tagClass, tagNumber, data);
	}

	/**
	 * Instantiates a {@code CHOICE} and initialize it with parameters.
	 * 
	 * @param elementName
	 *            The element name used for select an alternative.
	 * @param data
	 *            The data to be assigned.
	 */
	public CHOICE(String elementName, ASN1Type data) {
		super(elementName, data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.type.ASN1Type#accept(jp.bitmeister.asn1.processor.
	 * ASN1Visitor)
	 */
	@Override
	public <R, E extends Throwable> R accept(ASN1Visitor<R, E> visitor)
			throws E {
		return visitor.visit(this);
	}

}
