/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import java.io.UnsupportedEncodingException;
import java.nio.charset.Charset;
import java.util.regex.Pattern;

import jp.bitmeister.asn1.exception.ASN1IllegalArgument;
import jp.bitmeister.asn1.exception.ASN1RuntimeException;
import jp.bitmeister.asn1.processor.ASN1Visitor;
import jp.bitmeister.asn1.type.builtin.OCTET_STRING;

/**
 * The base class for character string types.
 * 
 * <p>
 * This class provides generic interfaces and common methods for classes that
 * represents character string types.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
public abstract class StringType extends OCTET_STRING {
	
	/**
	 * Set the {@code String} to this instance. If the type has character
	 * limitation, characters contained in the string are checked before
	 * assignment.
	 * 
	 * @param value
	 *            The string value to be assigned.
	 */
	public void set(String value) {
		checkCharacters(value);
		try {
			set(value.getBytes(charset().displayName()));
		} catch (UnsupportedEncodingException e) {
			ASN1RuntimeException ex = new ASN1RuntimeException();
			ex.setMessage("Character set unsupported.", e, getClass(), null, this);
			throw ex;
		}
	}

	/**
	 * Converts the array of {@code byte} value of this instance to a
	 * {@code String} and returns it.
	 * 
	 * @return The string value assigned to this instance.
	 */
	public String stringValue() {
		try {
			return new String(value(), charset().displayName());
		} catch (UnsupportedEncodingException e) {
			ASN1RuntimeException ex = new ASN1RuntimeException();
			ex.setMessage("Character set unsupported.", e, getClass(), null, this);
			throw ex;
		}
	}

	/**
	 * Returns the {@code Charset} used for encoding and decodiong the value of
	 * this data.
	 * 
	 * @return The {@code Charset}.
	 */
	protected abstract Charset charset();

	/**
	 * Returns the {@code Pattern} used for checking characters that to be set
	 * to this data.
	 * 
	 * @return The {@code Pattern}.
	 */
	protected abstract Pattern pattern();

	/**
	 * Tests if the characters are valid for this type.
	 * 
	 * @param value
	 *            {@code true} when the characters is valid.
	 */
	protected void checkCharacters(String value) {
		if (pattern() != null && !pattern().matcher(value).matches()) {
			ASN1IllegalArgument ex = new ASN1IllegalArgument();
			ex.setMessage(
					"The value '"
							+ value
							+ "' contains characters which are not allowed to be contained in this type or is in invalid format for this type.",
					null, getClass(), null, null);
			throw ex;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.type.builtin.OCTET_STRING#accept(jp.bitmeister.asn1
	 * .processor.ASN1Visitor)
	 */
	@Override
	public <R, E extends Throwable> R accept(ASN1Visitor<R, E> visitor) throws E {
		return visitor.visit(this);
	}

}
