/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import static java.lang.reflect.Modifier.ABSTRACT;
import static java.lang.reflect.Modifier.INTERFACE;
import static java.lang.reflect.Modifier.PUBLIC;
import static java.lang.reflect.Modifier.STATIC;

import java.util.HashMap;
import java.util.Map;

import jp.bitmeister.asn1.exception.ASN1IllegalDefinition;

/**
 * Specifications of an ASN.1 module.
 * 
 * <p>
 * An instance of this class contains static information of an ASN.1 module.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
class ModuleSpecification {

	private String identifier;

	private ASN1TagDefault tagDefault;

	private Map<String, Class<? extends ASN1Type>> identifierMap = new HashMap<String, Class<? extends ASN1Type>>();

	private Map<ASN1TagClass, Map<Integer, Class<? extends ASN1Type>>> tagMap = new HashMap<ASN1TagClass, Map<Integer, Class<? extends ASN1Type>>>();

	private Map<String, Class<? extends ASN1Module>> importedModules = new HashMap<String, Class<? extends ASN1Module>>();

	/**
	 * Instantiates a {@code ModuleSpecification} contains specifications of the
	 * ASN.1 module.
	 * 
	 * @param module
	 *            The {@code Class} object of an {@code ASN1Module}.
	 */
	ModuleSpecification(Class<? extends ASN1Module> module) {
		identifier = ASN1Module.identifier(module);
		tagDefault = ASN1Module.tagDefault(module);
		if (ASN1Module.isUniversal(module)) {
			tagMap.put(ASN1TagClass.UNIVERSAL,
					new HashMap<Integer, Class<? extends ASN1Type>>());
		} else {
			tagMap.put(ASN1TagClass.APPLICATION,
					new HashMap<Integer, Class<? extends ASN1Type>>());
			tagMap.put(ASN1TagClass.PRIVATE,
					new HashMap<Integer, Class<? extends ASN1Type>>());
		}
		ASN1Module.registerTypes(module, this);
		ASN1Module.importModules(module, this);
	}

	/**
	 * Registers an imported module to this module.
	 * 
	 * @param module
	 *            An ASN.1 module imported by this module.
	 */
	void imports(Class<? extends ASN1Module> module) {
		importedModules.put(ASN1Module.identifier(module), module);
	}

	/**
	 * Registers a ASN.1 type to this module.
	 * 
	 * @param type
	 *            An ASN.1 type contained in this module.
	 */
	void register(Class<? extends ASN1Type> type) {
		if ((type.getModifiers() & (ABSTRACT | INTERFACE)) != 0) {
			ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
			ex.setMessage("An ASN.1 type class must be instantiatable.", null,
					type, null, null);
			throw ex;
		}
		if ((type.getModifiers() & PUBLIC) == 0) {
			ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
			ex.setMessage("An ASN.1 type class must be a public class", null,
					type, null, null);
			throw ex;
		}
		if (type.isMemberClass() && (type.getModifiers() & STATIC) == 0) {
			ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
			ex.setMessage(
					"If an ASN.1 type class is a member class, it must be static.",
					null, type, null, null);
			throw ex;
		}
		TypeSpecification typeSpec = TypeSpecification.getSpecification(type);
		if (typeSpec.tag() != null) {
			Map<Integer, Class<? extends ASN1Type>> map = tagMap.get(typeSpec
					.tag().tagClass());
			if (map == null) {
				ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
				ex.setMessage("Tag class '" + typeSpec.tag().tagClass()
						+ "'is not allowed to this module.", null, type, null,
						null);
				throw ex;
			}
			if (map.containsKey(typeSpec.tag().tagNumber())) {
				ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
				ex.setMessage("Tag number '" + typeSpec.tag().tagNumber()
						+ "' is dupulicated.", null, type, null, null);
				throw ex;
			}
			map.put(typeSpec.tag().tagNumber(), type);
		}
		identifierMap.put(typeSpec.xmlTypeName(), type);
	}

	/**
	 * Returns identifier of this module.
	 * 
	 * @return Identifier.
	 */
	String identifier() {
		return identifier;
	}

	/**
	 * Returns default tag mode of this module.
	 * 
	 * @return Default tag mode.
	 */
	ASN1TagDefault tagDefault() {
		return tagDefault;
	}

	/**
	 * Instantiates an ASN.1 data specified by the tag.
	 * 
	 * @param tagClass
	 *            The tag class.
	 * @param tagNumber
	 *            The tag number.
	 * @return An instance of ASN.1 data.
	 */
	ASN1Type instantiate(ASN1TagClass tagClass, int tagNumber) {
		Class<? extends ASN1Type> type = tagMap.get(tagClass).get(tagNumber);
		if (type != null) {
			return ASN1Type.instantiate(type);
		}
		for (Class<? extends ASN1Module> clazz : importedModules
				.values()) {
			ModuleSpecification spec = ASN1ModuleManager.specification(clazz);
			ASN1Type data = spec.instantiate(tagClass, tagNumber);
			if (data != null) {
				return data;
			}
		}
		return null;
	}

	/**
	 * Instantiates an ASN.1 data specified by the type identifier.
	 * 
	 * @param typeIdentifier
	 *            The identifier of the type.
	 * @return An instance of ASN.1 data.
	 */
	ASN1Type instantiate(String typeIdentifier) {
		if (typeIdentifier.indexOf('.') >= 0) {
			String[] identifier = typeIdentifier.split(".");
			return instantiate(identifier[0], identifier[1]);
		}
		Class<? extends ASN1Type> type = identifierMap.get(typeIdentifier);
		if (type != null) {
			return ASN1Type.instantiate(type);
		}
		return null;
	}

	/**
	 * Instantiates an ASN.1 data specified by the module identifier and the
	 * type identifier.
	 * 
	 * @param moduleIdentifier
	 *            The identifier of the module.
	 * @param typeIdentifier
	 *            The identifier of the type.
	 * @return An instance of ASN.1 data.
	 */
	private ASN1Type instantiate(String moduleIdentifier, String typeIdentifier) {
		ModuleSpecification spec = ASN1ModuleManager
				.specification(importedModules.get(moduleIdentifier));
		if (spec != null) {
			return spec.instantiate(typeIdentifier);
		}
		return null;
	}

}
