/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import java.util.ArrayList;
import java.util.List;

import jp.bitmeister.asn1.exception.ASN1IllegalDefinition;

/**
 * Used for checking element definitions.
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
abstract class ElementsChecker {

	Class<? extends StructuredType> type;

	private List<Integer> contextTags = new ArrayList<Integer>();

	private List<Integer> privateTags = new ArrayList<Integer>();

	/**
	 * Instantiates an {@code ElementChecker}.
	 * 
	 * @param type
	 *            The type contains elements.
	 */
	ElementsChecker(Class<? extends StructuredType> type) {
		this.type = type;
	}

	/**
	 * Checks element tags are not duplicated.
	 * 
	 * @param element
	 *            The element definition to be checked.
	 */
	void checkTagDuplication(NamedTypeSpecification element) {
		try {
			switch (element.tag().tagClass()) {
			case CONTEXT_SPECIFIC:
				if (contextTags.contains(element.tag().tagNumber())) {
					throw new ASN1IllegalDefinition();
				}
				contextTags.add(element.tag().tagNumber());
				break;
			case PRIVATE:
				if (privateTags.contains(element.tag().tagNumber())) {
					throw new ASN1IllegalDefinition();
				}
				privateTags.add(element.tag().tagNumber());
				break;
			}
		} catch (ASN1IllegalDefinition ex) {
			ex.setMessage("Tag '" + element.tag().tagClass() + " "
					+ element.tag().tagNumber() + "' is duplicated.", null,
					type, element.identifier(), null);
			throw ex;
		}
	}

}

/**
 * Used for checking elements of {@code SET} and {@code CHOICE} types.
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
class UnorderedElementsChecker extends ElementsChecker {

	private List<NamedTypeSpecification> untagged = new ArrayList<NamedTypeSpecification>();

	/**
	 * Instantiates an {@code UnorderedElementsChecker}.
	 * 
	 * @param type
	 *            The type contains elements.
	 */
	UnorderedElementsChecker(Class<? extends StructuredType> type) {
		super(type);
	}

	/**
	 * Checks all defined elements.
	 * 
	 * @param elements
	 *            The defined elements.
	 */
	void check(NamedTypeSpecification[] elements) {
		for (NamedTypeSpecification e : elements) {
			if (e.tag() == null) {
				checkUntagged(e);
			} else {
				checkTagDuplication(e);
			}
		}
	}

	/**
	 * Checks the untagged element is not the same type of any other untagged
	 * elements.
	 * 
	 * @param element
	 *            The untagged element.
	 */
	private void checkUntagged(NamedTypeSpecification element) {
		for (NamedTypeSpecification u : untagged) {
			if (element != u && element.isSameType(u)) {
				ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
				ex.setMessage(
						"Ambiguous element declaration. Check '"
								+ u.identifier() + "'.", null, type,
						element.identifier(), null);
				throw ex;
			}
		}
		untagged.add(element);
	}

}

/**
 * Used for checking elements of {@code SEQUENCE} types.
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
class OrderedElementsChecker extends ElementsChecker {

	private ElementSpecification prevElement;

	/**
	 * Instantiates an {@code OrderedElementsChecker}.
	 * 
	 * @param type
	 *            The type contains elements.
	 */
	OrderedElementsChecker(Class<? extends StructuredType> type) {
		super(type);
	}

	/**
	 * Checks all defined elements.
	 * 
	 * @param elements
	 *            The defined elements.
	 */
	void check(ElementSpecification[] elements) {
		for (ElementSpecification e : elements) {
			if (e.tag() == null) {
				if (prevElement != null && prevElement.tag() == null
						&& prevElement.optional() && prevElement.isSameType(e)) {
					ASN1IllegalDefinition ex = new ASN1IllegalDefinition();
					ex.setMessage("Ambiguous element declaration. Check '"
							+ prevElement.identifier() + "'.", null, type,
							e.identifier(), null);
					throw ex;
				}
			} else {
				checkTagDuplication(e);
			}
			prevElement = e;
		}
	}
}
