/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import java.util.Collection;

import jp.bitmeister.asn1.type.builtin.SEQUENCE_OF;
import jp.bitmeister.asn1.type.builtin.SET_OF;

/**
 * The base class for structured types defined by referencing a single ASN.1
 * type.
 * 
 * <p>
 * This class provides generic interfaces and common methods for classes that
 * represents structured types which defined by referencing a single ASN.1 type.
 * This class is the parent class of {@link SEQUENCE_OF} and {@link SET_OF}.
 * </p>
 * 
 * @see SEQUENCE_OF
 * @see SET_OF
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
public abstract class CollectionType<T extends ASN1Type, C extends Collection<T>> extends ASN1Type
		implements SizeCountable {

	private Class<T> componentType;

	private C collection;

	/**
	 * Instantiate a {@code CollectionType} instance whose component is the
	 * type.
	 * 
	 * @param componentType
	 *            The {@code class} object of the component of this collection.
	 */
	protected CollectionType(Class<T> componentType) {
		this.componentType = componentType;
		collection = newCollection();
	}

	/**
	 * Instantiate a {@code CollectionType} instance whose component is the type
	 * and initialize it with the components.
	 * 
	 * @param componentType
	 *            The {@code class} object of the component of this collection.
	 * @param components
	 *            Components to be set this instance.
	 */
	protected CollectionType(Class<T> componentType, T... components) {
		this(componentType);
		for (T e : components) {
			collection.add(e);
		}
	}

	/**
	 * Returns the collection that contained in the data.
	 * 
	 * @return The collection.
	 */
	public C collection() {
		return collection;
	}
	
	/**
	 * Returns the type of component of this data.
	 * 
	 * @return The type of component.
	 */
	public Class<T> componentType() {
		return componentType;
	}

	/**
	 * Returns a {@link TypeSpecification} instance that associated to the
	 * component type.
	 * 
	 * @return {@link TypeSpecification} instance
	 */
	public TypeSpecification componentSpecification() {
		return TypeSpecification.getSpecification(componentType);
	}

	/**
	 * Instantiates a new collection of the component type.
	 * 
	 * @return A new collection.
	 */
	protected abstract C newCollection();

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.SizeCountable#size()
	 */
	public int size() {
		return collection.size();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#clear()
	 */
	@Override
	public void clear() {
		collection = newCollection();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#hasValue()
	 */
	@Override
	public boolean hasValue() {
		return true;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#valueEquals(java.lang.Object)
	 */
	@Override
	public boolean valueEquals(Object other) {
		if (other instanceof CollectionType<?, ?>) {
			return collection.equals(((CollectionType<?, ?>) other).collection);
		}
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#hashCode()
	 */
	@Override
	public int hashCode() {
		return collection.hashCode();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.type.ASN1Type#clone()
	 */
	@Override
	@SuppressWarnings("unchecked")
	public Object clone() {
		CollectionType<T, C> clone = ASN1Type.instantiate(getClass());
		clone.collection.addAll(collection);
		return clone;
	}

}
