/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.type;

import jp.bitmeister.asn1.annotation.ASN1DefinedTypes;
import jp.bitmeister.asn1.annotation.ASN1Identifier;
import jp.bitmeister.asn1.annotation.ASN1ImportedModules;
import jp.bitmeister.asn1.annotation.ASN1ModuleRef;
import jp.bitmeister.asn1.annotation.ASN1ModuleTags;

/**
 * The base class for classes which represents an ASN.1 module.
 * 
 * <p>
 * An instance of sub-class of {@code ASN1Module} represents an ASN.1 module.
 * ASN.1 types can indicate a module that they are included in by
 * {@code ASN1ModuleRef} annotation. In addition, ASN.1 types declared as
 * {@code public} {@code static} member classes of an sub-class of
 * {@code ASN1Module} class are automatically included in the module.
 * </p>
 * <p>
 * Default tagging mode for a module may be indicated by {@code @ASN1ModuleTags}
 * annotation or explicit tagging is used as default.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see ASN1ModuleManager
 * @see ASN1ModuleTags
 * @see ASN1ModuleRef
 */
public abstract class ASN1Module {

	/**
	 * Returns the default tagging mode of the ASN.1 module.
	 * 
	 * @param module
	 *            The ASN.1 module.
	 * @return The default tagging mode of the module.
	 */
	static ASN1TagDefault tagDefault(Class<? extends ASN1Module> module) {
		if (module.isAnnotationPresent(ASN1ModuleTags.class)) {
			return module.getAnnotation(ASN1ModuleTags.class).value();
		}
		return ASN1TagDefault.EXPLICIT_TAGS;
	}

	/**
	 * Returns the identifier of the ASN.1 module.
	 * 
	 * @param module
	 *            The ASN.1 module.
	 * @return The identifier of the module.
	 */
	static String identifier(Class<? extends ASN1Module> module) {
		if (module.isAnnotationPresent(ASN1Identifier.class)) {
			return module.getAnnotation(ASN1Identifier.class).value();
		}
		return module.getSimpleName();
	}

	/**
	 * Tests if the module is the universal module.
	 * 
	 * @param module
	 *            The module to be tested.
	 * @return {@code true} if the module is the universal module.
	 */
	static boolean isUniversal(Class<? extends ASN1Module> module) {
		return module == BuiltInModule.class;
	}

	/**
	 * Registers ASN.1 types which are contained in the ASN.1 module to the
	 * specification.
	 * 
	 * @param module
	 *            The ASN.1 module.
	 * @param spec
	 *            The module specification.
	 */
	static void registerTypes(Class<? extends ASN1Module> module,
			ModuleSpecification spec) {
		for (Class<?> e : module.getDeclaredClasses()) {
			if (ASN1Type.class.isAssignableFrom(e)) {
				@SuppressWarnings("unchecked")
				Class<? extends ASN1Type> type = (Class<? extends ASN1Type>) e;
				spec.register(type);
			}
		}
		if (module.isAnnotationPresent(ASN1DefinedTypes.class)) {
			for (Class<? extends ASN1Type> e : module.getAnnotation(
					ASN1DefinedTypes.class).value()) {
				spec.register(e);
			}
		}
	}

	/**
	 * Registers ASN.1 modules which are imported by the ASN.1 module to the
	 * specification.
	 * 
	 * @param module
	 *            The ASN.1 module.
	 * @param spec
	 *            The module specification.
	 */
	static void importModules(Class<? extends ASN1Module> module,
			ModuleSpecification spec) {
		if (module.isAnnotationPresent(ASN1ImportedModules.class)) {
			for (Class<? extends ASN1Module> e : module.getAnnotation(
					ASN1ImportedModules.class).value()) {
				spec.imports(e);
			}
		}
	}
	
}
