/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.codec.xer;

import static jp.bitmeister.asn1.codec.xer.XerConstants.SINGLE_TAG_END;
import static jp.bitmeister.asn1.codec.xer.XerConstants.TAG_START;

/**
 * Utility class used for escaping/unescaping strings.
 * 
 * <p>
 * This class provides utility methods used for escaping and unescaping strings
 * to follow XER encoding rules.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 */
class XerStringEscapeUtil {

	private static final String AMP = "amp";
	private static final String LT = "lt";
	private static final String GT = "gt";

	private static final String CTRL[] = { "nul", "soh", "stx", "etx", "eot",
			"enq", "ack", "bel", "bs", "", "", "vt", "ff", "", "so", "si",
			"dle", "dc1", "dc2", "dc3", "dc4", "nak", "syn", "etb", "can",
			"em", "sub", "esc", "is4", "is3", "is2", "is1" };

	/**
	 * Converts the source string to XER escaped string.
	 * 
	 * @param string
	 *            The source string.
	 * @return Escaped string.
	 */
	static String escape(String string) {
		StringBuilder builder = new StringBuilder();
		for (char c : string.toCharArray()) {
			if (c < 32 && c != 9 && c != 10 && c != 13) {
				builder.append(TAG_START).append(CTRL[c])
						.append(SINGLE_TAG_END);
			} else if (c == '&') {
				builder.append('&').append(AMP).append(';');
			} else if (c == '<') {
				builder.append('&').append(LT).append(';');
			} else if (c == '>') {
				builder.append('&').append(GT).append(';');
			} else {
				builder.append(c);
			}
		}
		return builder.toString();
	}

	/**
	 * Returns a control character specified by the string represents a control
	 * code.
	 * 
	 * @param string
	 *            The string represents a control code.
	 * @return A control character.
	 */
	static char unescapeCtrl(String string) {
		for (char i = 0; i < CTRL.length; i++) {
			if (string.equals(CTRL[i])) {
				return i;
			}
		}
		return 0;
	}

}
