/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.codec.xer;

import static jp.bitmeister.asn1.codec.xer.XerConstants.BOOLEAN_FALSE;
import static jp.bitmeister.asn1.codec.xer.XerConstants.BOOLEAN_TRUE;
import static jp.bitmeister.asn1.codec.xer.XerConstants.END_TAG_START;
import static jp.bitmeister.asn1.codec.xer.XerConstants.REAL_MINUS_INFINITY;
import static jp.bitmeister.asn1.codec.xer.XerConstants.REAL_PLUS_INFINITY;
import static jp.bitmeister.asn1.codec.xer.XerConstants.REAL_ZERO;
import static jp.bitmeister.asn1.codec.xer.XerConstants.SINGLE_TAG_END;
import static jp.bitmeister.asn1.codec.xer.XerConstants.TAG_END;
import static jp.bitmeister.asn1.codec.xer.XerConstants.TAG_START;
import static jp.bitmeister.asn1.codec.xer.XerConstants.XML_PROLOG;

import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;

import jp.bitmeister.asn1.codec.ASN1Encoder;
import jp.bitmeister.asn1.exception.ASN1EncodingException;
import jp.bitmeister.asn1.processor.ASN1Visitor;
import jp.bitmeister.asn1.type.ASN1Module;
import jp.bitmeister.asn1.type.ASN1Type;
import jp.bitmeister.asn1.type.CollectionType;
import jp.bitmeister.asn1.type.ConstructiveType;
import jp.bitmeister.asn1.type.ElementSpecification;
import jp.bitmeister.asn1.type.StringType;
import jp.bitmeister.asn1.type.TimeType;
import jp.bitmeister.asn1.type.UnknownType;
import jp.bitmeister.asn1.type.builtin.ANY;
import jp.bitmeister.asn1.type.builtin.BIT_STRING;
import jp.bitmeister.asn1.type.builtin.BOOLEAN;
import jp.bitmeister.asn1.type.builtin.CHOICE;
import jp.bitmeister.asn1.type.builtin.ENUMERATED;
import jp.bitmeister.asn1.type.builtin.INTEGER;
import jp.bitmeister.asn1.type.builtin.NULL;
import jp.bitmeister.asn1.type.builtin.OBJECT_IDENTIFIER;
import jp.bitmeister.asn1.type.builtin.OCTET_STRING;
import jp.bitmeister.asn1.type.builtin.REAL;
import jp.bitmeister.asn1.type.builtin.RELATIVE_OID;
import jp.bitmeister.asn1.type.builtin.SEQUENCE;
import jp.bitmeister.asn1.type.builtin.SEQUENCE_OF;
import jp.bitmeister.asn1.type.builtin.SET;
import jp.bitmeister.asn1.type.builtin.SET_OF;
import jp.bitmeister.asn1.value.BinString;
import jp.bitmeister.asn1.value.HexString;

/**
 * XER (XML Encoding Rules) encoder.
 * 
 * <p>
 * {@code XerEncoder} is an implementation of {@code ASN1Encoder}. It encodes an
 * ASN.1 data to an XML document with XML Encoding Rules(XER) and writes the
 * result to {@code OutputStream} that is specified when the encoder was
 * instantiated. XER encoding process is light-weight compared with Canonical
 * XML Encoding Rules (CXER) encoding because some restrictions on CXER are not
 * considered.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see ASN1Encoder
 * @see XerDecoder
 */
public class XerEncoder implements ASN1Encoder,
		ASN1Visitor<String, ASN1EncodingException> {

	private Class<? extends ASN1Module> module;

	private OutputStream out;

	private StringBuilder builder;

	private boolean prologIsEmpty = false;

	/**
	 * Instantiates a {@code XEREncoder}.
	 * 
	 * @param out
	 *            The {@code OutputStream} that encoded octets will be written.
	 */
	public XerEncoder(OutputStream out) {
		this.out = out;
	}

	/**
	 * Instantiates a {@code XEREncoder}.
	 * 
	 * @param module
	 *            The ASN.1 module used for encoding.
	 * @param out
	 *            The {@code OutputStream} that encoded octets will be written.
	 */
	public XerEncoder(Class<? extends ASN1Module> module, OutputStream out) {
		this(out);
		this.module = module;
	}

	/**
	 * Instantiates a {@code XEREncoder}.
	 * 
	 * @param out
	 *            The {@code OutputStream} that encoded octets will be written.
	 * @param prologIsEmpty
	 *            If {@code true}, the XML prolog for result XML document will
	 *            be empty.
	 */
	public XerEncoder(OutputStream out, boolean prologIsEmpty) {
		this(out);
		this.prologIsEmpty = prologIsEmpty;
	}

	/**
	 * Instantiates a {@code XEREncoder}.
	 * 
	 * @param module
	 *            The ASN.1 module used for encoding.
	 * @param out
	 *            The {@code OutputStream} that encoded octets will be written.
	 * @param prologIsEmpty
	 *            If {@code true}, the XML prolog for result XML document will
	 *            be empty.
	 */
	public XerEncoder(Class<? extends ASN1Module> module, OutputStream out, boolean prologIsEmpty) {
		this(module, out);
		this.prologIsEmpty = prologIsEmpty;
	}
	
	/**
	 * Encodes an ASN.1 data.
	 * 
	 * @param data
	 *            The ASN.1 data to be encoded.
	 * @return The size of encoded octets.
	 * @throws ASN1EncodingException
	 *             When an error occurred while the encoding process.
	 */	
	public int encode(ASN1Type data) throws ASN1EncodingException {
		if (module == null) {
			module = data.specification().module();
		}
		builder = new StringBuilder();
		if (!prologIsEmpty) {
			builder.append(XML_PROLOG);
		}
		encodeImpl(data, data.specification().xmlTypeName(module));
		try {
			byte[] result = builder.toString().getBytes("UTF-8");
			out.write(result);
			return result.length;
		} catch (Exception e) {
			ASN1EncodingException ex = new ASN1EncodingException();
			ex.setMessage("Failed to write result to stream.", e, null, null,
					data);
			throw ex;
		}
	}

	/**
	 * Encodes the source data to xml documents and writes it to the
	 * {@code builder}.
	 * 
	 * @param data
	 *            The ASN.1 data to be encoded.
	 * @param xmlTagName
	 *            The xml tag name to be set.
	 * @throws ASN1EncodingException
	 *             When an error occurred while the encoding process.
	 */
	private void encodeImpl(ASN1Type data, String xmlTagName)
			throws ASN1EncodingException {
		String contents = data.accept(this);
		if (contents == null) {
			return;
		} else if (contents.length() == 0) {
			builder.append(TAG_START).append(xmlTagName).append(SINGLE_TAG_END);
		} else {
			builder.append(TAG_START).append(xmlTagName).append(TAG_END);
			builder.append(contents);
			builder.append(END_TAG_START).append(xmlTagName).append(TAG_END);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.BOOLEAN)
	 */
	public String visit(BOOLEAN data) {
		return TAG_START + (data.value() ? BOOLEAN_TRUE : BOOLEAN_FALSE)
				+ SINGLE_TAG_END;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.INTEGER)
	 */
	public String visit(INTEGER data) {
		String numberId = data.identifier();
		if (numberId != null) {
			return TAG_START + numberId + SINGLE_TAG_END;
		} else {
			return data.value().toString();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.BIT_STRING)
	 */
	public String visit(BIT_STRING data) {
		if (data.hasNamedBits()) {
			StringBuilder builder = new StringBuilder();
			for (int i = 0; i < data.size(); i++) {
				if (data.bit(i)) {
					String name = data.nameOfBit(i);
					if (name != null) {
						builder.append(TAG_START).append(name)
								.append(SINGLE_TAG_END);
					}
				}
			}
			return builder.toString();
		} else {
			return new BinString(data.value()).string();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.OCTET_STRING)
	 */
	public String visit(OCTET_STRING data) {
		return new HexString(data.value()).string();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.NULL)
	 */
	public String visit(NULL data) {
		return "";
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.OBJECT_IDENTIFIER)
	 */
	public String visit(OBJECT_IDENTIFIER data) {
		StringBuilder builder = new StringBuilder();
		if (data.value().size() > 0) {
			builder.append(data.value().get(0));
			for (int i = 1; i < data.value().size(); i++) {
				builder.append('.').append(data.value().get(i));
			}
		}
		return builder.toString();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.RELATIVE_OID)
	 */
	public String visit(RELATIVE_OID data) {
		return visit((OBJECT_IDENTIFIER) data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.REAL)
	 */
	public String visit(REAL data) {
		if (data.value().isInfinite()) {
			return TAG_START
					+ (data.value() == Double.POSITIVE_INFINITY ? REAL_PLUS_INFINITY
							: REAL_MINUS_INFINITY) + SINGLE_TAG_END;
		}
		if (data.value() == 0) {
			return REAL_ZERO;
		}
		return BigDecimal.valueOf(data.value()).stripTrailingZeros()
				.toPlainString();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.ENUMERATED)
	 */
	public String visit(ENUMERATED data) {
		return visit((INTEGER) data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.ANY)
	 */
	public String visit(ANY data) throws ASN1EncodingException {
		encodeImpl(data.value(),
				data.value().specification().xmlTypeName(module));
		return null;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.CHOICE)
	 */
	public String visit(CHOICE data) throws ASN1EncodingException {
		StringBuilder enclosure = builder;
		builder = new StringBuilder();
		encodeImpl(data.selectedValue(), data.selectedIdentifier());
		String result = builder.toString();
		builder = enclosure;
		return result;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SEQUENCE_OF)
	 */
	public String visit(SEQUENCE_OF<? extends ASN1Type> data)
			throws ASN1EncodingException {
		return processCollection(data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SEQUENCE)
	 */
	public String visit(SEQUENCE data) throws ASN1EncodingException {
		return processConstructive(data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SET_OF)
	 */
	public String visit(SET_OF<? extends ASN1Type> data)
			throws ASN1EncodingException {
		return processCollection(data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SET)
	 */
	public String visit(SET data) throws ASN1EncodingException {
		return processConstructive(data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .StringType)
	 */
	public String visit(StringType data) throws ASN1EncodingException {
		try {
			return new String(XerStringEscapeUtil.escape(data.stringValue())
					.getBytes("Shift_JIS"));
		} catch (UnsupportedEncodingException e) {
			throw new ASN1EncodingException();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .TimeType)
	 */
	public String visit(TimeType data) throws ASN1EncodingException {
		return data.stringValue();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .UnknownType)
	 */
	public String visit(UnknownType data) throws ASN1EncodingException {
		ASN1EncodingException ex = new ASN1EncodingException();
		ex.setMessage("Can't encode unknown type.", null, data.getClass(),
				null, data);
		throw ex;
	}

	/**
	 * Encodes each element of the {@code ConstructiveType} data.
	 * 
	 * @param data
	 *            The data to be encoded.
	 * @throws ASN1EncodingException
	 *             When an error occurred while the encoding process.
	 */
	private String processConstructive(ConstructiveType data)
			throws ASN1EncodingException {
		StringBuilder enclosure = builder;
		builder = new StringBuilder();
		for (ElementSpecification e : data.getElementTypeList()) {
			ASN1Type component = data.getComponent(e);
			if (component != null) {
				encodeImpl(component, e.identifier());
			}
		}
		String result = builder.toString();
		builder = enclosure;
		return result;
	}

	/**
	 * Encodes each component of the {@code CollectionType} data.
	 * 
	 * @param data
	 *            The data to be encoded.
	 * @throws ASN1EncodingException
	 *             When an error occurred while the encoding process.
	 */
	private String processCollection(CollectionType<? extends ASN1Type, ?> data)
			throws ASN1EncodingException {
		StringBuilder enclosure = builder;
		builder = new StringBuilder();
		if (BOOLEAN.class.isAssignableFrom(data.componentType())
				|| ENUMERATED.class.isAssignableFrom(data.componentType())
				|| CHOICE.class.isAssignableFrom(data.componentType())) {
			for (ASN1Type e : data.collection()) {
				builder.append(e.accept(this));
			}
		} else {
			for (ASN1Type e : data.collection()) {
				encodeImpl(e, e.specification().xmlTypeName(module));
			}
		}
		String result = builder.toString();
		builder = enclosure;
		return result;
	}

}
