/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.codec.xer;

import static jp.bitmeister.asn1.codec.xer.XerConstants.BOOLEAN_FALSE;
import static jp.bitmeister.asn1.codec.xer.XerConstants.BOOLEAN_TRUE;
import static jp.bitmeister.asn1.codec.xer.XerConstants.REAL_MINUS_INFINITY;
import static jp.bitmeister.asn1.codec.xer.XerConstants.REAL_PLUS_INFINITY;

import java.io.InputStream;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;

import jp.bitmeister.asn1.codec.ASN1Decoder;
import jp.bitmeister.asn1.exception.ASN1DecodingException;
import jp.bitmeister.asn1.processor.ASN1Visitor;
import jp.bitmeister.asn1.type.ASN1Module;
import jp.bitmeister.asn1.type.ASN1ModuleManager;
import jp.bitmeister.asn1.type.ASN1Type;
import jp.bitmeister.asn1.type.CollectionType;
import jp.bitmeister.asn1.type.NamedTypeSpecification;
import jp.bitmeister.asn1.type.StringType;
import jp.bitmeister.asn1.type.StructuredType;
import jp.bitmeister.asn1.type.TimeType;
import jp.bitmeister.asn1.type.UnknownType;
import jp.bitmeister.asn1.type.builtin.ANY;
import jp.bitmeister.asn1.type.builtin.BIT_STRING;
import jp.bitmeister.asn1.type.builtin.BOOLEAN;
import jp.bitmeister.asn1.type.builtin.CHOICE;
import jp.bitmeister.asn1.type.builtin.ENUMERATED;
import jp.bitmeister.asn1.type.builtin.INTEGER;
import jp.bitmeister.asn1.type.builtin.NULL;
import jp.bitmeister.asn1.type.builtin.OBJECT_IDENTIFIER;
import jp.bitmeister.asn1.type.builtin.OCTET_STRING;
import jp.bitmeister.asn1.type.builtin.REAL;
import jp.bitmeister.asn1.type.builtin.RELATIVE_OID;
import jp.bitmeister.asn1.type.builtin.SEQUENCE;
import jp.bitmeister.asn1.type.builtin.SEQUENCE_OF;
import jp.bitmeister.asn1.type.builtin.SET;
import jp.bitmeister.asn1.type.builtin.SET_OF;
import jp.bitmeister.asn1.value.BinString;
import jp.bitmeister.asn1.value.HexString;

import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;
import org.xml.sax.helpers.XMLReaderFactory;

/**
 * XER (XML Encoding Rules) decoder.
 * 
 * <p>
 * {@code XerDecoder} is an implementation of {@code ASN1Decoder}. It reads a
 * number of bytes from an {@code InputStream} that is specified when a decoder
 * is instantiated, and decodes them to an ASN.1 data with XML Encoding Rules
 * (XER).
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see ASN1Decoder
 * @see XerEncoder
 */
public class XerDecoder implements ASN1Decoder,
		ASN1Visitor<XerDecoder.DataDecoder, SAXException> {

	/**
	 * Abstract handler class used for decoding an XML element to an ASN.1 data.
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 */
	abstract class DataDecoder {

		/**
		 * Callback method that is called when the data element started.
		 */
		void startElement(String qName) throws SAXException {
		}

		/**
		 * Callback method that is called when a child element started.
		 * 
		 * @param qName
		 *            The element name.
		 * @throws SAXException
		 *             Thrown when an error occurred while the process.
		 */
		void startChildElement(String qName) throws SAXException {
		}

		/**
		 * Callback method that is called when a character sequence detected in
		 * the XML element.
		 * 
		 * @param characters
		 *            The character sequence.
		 */
		void characters(String characters) {
		}

		/**
		 * Callback method that is called when a child element ended.
		 * 
		 * @param qName
		 *            The element name.
		 * @throws SAXException
		 *             Thrown when an error occurred while the process.
		 */
		void endChildElement(String qName) {
		}

		/**
		 * Callback method that is called when the data element ended.
		 */
		void endElement() {
		}

	}

	/**
	 * Handler class for constructed data ({@code CollectionType} and
	 * {@code StructuredType}).
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 */
	private abstract class ConstructedDataDecoder extends DataDecoder {

		ElementDecoder decoder;

		/*
		 * (non-Javadoc)
		 * 
		 * @see
		 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#characters(java
		 * .lang.String)
		 */
		@Override
		void characters(String characters) {
			if (decoder != null) {
				decoder.characters(characters);
			}
		}

	}

	/**
	 * Abstract handler class used for decoding a sequence of XML elements to an
	 * ASN.1 type data.
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 */
	private abstract class ElementDecoder {

		DataDecoder decoder;

		/**
		 * Instantiate an {@code ElementDecoder}.
		 * 
		 * @param decoder
		 *            The {@code DataDecoder} used for decode the data.
		 */
		ElementDecoder(DataDecoder decoder) {
			this.decoder = decoder;
		}

		/**
		 * Callback method that is called when an XML element started.
		 * 
		 * @param qName
		 *            The tag name of the element.
		 * @throws SAXException
		 *             Thrown when an error occurred while the process.
		 */
		abstract void startElement(String qName) throws SAXException;

		/**
		 * Callback method that is called when a character sequence detected in
		 * an XML element.
		 * 
		 * @param characters
		 *            The character sequence.
		 */
		void characters(String characters) {
			decoder.characters(characters);
		}

		/**
		 * Callback method that is called when an XML element ended.
		 * 
		 * @param qName
		 *            The tag name of the element.
		 * @return {@code true} when the element is enclosing element.
		 */
		abstract boolean endElement(String qName);

	}

	/**
	 * Handler class used for decoding a sequence of data elements that enclosed
	 * by an element.
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 */
	private class NamedElementDecoder extends ElementDecoder {

		private String tagName;

		/**
		 * Instantiate a {@code NamedElementDecoder}.
		 * 
		 * @param tagName
		 *            The tag name of enclosing element.
		 * @param decoder
		 *            The {@code DataDecoder} used for decoding data elements.
		 */
		NamedElementDecoder(String tagName, DataDecoder decoder) {
			super(decoder);
			this.tagName = tagName;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see
		 * jp.bitmeister.asn1.codec.xer.XerDecoder.ElementDecoder#startElement
		 * (java.lang.String)
		 */
		public void startElement(String qName) throws SAXException {
			if (qName.equals(tagName)) {
				decoder.startElement(qName);
			} else {
				decoder.startChildElement(qName);
			}
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see
		 * jp.bitmeister.asn1.codec.xer.XerDecoder.ElementDecoder#endElement
		 * (java.lang.String)
		 */
		public boolean endElement(String qName) {
			if (qName.equals(tagName)) {
				decoder.endElement();
				return true;
			} else {
				decoder.endChildElement(qName);
				return false;
			}
		}

	}

	/**
	 * Handler class used for decoding an data element that is not enclosed by
	 * element.
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 */
	private class ValueListElementDecoder extends ElementDecoder {

		/**
		 * Instantiate a {@code ValueListElementDecoder}.
		 * 
		 * @param decoder
		 *            The {@code DataDecoder} used for decoding the data
		 *            element.
		 */
		ValueListElementDecoder(DataDecoder decoder) {
			super(decoder);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see
		 * jp.bitmeister.asn1.codec.xer.XerDecoder.ElementDecoder#startElement
		 * (java.lang.String)
		 */
		public void startElement(String qName) throws SAXException {
			decoder.startElement(qName);
			decoder.startChildElement(qName);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see
		 * jp.bitmeister.asn1.codec.xer.XerDecoder.ElementDecoder#endElement
		 * (java.lang.String)
		 */
		public boolean endElement(String qName) {
			decoder.endChildElement(qName);
			decoder.endElement();
			return true;
		}

	}

	/**
	 * Abstract handler class used for decosing an XML document to an ASN.1
	 * data.
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 */
	private abstract class XerHandler<T extends ASN1Type> extends
			DefaultHandler {

		T data;

		private ElementDecoder decoder;

		/**
		 * Instantiates an ASN.1 data and sets it to this instance.
		 * 
		 * @param qName
		 *            The tag name of the head element of the XML document.
		 * @throws SAXException
		 *             Thrown when an error occurred while the process.
		 */
		abstract void initializeData(String qName) throws SAXException;

		/*
		 * (non-Javadoc)
		 * 
		 * @see
		 * org.xml.sax.helpers.DefaultHandler#startElement(java.lang.String,
		 * java.lang.String, java.lang.String, org.xml.sax.Attributes)
		 */
		@Override
		public void startElement(String uri, String localName, String qName,
				Attributes attributes) throws SAXException {
			if (data == null) {
				initializeData(qName);
				decoder = new NamedElementDecoder(qName,
						data.accept(XerDecoder.this));
			}
			decoder.startElement(qName);
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.xml.sax.helpers.DefaultHandler#characters(char[], int, int)
		 */
		@Override
		public void characters(char[] ch, int start, int length) {
			char[] tmp = new char[length];
			System.arraycopy(ch, start, tmp, 0, length);
			decoder.characters(String.valueOf(tmp));
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see org.xml.sax.helpers.DefaultHandler#endElement(java.lang.String,
		 * java.lang.String, java.lang.String)
		 */
		@Override
		public void endElement(String uri, String localName, String qName) {
			decoder.endElement(qName);
		}

	}
	
	private Class<? extends ASN1Module> module;

	private InputStream in;

	private XMLReader reader;

	/**
	 * Instantiates a {@code XerDecoder}.
	 * 
	 * @param in
	 *            The {@code InputStream} to be read.
	 */
	public XerDecoder(InputStream in) {
		this.in = in;
	}

	/**
	 * Instantiates a {@code XerDecoder}.
	 * 
	 * @param in
	 *            The {@code InputStream} to be read.
	 */
	public XerDecoder(Class<? extends ASN1Module> module, InputStream in) {
		this(in);
		this.module = module;
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.codec.ASN1Decoder#decode(java.lang.Class)
	 */
	public <T extends ASN1Type> T decode(final Class<T> type)
			throws ASN1DecodingException {
		XerHandler<T> handler = new XerHandler<T>() {

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.XerHandler#initializeData
			 * (java.lang.String)
			 */
			void initializeData(String qName) throws SAXException {
				data = ASN1Type.instantiate(type);
				if (module == null) {
					module = data.specification().module();
				}
				if (!(data instanceof ANY)
						&& !qName.equals(data.specification().xmlTypeName(module))) {
					throw new SAXException("Unexpected xml tag name '" + qName
							+ "'. '" + data.specification().xmlTypeName(module)
							+ "' is expected.");
				}
			}

		};
		parse(handler);
		return handler.data;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see jp.bitmeister.asn1.codec.ASN1Decoder#decode()
	 */
	public ASN1Type decode() throws ASN1DecodingException {
		XerHandler<ASN1Type> handler = new XerHandler<ASN1Type>() {

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.XerHandler#initializeData
			 * (java.lang.String)
			 */
			void initializeData(String qName) throws SAXException {
				data = ASN1ModuleManager.instantiate(module, qName);
				if (data == null) {
					throw new SAXException("Unknown data type '" + qName + "'.");
				}
			}

		};
		parse(handler);
		return handler.data;
	}

	/**
	 * Decodes the source XML data read from the {@code InputStream} to the
	 * ASN.1 data.
	 * 
	 * @throws ASN1DecodingException
	 *             When an error occurred while the decoding process.
	 */
	private void parse(XerHandler<?> handler) throws ASN1DecodingException {
		try {
			reader = XMLReaderFactory.createXMLReader();
			reader.setContentHandler(handler);
			reader.parse(new InputSource(in));
		} catch (Exception e) {
			ASN1DecodingException ex = new ASN1DecodingException();
			ex.setMessage("An exception thrown while decoding process.", e,
					null, null, null);
			throw ex;
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.BOOLEAN)
	 */
	public DataDecoder visit(final BOOLEAN data) {
		return new DataDecoder() {

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#startChildElement
			 * (java.lang.String)
			 */
			@Override
			void startChildElement(String qName) {
				if (qName.equals(BOOLEAN_TRUE)) {
					data.set(true);
				} else if (qName.equals(BOOLEAN_FALSE)) {
					data.set(false);
				}
			}

		};
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.INTEGER)
	 */
	public DataDecoder visit(final INTEGER data) {
		return new DataDecoder() {

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#startChildElement
			 * (java.lang.String)
			 */
			@Override
			void startChildElement(String qName) {
				data.set(qName);
			}

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#characters
			 * (java.lang.String)
			 */
			@Override
			public void characters(String characters) {
				data.set(new BigInteger(characters));
			}

		};
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.BIT_STRING)
	 */
	public DataDecoder visit(final BIT_STRING data) {
		return new DataDecoder() {

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#startChildElement
			 * (java.lang.String)
			 */
			@Override
			public void startChildElement(String qName) {
				data.set(qName);
			}

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#characters
			 * (java.lang.String)
			 */
			@Override
			public void characters(String characters) {
				data.set(new BinString(characters));
			}

		};
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.OCTET_STRING)
	 */
	public DataDecoder visit(final OCTET_STRING data) {
		return new DataDecoder() {

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#characters
			 * (java.lang.String)
			 */
			@Override
			public void characters(String characters) {
				data.set(new HexString(characters));
			}

		};
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.NULL)
	 */
	public DataDecoder visit(NULL data) {
		return new DataDecoder() {
		};
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.OBJECT_IDENTIFIER)
	 */
	public DataDecoder visit(final OBJECT_IDENTIFIER data) {
		return new DataDecoder() {

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#characters
			 * (java.lang.String)
			 */
			@Override
			public void characters(String characters) {
				List<Integer> oids = new ArrayList<Integer>();
				for (String e : characters.split("\\.")) {
					e.trim();
					if (Character.isLetter(e.charAt(0))) {
						if (!e.endsWith(")")) {
							oids.add(OBJECT_IDENTIFIER.nameFormToInt(oids, e));
							continue;
						}
						e = e.substring(e.indexOf('(') + 1, e.indexOf(')'));
					}
					oids.add(Integer.parseInt(e));
				}
				data.set(oids);
			}

		};
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.REAL)
	 */
	public DataDecoder visit(final REAL data) {
		return new DataDecoder() {

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#startChildElement
			 * (java.lang.String)
			 */
			@Override
			public void startChildElement(String qName) {
				if (qName.equals(REAL_PLUS_INFINITY)) {
					data.set(Double.POSITIVE_INFINITY);
				} else if (qName.equals(REAL_MINUS_INFINITY)) {
					data.set(Double.NEGATIVE_INFINITY);
				}
			}

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#characters
			 * (java.lang.String)
			 */
			@Override
			public void characters(String characters) {
				data.set(Double.parseDouble(characters));
			}

		};
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.ENUMERATED)
	 */
	public DataDecoder visit(ENUMERATED data) {
		return visit((INTEGER) data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.RELATIVE_OID)
	 */
	public DataDecoder visit(RELATIVE_OID data) {
		return visit((OBJECT_IDENTIFIER) data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.ANY)
	 */
	public DataDecoder visit(final ANY data) {
		return new DataDecoder() {

			private DataDecoder decoder;

			@Override
			void startElement(String qName) throws SAXException {
				data.set(ASN1ModuleManager.instantiate(module, qName));
				decoder = data.value().accept(XerDecoder.this);
				decoder.startElement(qName);
			}

			@Override
			void startChildElement(String qName) throws SAXException {
				decoder.startChildElement(qName);
			}

			@Override
			void characters(String characters) {
				decoder.characters(characters);
			}

			@Override
			void endChildElement(String qName) {
				decoder.endChildElement(qName);
			}

			@Override
			void endElement() {
				decoder.endElement();
			}

		};
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.CHOICE)
	 */
	public DataDecoder visit(CHOICE data) {
		return decoderForStructuredType(data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SEQUENCE_OF)
	 */
	public DataDecoder visit(SEQUENCE_OF<? extends ASN1Type> data) {
		return decoderForCollectionType(data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SEQUENCE)
	 */
	public DataDecoder visit(SEQUENCE data) {
		return decoderForStructuredType(data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SET_OF)
	 */
	public DataDecoder visit(SET_OF<? extends ASN1Type> data) {
		return decoderForCollectionType(data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SET)
	 */
	public DataDecoder visit(SET data) {
		return decoderForStructuredType(data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .StringType)
	 */
	public DataDecoder visit(final StringType data) {
		return new DataDecoder() {

			private StringBuilder builder = new StringBuilder();

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#startChildElement
			 * (java.lang.String)
			 */
			@Override
			public void startChildElement(String qName) {
				builder.append(XerStringEscapeUtil.unescapeCtrl(qName));
			}

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#characters
			 * (java.lang.String)
			 */
			@Override
			public void characters(String characters) {
				builder.append(characters);
			}

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#endElement()
			 */
			@Override
			void endElement() {
				if (builder.length() > 0) {
					data.set(builder.toString());
				}
			}

		};
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .TimeType)
	 */
	public DataDecoder visit(TimeType data) {
		return visit((StringType) data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .UnknownType)
	 */
	public DataDecoder visit(UnknownType data) throws SAXException {
		throw new SAXException("UnkownType is not supported by XerDecoder.");
	}

	/**
	 * Returns handler for the {@code StructuredType}.
	 * 
	 * @param data
	 *            The {@code StructuredType} data.
	 * @return Handler for the data.
	 */
	private DataDecoder decoderForStructuredType(final StructuredType data) {
		return new ConstructedDataDecoder() {

			private NamedTypeSpecification namedType;

			private ASN1Type element;

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#startChildElement
			 * (java.lang.String)
			 */
			@Override
			void startChildElement(String qName) throws SAXException {
				if (decoder == null) {
					namedType = data.getElement(qName);
					element = namedType.instantiate();
					decoder = new NamedElementDecoder(qName,
							element.accept(XerDecoder.this));
				}
				decoder.startElement(qName);
			}

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#endChildElement
			 * (java.lang.String)
			 */
			@Override
			void endChildElement(String qName) {
				if (decoder.endElement(qName)) {
					data.set(namedType, element);
					decoder = null;
				}
			}

		};
	}

	/**
	 * Returns handler for the {@code CollectionType}.
	 * 
	 * @param data
	 *            The {@code CollectionType} data.
	 * @return Handler for the data.
	 */
	private <T extends ASN1Type> DataDecoder decoderForCollectionType(
			final CollectionType<T, ?> data) {
		return new ConstructedDataDecoder() {

			private T component;

			private boolean useXmlValueList = false;

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#startElement
			 * ()
			 */
			@Override
			void startElement(String qName) {
				if (BOOLEAN.class.isAssignableFrom(data.componentType())
						|| ENUMERATED.class.isAssignableFrom(data.componentType())
						|| CHOICE.class.isAssignableFrom(data.componentType())) {
					useXmlValueList = true;
				}
			}

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#startChildElement
			 * (java.lang.String)
			 */
			@Override
			void startChildElement(String qName) throws SAXException {
				if (decoder == null) {
					component = ASN1Type.instantiate(data.componentType());
					if (useXmlValueList) {
						decoder = new ValueListElementDecoder(
								component.accept(XerDecoder.this));
					} else {
						decoder = new NamedElementDecoder(qName,
								component.accept(XerDecoder.this));
					}
				}
				decoder.startElement(qName);
			}

			/*
			 * (non-Javadoc)
			 * 
			 * @see
			 * jp.bitmeister.asn1.codec.xer.XerDecoder.DataDecoder#endChildElement
			 * (java.lang.String)
			 */
			@Override
			void endChildElement(String qName) {
				if (decoder.endElement(qName)) {
					data.collection().add(component);
					decoder = null;
				}
			}

		};
	}

}
