/*
 * Copyright 2011 BitMeister Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package jp.bitmeister.asn1.codec.ber;

import java.io.IOException;
import java.io.OutputStream;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;

import jp.bitmeister.asn1.codec.ASN1Encoder;
import jp.bitmeister.asn1.exception.ASN1EncodingException;
import jp.bitmeister.asn1.processor.ASN1Visitor;
import jp.bitmeister.asn1.type.ASN1TagClass;
import jp.bitmeister.asn1.type.ASN1TagMode;
import jp.bitmeister.asn1.type.ASN1TagValue;
import jp.bitmeister.asn1.type.ASN1Type;
import jp.bitmeister.asn1.type.CollectionType;
import jp.bitmeister.asn1.type.ConstructiveType;
import jp.bitmeister.asn1.type.ElementSpecification;
import jp.bitmeister.asn1.type.StringType;
import jp.bitmeister.asn1.type.TimeType;
import jp.bitmeister.asn1.type.TypeSpecification;
import jp.bitmeister.asn1.type.UnknownType;
import jp.bitmeister.asn1.type.builtin.ANY;
import jp.bitmeister.asn1.type.builtin.BIT_STRING;
import jp.bitmeister.asn1.type.builtin.BOOLEAN;
import jp.bitmeister.asn1.type.builtin.CHOICE;
import jp.bitmeister.asn1.type.builtin.ENUMERATED;
import jp.bitmeister.asn1.type.builtin.INTEGER;
import jp.bitmeister.asn1.type.builtin.NULL;
import jp.bitmeister.asn1.type.builtin.OBJECT_IDENTIFIER;
import jp.bitmeister.asn1.type.builtin.OCTET_STRING;
import jp.bitmeister.asn1.type.builtin.REAL;
import jp.bitmeister.asn1.type.builtin.RELATIVE_OID;
import jp.bitmeister.asn1.type.builtin.SEQUENCE;
import jp.bitmeister.asn1.type.builtin.SEQUENCE_OF;
import jp.bitmeister.asn1.type.builtin.SET;
import jp.bitmeister.asn1.type.builtin.SET_OF;

/**
 * BER (Basic Encoding Rules) encoder.
 * 
 * <p>
 * {@code BerEncoder} is an implementation of {@code ASN1Encoder}. It encodes an
 * ASN.1 data to an array of {@code byte} with Basic Encoding Rules(BER) and
 * writes the result to {@code OutputStream} that is specified when the encoder
 * was instantiated. BER encoding process is light-weight compared with
 * Distinguished Encoding Rules (DER) encoding because some restrictions on DER
 * are not considered.
 * </p>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see ASN1Encoder
 * @see BerDecoder
 */
public class BerEncoder implements ASN1Encoder,
		ASN1Visitor<EncodedOctets, ASN1EncodingException> {

	private OutputStream out;

	/**
	 * Instantiates a BER encoder.
	 * 
	 * @param out
	 *            The {@code OutputStream} that encoded octets will be written.
	 */
	public BerEncoder(OutputStream out) {
		this.out = out;
	}

	/**
	 * Encodes an ASN.1 data.
	 * 
	 * @param data
	 *            The ASN.1 data to be encoded
	 * @return The size of encoded octets.
	 * @throws ASN1EncodingException
	 *             When an error occurred while the encoding process.
	 */
	public int encode(ASN1Type data) throws ASN1EncodingException {
		try {
			data.validate();
			return encode(data, null, null).write(out);
		} catch (ASN1EncodingException e) {
			throw e;
		} catch (Exception e) {
			ASN1EncodingException ex = new ASN1EncodingException();
			ex.setMessage("Exception thrown while encoding process.", e,
					data.getClass(), null, data);
			throw ex;
		}
	}

	/**
	 * Encodes an ASN.1 data or an element of ASN.1 data.
	 * 
	 * @param data
	 *            The ASN.1 data to be encoded.
	 * @param tag
	 *            The tag assigned to the element to be encoded.
	 * @param typeSpec
	 *            The {@code TypeSpecification} of a type that tagged
	 *            explicitly.
	 * @throws ASN1EncodingException
	 *             When an error occurred while the encoding process.
	 */
	EncodedOctets encode(ASN1Type data, ASN1TagValue tag,
			TypeSpecification typeSpec) throws ASN1EncodingException {
		if (typeSpec == null) {
			typeSpec = data.specification();
		}
		if (tag == null) {
			do {
				tag = typeSpec.tag();
				typeSpec = typeSpec.reference();
			} while (typeSpec != null && tag == null);
		}
		EncodedOctets octets;
		if (tag != null && tag.tagMode() == ASN1TagMode.EXPLICIT) {
			ConstructedOctets constructed = newConstructedOctets();
			constructed.addElement(encode(data, null, typeSpec));
			octets = constructed;
		} else {
			octets = data.accept(this);
		}
		if (tag != null) {
			octets.fix(tag.tagClass(), tag.tagNumber());
		}
		return octets;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.BOOLEAN)
	 */
	public EncodedOctets visit(BOOLEAN data) {
		return newPrimitiveOctets(data.value() ? (byte) 0xff : (byte) 0x00);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.INTEGER)
	 */
	public EncodedOctets visit(INTEGER data) {
		return newPrimitiveOctets(data.value().toByteArray());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.ENUMERATED)
	 */
	public EncodedOctets visit(ENUMERATED data) {
		return visit((INTEGER) data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.REAL)
	 */
	public EncodedOctets visit(REAL data) {
		byte[] encoded;
		if (data.value() == 0) { // zero.
			encoded = new byte[0];
		} else if (data.value().isInfinite()) { // special value.
			encoded = new byte[] { data.value() == Double.POSITIVE_INFINITY ? (byte) 0x40
					: 0x41 };
		} else { // ISO6093
			String str = data.value().toString();
			encoded = new byte[str.length() + 1];
			if (str.indexOf("E") < 0) {
				encoded[0] = 0x02; // NR2
			} else {
				encoded[0] = 0x03; // NR3
			}
			System.arraycopy(str.getBytes(), 0, encoded, 1, str.length());
		}
		return newPrimitiveOctets(encoded);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.BIT_STRING)
	 */
	public EncodedOctets visit(BIT_STRING data) {
		if (data.value().length == 0) {
			return newPrimitiveOctets((byte) 0x00);
		}
		int mod = data.value().length % 8;
		byte[] encoded = new byte[1 + data.value().length / 8
				+ (mod == 0 ? 0 : 1)];
		encoded[0] = (byte) (mod == 0 ? 0 : 8 - mod);
		int index = 1;
		int mask = 0x80;
		for (boolean b : data.value()) {
			if (b) {
				encoded[index] |= mask;
			}
			mask >>>= 1;
			if (mask == 0) {
				mask = 0x80;
				index++;
			}
		}
		return newPrimitiveOctets(encoded);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.OCTET_STRING)
	 */
	public EncodedOctets visit(OCTET_STRING data) throws ASN1EncodingException {
		return newPrimitiveOctets(data.value());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.NULL)
	 */
	public EncodedOctets visit(NULL data) throws ASN1EncodingException {
		return newPrimitiveOctets();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SEQUENCE)
	 */
	public EncodedOctets visit(SEQUENCE data) throws ASN1EncodingException {
		return processConstructive(data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SEQUENCE_OF)
	 */
	public EncodedOctets visit(SEQUENCE_OF<?> data)
			throws ASN1EncodingException {
		return processCollection(data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SET)
	 */
	public ConstructedOctets visit(SET data) throws ASN1EncodingException {
		return processConstructive(data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.SET_OF)
	 */
	public ConstructedOctets visit(SET_OF<?> data)
			throws ASN1EncodingException {
		return processCollection(data);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.CHOICE)
	 */
	public EncodedOctets visit(CHOICE data) throws ASN1EncodingException {
		return encode(data.selectedValue(), data.selectedTag(), null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.OBJECT_IDENTIFIER)
	 */
	public EncodedOctets visit(OBJECT_IDENTIFIER data)
			throws ASN1EncodingException {
		if (data.value().size() < 2 || data.value().get(0) < 0
				|| 2 < data.value().get(0) || data.value().get(1) < 0
				|| 39 < data.value().get(1)) {
			ASN1EncodingException ex = new ASN1EncodingException();
			ex.setMessage("Invalid OBJECT IDENTIFIER value.", null,
					data.getClass(), null, data);
			throw ex;
		}
		byte[] encoded = new byte[calculateEncodedOidSize(data, 2) + 1];
		encoded[0] = (byte) (data.value().get(0) * 40 + data.value().get(1));
		encodeOid(data, encoded, 2, 1);
		return newPrimitiveOctets(encoded);
	}
	
	/* (non-Javadoc)
	 * @see jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type.builtin.RELATIVE_OID)
	 */
	public EncodedOctets visit(RELATIVE_OID data) throws ASN1EncodingException {
		byte[] encoded = new byte[calculateEncodedOidSize(data, 0)];
		encodeOid(data, encoded, 0, 0);
		return newPrimitiveOctets(encoded);
	}	

	private int calculateEncodedOidSize(OBJECT_IDENTIFIER data, int start) throws ASN1EncodingException {
		int size = 0;
		for (int i = start; i < data.value().size(); i++) {
			if (data.value().get(i) < 0) {
				ASN1EncodingException ex = new ASN1EncodingException();
				ex.setMessage(
						"OBJECT IDENTIFIER value must be a positive number.",
						null, data.getClass(), null, data);
				throw ex;
			}
			size += sizeBy7bits(data.value().get(i));
		}
		return size;
	}
	
	private void encodeOid(OBJECT_IDENTIFIER data, byte[] dest, int start, int offset) {
		for (int i = start; i < data.value().size(); i++) {
			offset += encodeToMutipleOctets(dest, offset, data.value().get(i));
		}
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .StringType)
	 */
	public EncodedOctets visit(StringType data) throws ASN1EncodingException {
		return newPrimitiveOctets(data.value());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .TimeType)
	 */
	public EncodedOctets visit(TimeType data) throws ASN1EncodingException {
		return newPrimitiveOctets(data.value());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .builtin.ANY)
	 */
	public EncodedOctets visit(ANY data) throws ASN1EncodingException {
		return encode(data.value(), null, null);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * jp.bitmeister.asn1.processor.ASN1Visitor#visit(jp.bitmeister.asn1.type
	 * .UnknownType)
	 */
	public EncodedOctets visit(UnknownType data) throws ASN1EncodingException {
		EncodedOctets octets = newPrimitiveOctets(data.value());
		octets.fix(data.tagClass(), data.tagNumber());
		return octets;
	}

	/**
	 * Encodes each element of the {@code ConstructiveType} data.
	 * 
	 * @param data
	 *            The data to be encoded.
	 * @throws ASN1EncodingException
	 *             When an error occurred while the encoding process.
	 */
	ConstructedOctets processConstructive(ConstructiveType data)
			throws ASN1EncodingException {
		ConstructedOctets octets = newConstructedOctets();
		for (ElementSpecification e : data.getElementTypeList()) {
			ASN1Type element = data.getComponent(e);
			if (element == null || !element.hasValue()) {
				continue;
			}
			octets.addElement(encode(element, e.tag(), null));
		}
		return octets;
	}

	/**
	 * Encodes each component of the {@code CollectionType} data.
	 * 
	 * @param data
	 *            The data to be encoded.
	 * @throws ASN1EncodingException
	 *             When an error occurred while the encoding process.
	 */
	ConstructedOctets processCollection(CollectionType<? extends ASN1Type, ?> data)
			throws ASN1EncodingException {
		ConstructedOctets octets = newConstructedOctets();
		for (ASN1Type e : data.collection()) {
			octets.addElement(encode(e, null, null));
		}
		return octets;
	}

	/**
	 * Encodes ASN.1 tag to octets.
	 * 
	 * @param tagClass
	 *            The tag class.
	 * @param tagNumber
	 *            The tag number.
	 * @param isConstructed
	 *            {@code true} the data applied to this tag is constructed.
	 * @return Encoded octets.
	 */
	byte[] encodeTag(ASN1TagClass tagClass, int tagNumber, boolean isConstructed) {
		byte leading = 0;
		switch (tagClass) {
		case UNIVERSAL:
			// 0x00;
			break;
		case APPLICATION:
			leading = 0x40;
			break;
		case CONTEXT_SPECIFIC:
			leading = (byte) 0x80;
			break;
		case PRIVATE:
			leading = (byte) 0xc0;
			break;
		}
		if (isConstructed) {
			// constructed.
			leading |= 0x20;
		}
		if (tagNumber <= 0x1e) {
			leading |= tagNumber;
			return new byte[] { leading };
		}
		leading |= 0x1f;
		int size = sizeBy7bits(tagNumber);
		byte[] id = new byte[size + 1];
		id[0] = leading;
		encodeToMutipleOctets(id, 1, tagNumber);
		return id;
	}

	/**
	 * Encodes the length to octets.
	 * 
	 * @param length
	 *            The length.
	 * @return Encoded octets.
	 */
	byte[] encodeLength(int length) {
		if (length <= 0x7f) {
			// length in short form.
			return new byte[] { (byte) length };
		}
		// length in long form.
		byte[] value = BigInteger.valueOf(length).toByteArray();
		byte[] encoded = new byte[value.length + 1];
		encoded[0] = (byte) (value.length | 0x80);
		System.arraycopy(value, 0, encoded, 1, value.length);
		return encoded;
	}

	/**
	 * Encodes the value to long form octets.
	 * 
	 * @param dest
	 *            The array of byte where the result to be written.
	 * @param offset
	 *            The index in the {@code dest} array that indicates where the
	 *            result to be written.
	 * @param value
	 *            The integer value.
	 * @return The size of the result.
	 */
	private int encodeToMutipleOctets(byte[] dest, int offset, long value) {
		int size = sizeBy7bits(value);
		int lastIndex = offset + size - 1;
		for (int index = lastIndex; index >= offset; index--) {
			dest[index] |= (byte) (value & 0x7f);
			if (index != lastIndex) {
				dest[index] |= 0x80;
			}
			value >>= 7;
		}
		return size;
	}

	/**
	 * Estimates the size of the result that the value is encoded to long form.
	 * 
	 * @param value
	 *            The value to be encoded.
	 * @return Estimated size.
	 */
	private int sizeBy7bits(long value) {
		int size = 1;
		while ((value >>= 7) > 0) {
			size++;
		}
		return size;
	}

	/**
	 * Instantiates a new primitive octets.
	 * 
	 * @param contents
	 *            The content octets.
	 * @return An instance of primitive octets.
	 */
	EncodedOctets newPrimitiveOctets(byte... contents) {
		return new BerPrimitiveOctets(contents);
	}

	/**
	 * Instantiates a new constructed octets.
	 * 
	 * @return An instance of constructed octets.
	 */
	ConstructedOctets newConstructedOctets() {
		return new BerConstructedOctets();
	}

	/**
	 * Abstract base class for classes represent BER encoded octets.
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 */
	abstract class BerOctets implements EncodedOctets {

		byte[] identifier;

		byte[] length;

		/**
		 * Returns BER encoded octets length includes prefix and contents.
		 * 
		 * @return Total length of BER encoded octets.
		 */
		public int totalLength() {
			return identifier.length + length.length + contentsLength();
		}

		/**
		 * Sets identifier and length octets to fix this octets.
		 * 
		 * @param tagClass
		 *            The ASN.1 tag class for this data.
		 * @param tagNumber
		 *            The tag number for this data.
		 */
		public void fix(ASN1TagClass tagClass, int tagNumber) {
			identifier = encodeTag(tagClass, tagNumber, isConstructed());
			length = encodeLength(contentsLength());
		}

		/**
		 * Writes all BER encoded octets to the {@code OutputStream}
		 * 
		 * @param out
		 *            The stream to be written.
		 * @throws IOException
		 *             when {@code IOException} thrown by {@code OutputStream}.
		 */
		public int write(OutputStream out) throws IOException {
			out.write(identifier);
			out.write(length);
			writeContents(out);
			return totalLength();
		}

		/**
		 * Writes contents octets to the {@code OutputStream}.
		 * 
		 * @param out
		 *            The stream to be written.
		 * @throws IOException
		 *             when {@code OutputStream} throws {@code IOException}.
		 */
		abstract void writeContents(OutputStream out) throws IOException;

	}

	/**
	 * Represents primitive BER octets.
	 * 
	 * <p>
	 * This class represents a BER encoded octets that has only one contents
	 * octets.
	 * </p>
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 */
	class BerPrimitiveOctets extends BerOctets {

		private byte[] contents;

		/**
		 * Instantiates a {@code PrimitiveOctets}.
		 * 
		 * @param contents
		 */
		BerPrimitiveOctets(byte... contents) {
			this.contents = contents;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see jp.bitmeister.asn1.codec.ber.BerOctets#isConstructed()
		 */
		public boolean isConstructed() {
			return false;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see
		 * jp.bitmeister.asn1.codec.ber.BerEncoder2.BerOctets#contentsLength()
		 */
		public int contentsLength() {
			return contents.length;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see
		 * jp.bitmeister.asn1.codec.ber.BerEncoder2.BerOctets#writeContents(
		 * java.io.OutputStream)
		 */
		void writeContents(OutputStream out) throws IOException {
			out.write(contents);
		}

	}

	/**
	 * Represents constructed BER octets.
	 * 
	 * <p>
	 * This class represents a BER encoded octets that contains some contents
	 * octets.
	 * </p>
	 * 
	 * @author WATANABE, Jun. <jwat at bitmeister.jp>
	 */
	class BerConstructedOctets extends BerOctets implements ConstructedOctets {

		private List<EncodedOctets> elements = new ArrayList<EncodedOctets>();

		private int length;

		/**
		 * Appends an element to this octets.
		 * 
		 * @param element
		 *            The element to be added.
		 */
		public void addElement(EncodedOctets element) {
			elements.add(element);
			length += element.totalLength();
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see jp.bitmeister.asn1.codec.ber.BerOctets#isConstructed()
		 */
		public boolean isConstructed() {
			return true;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see jp.bitmeister.asn1.codec.ber.BerOctets#octetsLength()
		 */
		public int contentsLength() {
			return length;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see
		 * jp.bitmeister.asn1.codec.ber.BerEncoder.BerOctets#writeContents(java
		 * .io.OutputStream)
		 */
		void writeContents(OutputStream out) throws IOException {
			for (EncodedOctets e : elements) {
				e.write(out);
			}
		}

	}

}
