/*
 * IEEE802.19.1 Prototype System
 * $Rev$ $Date$
 *
 * Copyright 2011 National Institute of Information and Communications Technology 
 */
package jp.bitmeister.asn1.annotation;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

import jp.bitmeister.asn1.type.SelectiveType;
import jp.bitmeister.asn1.type.builtin.CHOICE;

/**
 * Indicates that a field is a 'CHOICE' alternative.
 * 
 * <p>
 * If an {@code @ASN1Alternative} annotation is present on a field, the field is
 * treated as an alternative of a sub-class of {@code SelectiveType}. A field
 * annotated as {@code @ASN1Alternative} must be declared as {@code public}
 * non-{@code final} non-{@code static}, and the type shall be a sub-class of
 * {@code ASN1Type}. The {@code value} parameter represents an order of
 * appearance of a field in an ASN.1 type definition and is used for automatic
 * tagging.
 * </p>
 * <p>
 * For example, following ASN.1 definition :
 * 
 * <pre>
 * Sample ::= CHOICE {
 * 	number	INTEGER,
 * 	bool	BOOLEAN }
 * </pre>
 * 
 * can be translated as :
 * 
 * <code><pre>
 * public class Sample extends CHOICE {
 * 
 * 	&#064;ASN1Alternative(0)
 * 	public INTEGER number;
 * 
 * 	&#064;ASN1Alternative(1)
 * 	public BOOLEAN bool;
 * 
 * }
 * </pre></code>
 * 
 * @author WATANABE, Jun. <jwat at bitmeister.jp>
 * 
 * @see SelectiveType
 * @see CHOICE
 */
@Target(ElementType.FIELD)
@Retention(RetentionPolicy.RUNTIME)
public @interface ASN1Alternative {

	/**
	 * Indicates defined order of this element.
	 * 
	 * @return Order of this element.
	 */
	public int value() default 0;

}
