/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.cat.glossary.ui;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.jface.preference.IPreferenceStore;
import org.osgi.framework.BundleContext;

import benten.cat.glossary.core.BentenGlossaryEngine;
import benten.cat.glossary.engine.AbstractBentenGlossaryEnginePlugin;
import benten.cat.glossary.ui.messages.CatGlossaryUiPluginMessages;
import benten.ui.AbstractBentenUiPlugin;
import benten.ui.preference.BentenPreference;
import benten.ui.preference.BentenPreference.Preference;

/**
 * CAT 用語集 UI プラグイン。
 *
 * <UL>
 * <LI>このプラグインを Eclipse 上で有効化するためのプラグイン・クラスです。
 * </UL>
 *
 * @author YAMAMOTO Koji
 */
public class CatGlossaryUiPlugin extends AbstractBentenUiPlugin {
	/**
	 * 用語集 UI プラグインのためのメッセージ。
	 */
	protected static final CatGlossaryUiPluginMessages fMsg = new CatGlossaryUiPluginMessages();

	/** 共用インスタンス */
	private static CatGlossaryUiPlugin plugin;

	/** loadExtensions メソッドが 1 度でも実行済みかどうか。 */
	private boolean fLoadExtensionsCalled = false;

	/**
	 * このプラグインが保持している用語集エンジン・プラグインのインスタンス。
	 */
	private AbstractBentenGlossaryEnginePlugin fGlossaryEnginePlugin = null;

	@Override
	public void start(final BundleContext context) throws Exception {
		super.start(context);
		plugin = this;
	}

	@Override
	public void stop(final BundleContext context) throws Exception {
		plugin = null;
		super.stop(context);
	}

	/**
	 * このプラグインの共用インスタンスを取得。
	 * @return このプラグインの共用インスタンス
	 */
	public static CatGlossaryUiPlugin getDefault() {
		return plugin;
	}

	/**
	 * このプラグインが利用する拡張をロード。
	 *
	 * <UL>
	 * <LI>デフォルトの用語集エンジン・プラグインのインスタンスを取得します。
	 * <LI>用語集をロードします。
	 * </UL>
	 *
	 * @throws IllegalArgumentException ロードに失敗した場合。
	 */
	public void loadExtensions() {
		try {
			fGlossaryEnginePlugin = newGlossaryEnginePluginInstance();
		} catch (final Exception e) {
			log(e);
			throw new IllegalArgumentException(fMsg.getMsg002(e.getMessage()));
		}

		// 翻訳元・先の言語設定は、プロジェクト毎ではなく全体で設定します。
		// なぜなら、用語集のロードは Eclipse 単位だからです。
		final IPreferenceStore store = BentenPreference.getStore();
		final String pivotLang = store.getString(BentenPreference.Preference.TRANS_PIVOT_LANG.toString());
		final String sourceLang = store.getString(BentenPreference.Preference.TRANS_SOURCE_LANG.toString());
		final String targetLang = store.getString(BentenPreference.Preference.TRANS_TARGET_LANG.toString());

		// プリファランスの設定を、そのままドライバに引継。
		fGlossaryEnginePlugin.getEngine().setPivotLang(pivotLang);
		fGlossaryEnginePlugin.getEngine().setLang(sourceLang, targetLang);

		try {
			fGlossaryEnginePlugin.load();
		} catch (final Exception e) {
			log(e);
			throw new IllegalArgumentException(fMsg.getMsg001(e.getMessage()));
		}

		fLoadExtensionsCalled = true;
	}

	/**
	 * このプラグインが保持している 用語集エンジンを取得。
	 *
	 * <UL>
	 * <LI>高速に用語集エンジンを取得したい場合に利用します。
	 * <LI>「設定」で指定した用語集がすでに読み込み済みのエンジンを取得できます。
	 * </UL>
	 *
	 * @return 用語集エンジンのインスタンス。
	 */
	public BentenGlossaryEngine getGlossaryEngine() {
		if (fLoadExtensionsCalled == false) {
			loadExtensions();
		}
		return fGlossaryEnginePlugin.getEngine();
	}

	/**
	 * 用語集エンジン・プラグインのインスタンスを新規に作成。
	 *
	 * <UL>
	 * <LI>自力で用語集エンジンに用語集を読み込ませたい場合に利用します。
	 * <LI>一般的にはこのメソッドは利用せずに {@link #getGlossaryEngine} を利用してエンジンを取得するほうが手軽で高速です。
	 * <LI>デフォルトでプラグインが保持している用語集とは別の用語集を利用したい場合などにのみ、このメソッドを利用します。
	 * <LI>このメソッドでは、所定のルールに従って拡張ポイントを検索して用語集エンジン・プラグインのインスタンスを新規作成します。
	 * </UL>
	 *
	 * @return 新規作成された用語集エンジンのインスタンス。必ず null 以外が戻ります。
	 * @throws IllegalArgumentException 用語集エンジン・プラグインのインスタンスが取得できなかった場合。
	 */
	public AbstractBentenGlossaryEnginePlugin newGlossaryEnginePluginInstance() {
		final Map<String, AbstractBentenGlossaryEnginePlugin> glossaryEnginePluginNameMap = getGlossaryEnginePluginMap();

		// 「設定」から用語集エンジンの名称を取得。
		// 該当する用語集エンジンの用語集をロードする。
		final IPreferenceStore store = BentenPreference.getStore();
		final String tmName = store.getString(Preference.GLOSSARY_NAME.name());
		final AbstractBentenGlossaryEnginePlugin enginePlugin = glossaryEnginePluginNameMap.get(tmName);
		if (enginePlugin == null) {
			throw new IllegalArgumentException(fMsg.getGetGlossaryEnginePluginE002());
		}

		return enginePlugin;
	}

	/**
	 * 用語集エンジン・プラグイン名のセットを取得。
	 *
	 * <UL>
	 * <LI>利用可能な用語集エンジン・プラグイン名のリストを表示したい場合などに利用します。
	 * </UL>
	 *
	 * @return 用語集エンジン・プラグイン名のセット
	 */
	public Set<String> getGlossaryEnginePluginNameSet() {
		return getGlossaryEnginePluginMap().keySet();
	}

	/**
	 * 拡張ポイントから知ることができる用語集エンジン・プラグイン名のマップを取得。
	 *
	 * @return 用語集エンジン・プラグイン名のマップ
	 */
	private Map<String, AbstractBentenGlossaryEnginePlugin> getGlossaryEnginePluginMap() {
		final Map<String, AbstractBentenGlossaryEnginePlugin> glossaryEnginePluginNameMap = new HashMap<String, AbstractBentenGlossaryEnginePlugin>();

		final IExtensionRegistry registry = Platform.getExtensionRegistry();
		final IExtensionPoint point = registry.getExtensionPoint("benten.cat.glossary.engine"); //$NON-NLS-1$
		if (point == null) {
			throw new IllegalArgumentException(fMsg.getGetGlossaryEnginePluginE001());
		}

		// このシステムに登録されている、すべての用語集エンジンの一覧を取得。
		// 取得された用語集エンジンの名前を記憶。
		for (final IExtension extension : point.getExtensions()) {
			for (final IConfigurationElement cfgElem : extension.getConfigurationElements()) {
				try {
					final String name = cfgElem.getAttribute("name"); //$NON-NLS-1$
					final Object obj = cfgElem.createExecutableExtension("class"); //$NON-NLS-1$
					if (obj instanceof AbstractBentenGlossaryEnginePlugin) {
						glossaryEnginePluginNameMap.put(name, (AbstractBentenGlossaryEnginePlugin) obj);
					}
				} catch (final CoreException e) {
					log(e);
				}
			}
		}

		return glossaryEnginePluginNameMap;
	}
}
