/*
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 */
package benten.twa.tmx.core;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;

import benten.twa.tmx.core.valueobject.BentenExtractTmxFragmentProcessInput;

/**
 * Apache Antタスク [BentenExtractTmxFragment]のクラス。
 *
 * TMX フラグメント抽出のための Ant タスク。<br>
 * このクラスでは、Apache Antタスクで一般的に必要なチェックなどのコーディングを肩代わりします。
 * 実際の処理は パッケージ[benten.twa.tmx.core]にBentenExtractTmxFragmentBatchProcessクラスを作成して記述してください。<br>
 * <br>
 * Antタスクへの組み込み例<br>
 * <pre>
 * &lt;taskdef name=&quot;bentenextracttmxfragment&quot; classname=&quot;benten.twa.tmx.core.BentenExtractTmxFragmentTask">
 *     &lt;classpath&gt;
 *         &lt;fileset dir=&quot;lib&quot; includes=&quot;*.jar&quot; /&gt;
 *         &lt;fileset dir=&quot;lib.ant&quot; includes=&quot;*.jar&quot; /&gt;
 *     &lt;/classpath&gt;
 * &lt;/taskdef&gt;
 * </pre>
 */
public class BentenExtractTmxFragmentTask extends Task {
    /**
     * TMX フラグメント抽出のための Ant タスク。
     */
    protected BentenExtractTmxFragmentProcessInput fInput = new BentenExtractTmxFragmentProcessInput();

    /**
     * フィールド [xliffdir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldXliffdirProcessed = false;

    /**
     * フィールド [tmxdir] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTmxdirProcessed = false;

    /**
     * フィールド [transtargetid] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTranstargetidProcessed = false;

    /**
     * フィールド [extractbystate] に値がセットされたかどうか。
     */
    protected boolean fIsFieldExtractbystateProcessed = false;

    /**
     * フィールド [includedate] に値がセットされたかどうか。
     */
    protected boolean fIsFieldIncludedateProcessed = false;

    /**
     * フィールド [ignorewhitespacetmxextract] に値がセットされたかどうか。
     */
    protected boolean fIsFieldIgnorewhitespacetmxextractProcessed = false;

    /**
     * フィールド [ignoremnemonickeytmxextract] に値がセットされたかどうか。
     */
    protected boolean fIsFieldIgnoremnemonickeytmxextractProcessed = false;

    /**
     * フィールド [transsourcelang] に値がセットされたかどうか。
     */
    protected boolean fIsFieldTranssourcelangProcessed = false;

    /**
     * verboseモードで動作させるかどうか。
     *
     * @param arg verboseモードで動作させるかどうか。
     */
    public void setVerbose(final boolean arg) {
        fInput.setVerbose(arg);
    }

    /**
     * verboseモードで動作させるかどうか。
     *
     * @return verboseモードで動作させるかどうか。
     */
    public boolean getVerbose() {
        return fInput.getVerbose();
    }

    /**
     * Antタスクの[xliffdir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 1<br>
     * 抽出元となる XLIFF ディレクトリ。<br>
     *
     * @param arg セットしたい値
     */
    public void setXliffdir(final String arg) {
        fInput.setXliffdir(arg);
        fIsFieldXliffdirProcessed = true;
    }

    /**
     * Antタスクの[xliffdir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 1<br>
     * 抽出元となる XLIFF ディレクトリ。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getXliffdir() {
        return fInput.getXliffdir();
    }

    /**
     * Antタスクの[tmxdir]アトリビュートのセッターメソッド。
     *
     * 項目番号: 2<br>
     * 抽出先となる TMX を格納するディレクトリ。<br>
     *
     * @param arg セットしたい値
     */
    public void setTmxdir(final String arg) {
        fInput.setTmxdir(arg);
        fIsFieldTmxdirProcessed = true;
    }

    /**
     * Antタスクの[tmxdir]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 2<br>
     * 抽出先となる TMX を格納するディレクトリ。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTmxdir() {
        return fInput.getTmxdir();
    }

    /**
     * Antタスクの[transtargetid]アトリビュートのセッターメソッド。
     *
     * 項目番号: 3<br>
     * 翻訳対象 id  を格納します。この id はファイル名の一部に利用されます。<br>
     *
     * @param arg セットしたい値
     */
    public void setTranstargetid(final String arg) {
        fInput.setTranstargetid(arg);
        fIsFieldTranstargetidProcessed = true;
    }

    /**
     * Antタスクの[transtargetid]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 3<br>
     * 翻訳対象 id  を格納します。この id はファイル名の一部に利用されます。<br>
     * 必須アトリビュートです。Apache Antタスク上で必ず値が指定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTranstargetid() {
        return fInput.getTranstargetid();
    }

    /**
     * Antタスクの[extractbystate]アトリビュートのセッターメソッド。
     *
     * 項目番号: 4<br>
     * 状態で抽出する場合の state 値。無指定なら state で抽出しない。target無しで絞り込みたい場合には、半角空白 1 つをセット。<br>
     *
     * @param arg セットしたい値
     */
    public void setExtractbystate(final String arg) {
        fInput.setExtractbystate(arg);
        fIsFieldExtractbystateProcessed = true;
    }

    /**
     * Antタスクの[extractbystate]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 4<br>
     * 状態で抽出する場合の state 値。無指定なら state で抽出しない。target無しで絞り込みたい場合には、半角空白 1 つをセット。<br>
     *
     * @return このフィールドの値
     */
    public String getExtractbystate() {
        return fInput.getExtractbystate();
    }

    /**
     * Antタスクの[includedate]アトリビュートのセッターメソッド。
     *
     * 項目番号: 5<br>
     * 出力ファイル名に日付を含めるかどうか。<br>
     *
     * @param arg セットしたい値
     */
    public void setIncludedate(final boolean arg) {
        fInput.setIncludedate(arg);
        fIsFieldIncludedateProcessed = true;
    }

    /**
     * Antタスクの[includedate]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 5<br>
     * 出力ファイル名に日付を含めるかどうか。<br>
     * デフォルト値[false]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public boolean getIncludedate() {
        return fInput.getIncludedate();
    }

    /**
     * Antタスクの[ignorewhitespacetmxextract]アトリビュートのセッターメソッド。
     *
     * 項目番号: 6<br>
     * ホワイトスペースを無視した内容の TMX を抽出するかどうか。HTML の場合には true を推奨。<br>
     *
     * @param arg セットしたい値
     */
    public void setIgnorewhitespacetmxextract(final boolean arg) {
        fInput.setIgnorewhitespacetmxextract(arg);
        fIsFieldIgnorewhitespacetmxextractProcessed = true;
    }

    /**
     * Antタスクの[ignorewhitespacetmxextract]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 6<br>
     * ホワイトスペースを無視した内容の TMX を抽出するかどうか。HTML の場合には true を推奨。<br>
     * デフォルト値[false]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public boolean getIgnorewhitespacetmxextract() {
        return fInput.getIgnorewhitespacetmxextract();
    }

    /**
     * Antタスクの[ignoremnemonickeytmxextract]アトリビュートのセッターメソッド。
     *
     * 項目番号: 7<br>
     * ニーモニック・キーを無視した内容の TMX を抽出するかどうか。日本風のニーモニックのみ対応。<br>
     *
     * @param arg セットしたい値
     */
    public void setIgnoremnemonickeytmxextract(final boolean arg) {
        fInput.setIgnoremnemonickeytmxextract(arg);
        fIsFieldIgnoremnemonickeytmxextractProcessed = true;
    }

    /**
     * Antタスクの[ignoremnemonickeytmxextract]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 7<br>
     * ニーモニック・キーを無視した内容の TMX を抽出するかどうか。日本風のニーモニックのみ対応。<br>
     * デフォルト値[false]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public boolean getIgnoremnemonickeytmxextract() {
        return fInput.getIgnoremnemonickeytmxextract();
    }

    /**
     * Antタスクの[transsourcelang]アトリビュートのセッターメソッド。
     *
     * 項目番号: 8<br>
     * 翻訳元言語。TMX ヘッダーに使われます。<br>
     *
     * @param arg セットしたい値
     */
    public void setTranssourcelang(final String arg) {
        fInput.setTranssourcelang(arg);
        fIsFieldTranssourcelangProcessed = true;
    }

    /**
     * Antタスクの[transsourcelang]アトリビュートのゲッターメソッド。
     *
     * 項目番号: 8<br>
     * 翻訳元言語。TMX ヘッダーに使われます。<br>
     * デフォルト値[en-US]が設定されています。Apache Antタスク上でアトリビュートの指定が無い場合には、デフォルト値が設定されます。<br>
     *
     * @return このフィールドの値
     */
    public String getTranssourcelang() {
        return fInput.getTranssourcelang();
    }

    /**
     * Antタスクのメイン処理。Apache Antから このメソッドが呼び出されます。
     *
     * @throws BuildException タスクとしての例外が発生した場合。
     */
    @Override
    public final void execute() throws BuildException {
        System.out.println("BentenExtractTmxFragmentTask begin.");

        // 項目番号[1]、アトリビュート[xliffdir]は必須入力です。入力チェックを行います。
        if (fIsFieldXliffdirProcessed == false) {
            throw new BuildException("必須アトリビュート[xliffdir]が設定されていません。処理を中断します。");
        }
        // 項目番号[2]、アトリビュート[tmxdir]は必須入力です。入力チェックを行います。
        if (fIsFieldTmxdirProcessed == false) {
            throw new BuildException("必須アトリビュート[tmxdir]が設定されていません。処理を中断します。");
        }
        // 項目番号[3]、アトリビュート[transtargetid]は必須入力です。入力チェックを行います。
        if (fIsFieldTranstargetidProcessed == false) {
            throw new BuildException("必須アトリビュート[transtargetid]が設定されていません。処理を中断します。");
        }

        if (getVerbose()) {
            System.out.println("- verbose:[true]");
            System.out.println("- xliffdir:[" + getXliffdir() + "]");
            System.out.println("- tmxdir:[" + getTmxdir() + "]");
            System.out.println("- transtargetid:[" + getTranstargetid() + "]");
            System.out.println("- extractbystate:[" + getExtractbystate() + "]");
            System.out.println("- includedate:[" + getIncludedate() + "]");
            System.out.println("- ignorewhitespacetmxextract:[" + getIgnorewhitespacetmxextract() + "]");
            System.out.println("- ignoremnemonickeytmxextract:[" + getIgnoremnemonickeytmxextract() + "]");
            System.out.println("- transsourcelang:[" + getTranssourcelang() + "]");
        }

        try {
            // 実際のAntタスクの主処理を実行します。
            // この箇所でコンパイルエラーが発生する場合、BentenExtractTmxFragmentProcessインタフェースを実装して benten.twa.tmx.coreパッケージに BentenExtractTmxFragmentProcessImplクラスを作成することにより解決できる場合があります。
            final BentenExtractTmxFragmentProcess proc = new BentenExtractTmxFragmentProcessImpl();
            if (proc.execute(fInput) != BentenExtractTmxFragmentBatchProcess.END_SUCCESS) {
                throw new BuildException("タスクは異常終了しました。");
            }
        } catch (IllegalArgumentException e) {
            if (getVerbose()) {
                e.printStackTrace();
            }
            throw new BuildException(e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中に例外が発生しました。処理を中断します。" + e.toString());
        } catch (Error e) {
            e.printStackTrace();
            throw new BuildException("タスクを処理中にエラーが発生しました。処理を中断します。" + e.toString());
        }
    }
}
