/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.core.model;

/**
 * ヘルプ向け XLIFF trans-unit 要素の id 属性を表すクラス。
 * @author KASHIHARA Shinji
 */
public class HelpTransUnitId {

	private static final String PREFIX = "benten://"; //$NON-NLS-1$
	private static final String SEQ_ATTRIBUTE = "?seq="; //$NON-NLS-1$
	private static final String END_MARK = "END"; //$NON-NLS-1$

	/** パス。 */
	private final String fPath;
	/** 連番。 */
	private int fSeq = 1;
	/** サブ連番。 */
	private int fSubSeq;
	/** 最大サブ連番。 */
	private int fMaxSubSeq;

	/**
	 * コンストラクター。
	 * @param path パス
	 */
	public HelpTransUnitId(final String path) {
		fPath = path.replace('\\', '/').replaceFirst("^/", ""); //$NON-NLS-1$ //$NON-NLS-2$
	}

	/**
	 * seq のインクリメント。
	 * @return このインスタンス
	 */
	public HelpTransUnitId incrementSeq() {
		fSeq++;
		fSubSeq = 0;
		return this;
	}

	/**
	 * seq の枝番をインクリメント。
	 * @param maxSubSeq 枝番の最大値
	 * @return このインスタンス
	 */
	public HelpTransUnitId incrementSubSeq(final int maxSubSeq) {
		fSubSeq++;
		fMaxSubSeq = maxSubSeq;
		return this;
	}

	/**
	 * このインスタンスの文字列表現を取得。
	 *
	 * <UL>
	 * <LI>一般的に toString メソッドはデバッグ用途で使用されますが、このメソッドはヘルプ向け
	 * trans-unit の id 文字列表現を取得できることが保証されています。
	 * </UL>
	 */
	@Override
	public String toString() {
		final StringBuilder sb = new StringBuilder();
		sb.append(PREFIX);
		sb.append(fPath);
		sb.append(SEQ_ATTRIBUTE);
		sb.append(fSeq);
		if (fSubSeq != 0) {
			sb.append("-"); //$NON-NLS-1$
			sb.append((fSubSeq == fMaxSubSeq) ? END_MARK : String.valueOf(fSubSeq));
		}
		return sb.toString();
	}

	/**
	 * 「?seq=」を除いた文字列表現を取得。
	 * 
	 * @return 「?seq=」を除いた文字列表現。
	 */
	public String toStringWithoutSeq() {
		final StringBuilder sb = new StringBuilder();
		sb.append(PREFIX);
		sb.append(fPath);
		return sb.toString();
	}

	/**
	 * 指定された id に続きがあるか判定。
	 * @param id id
	 * @return 続きがある場合は true
	 */
	public static boolean hasContinue(final String id) {
		return id.startsWith(PREFIX) && id.contains(SEQ_ATTRIBUTE) && id.matches(".+\\d+-\\d+"); //$NON-NLS-1$
	}
}
