/*******************************************************************************
 * Copyright (c) 2009 Information-technology Promotion Agency, Japan.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
package benten.twa.xliff.core;

import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.util.Date;

import benten.core.io.Files;
import benten.core.io.XliffFileFilter;
import benten.core.model.BentenXliff;
import benten.twa.io.BentenTwaProcessUtil;
import benten.twa.process.BentenProcessResultInfo;
import benten.twa.xliff.core.valueobject.BentenExtractXliffFragmentProcessInput;
import benten.twa.xliff.messages.BentenExtractXliffFragmentMessages;
import blanco.xliff.valueobject.BlancoXliffFile;
import blanco.xliff.valueobject.BlancoXliffTransUnit;

/**
 * XLIFF フラグメントの抽出
 *
 * <pre>
 * XLIFF から XLIFF フラグメントを抽出します。
 *   1.  XLIFF から所定の条件に合う翻訳単位を抽出して XLIFF フラグメントを作成します。
 * </pre>
 *
 * ★基本設計「翻訳ワークフロー支援機能: 翻訳中間形式断片抽出・マージ機能: XLIFFフラグメント抽出機能」に対応します。
 *
 * @author KASHIHARA Shinji
 */
public class BentenExtractXliffFragmentProcessImpl implements BentenExtractXliffFragmentProcess {
	/**
	 * XLIFF フラグメント抽出機能のためのメッセージ。
	 */
	protected static final BentenExtractXliffFragmentMessages fMsg = new BentenExtractXliffFragmentMessages();

	/**
	 * この処理の入力オブジェクト。
	 */
	protected BentenExtractXliffFragmentProcessInput fInput = null;

	/** 
	 * ファイル名のサフィックスに利用する今日の日付。
	 * <UL>
	 * <LI>この文字列は、ファイル名のサフィックスの用途のみで利用します。
	 * <LI>この文字列は、国際化(i18n)の必要はありません。
	 * </UL>
	 */
	private final String nowDate = new SimpleDateFormat("yyyyMMdd").format(new Date()); //$NON-NLS-1$

	/**
	 * この処理の実行結果情報。
	 */
	protected BentenProcessResultInfo fResultInfo = new BentenProcessResultInfo();

	/**
	 * この処理の実行結果情報を取得します。
	 *
	 * @return 処理結果情報。
	 */
	public BentenProcessResultInfo getResultInfo() {
		return fResultInfo;
	}

	/**
	 * {@inheritDoc}
	 */
	public int execute(final BentenExtractXliffFragmentProcessInput input) throws IOException, IllegalArgumentException {
		if (input == null) {
			throw new IllegalArgumentException(
					"BentenExtractXliffFragmentProcessImpl#execute: argument 'input' is null."); //$NON-NLS-1$
		}
		fInput = input;

		if (progress(fMsg.getCoreP001())) {
			return 6;
		}

		final File fileSourcefile = new File(input.getSourcexlifffile());
		if (fileSourcefile.exists() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE021(fInput.getSourcexlifffile()));
		}
		if (fileSourcefile.isFile() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE022(fInput.getSourcexlifffile()));
		}
		if (fileSourcefile.canRead() == false) {
			throw new IllegalArgumentException(fMsg.getCoreE023(fInput.getSourcexlifffile()));
		}

		final File fileTargetdir = new File(input.getTargetxliffdir());
		if (fileTargetdir.exists() == false) {
			if (fileTargetdir.mkdirs() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE024(fInput.getTargetxliffdir()));
			}
		} else {
			if (fileTargetdir.isDirectory() == false) {
				throw new IllegalArgumentException(fMsg.getCoreE025(fInput.getTargetxliffdir()));
			}
		}

		if (progress(fMsg.getCoreP011(input.getSourcexlifffile()))) {
			return 6;
		}

		BentenXliff inXliff = null;
		try {
			inXliff = BentenXliff.loadInstance(fileSourcefile);
		} catch (final IllegalArgumentException e) {
			throw new IOException(fMsg.getCoreE011(input.getSourcexlifffile(), e.toString()));
		}

		int outXliffIndex = 0;

		BentenXliff outXliff = BentenXliff.newInstance(createFile(input, ++outXliffIndex));

		int transUnitCount = 0;
		int transUnitTotalCount = 0;
		for (final BlancoXliffFile inFile : inXliff.getFileList()) {
			transUnitTotalCount += inFile.getBody().getTransUnitList().size();

			BlancoXliffFile outFile = createBlancoXliffFile(outXliff, inFile);

			for (final BlancoXliffTransUnit inUnit : inFile.getBody().getTransUnitList()) {
				if (progress(fMsg.getCoreP101(input.getSourcexlifffile(), BigDecimal.valueOf(++transUnitCount),
						BigDecimal.valueOf(transUnitTotalCount)))) {
					break;
				}

				// state のチェック
				if (input.getExtractbystate() != null) {
					String stateString = null;
					if (inUnit.getTarget() != null) {
						if (inUnit.getTarget().getState() != null) {
							stateString = inUnit.getTarget().getState();
						}
					}

					if (input.getExtractbystate().trim().length() == 0) {
						// state 無しを探したい。
						if (stateString == null || stateString.trim().length() == 0) {
							// これが出力対象。
						} else {
							// 出力対象外。
							continue;
						}
					} else {
						// 通常の state 指定。
						if (input.getExtractbystate().equals(stateString)) {
							// state が一致します。処理対象です。
						} else {
							// state が異なるので処理対象外とします。
							continue;
						}
					}
				}

				fResultInfo.setUnitCount(fResultInfo.getUnitCount() + 1);

				outFile.getBody().getTransUnitList().add(inUnit);
				if (input.getLimitsize() > 0 && outXliff.getAllTransUnitList().size() == input.getLimitsize()) {
					if (outXliff.save()) {
						fResultInfo.setSuccessCount(fResultInfo.getSuccessCount() + 1);
					} else {
						fResultInfo.setSkipCount(fResultInfo.getSkipCount() + 1);
					}

					outXliff = BentenXliff.newInstance(createFile(input, ++outXliffIndex));
					outFile = createBlancoXliffFile(outXliff, inFile);
				}
			}
		}

		if (progress(fMsg.getCoreP012(input.getTargetxliffdir()))) {
			return 0;
		}

		if (outXliff.getAllTransUnitList().size() > 0) {
			if (outXliff.save()) {
				fResultInfo.setSuccessCount(fResultInfo.getSuccessCount() + 1);
			} else {
				fResultInfo.setSkipCount(fResultInfo.getSkipCount() + 1);
			}
		}

		if (progress(fMsg.getCoreP003(BentenTwaProcessUtil.getResultMessage(fResultInfo)))) {
			return 6;
		}

		return 0;
	}

	/**
	 * {@inheritDoc}
	 */
	public boolean progress(final String argProgressMessage) {
		if (fInput != null && fInput.getVerbose()) {
			System.out.println(argProgressMessage);
		}
		return false;
	}

	/**
	 * ファイルの作成。
	 * 
	 * <UL>
	 * <LI>出力ファイル・オブジェクトを作成します。
	 * <LI>実際のファイル出力は、このメソッドの利用者がおこないます。
	 * <LI>フルで条件が重なった場合には、ファイル名は「入力ファイルの基準名-日付-連番.xlf」となります。
	 * </UL>
	 * 
	 * @param input 入力パラメーター
	 * @param fileIndex ファイル・インデックス
	 * @return ファイル
	 */
	protected File createFile(final BentenExtractXliffFragmentProcessInput input, final int fileIndex) {
		final StringBuilder name = new StringBuilder();
		name.append(Files.baseName(new File(input.getSourcexlifffile())));
		// 日付-連番 の順でサフィックスを付与します。
		if (input.getIncludedate()) {
			name.append("-" + nowDate); //$NON-NLS-1$
		}
		if (input.getLimitsize() > 0) {
			name.append("-" + fileIndex); //$NON-NLS-1$
		}
		return XliffFileFilter.appendExtension(new File(input.getTargetxliffdir(), name.toString()));
	}

	/**
	 * blancoXliffFile の作成。
	 * @param outXliff 出力 XLIFF
	 * @param inFile 入力 XLIFF ファイル
	 * @return XLIFF ファイル
	 */
	protected BlancoXliffFile createBlancoXliffFile(final BentenXliff outXliff, final BlancoXliffFile inFile) {
		final BlancoXliffFile outFile = outXliff.createBlancoXliffFile();
		outFile.setSourceLanguage(inFile.getSourceLanguage());
		outFile.setTargetLanguage(inFile.getTargetLanguage());
		outFile.setDatatype(inFile.getDatatype());
		outFile.setHeader(inFile.getHeader());
		return outFile;
	}
}
