/*******************************************************************************
 * Copyright (c) 2010 IGA Tosiki.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *******************************************************************************/
/*
 * Copyright (C) 2010 IGA Tosiki.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */
package benten.twa.filter.core;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.List;

/**
 * PO ファイルを単に読み込むだけのクラスです。 文字列リテラルのエスケープ処理すら行ないません。
 * 
 * @author Tosiki Iga
 * @see "http://www.gnu.org/software/hello/manual/gettext/PO-Files.html"
 */
public class SimplePOReader {
	/**
	 * 処理の結果得られたエントリーのリスト。
	 */
	List<SimplePOEntry> entryList = new ArrayList<SimplePOEntry>();

	/**
	 * 現在処理中の文字列リテラルのリスト。
	 */
	final List<String> stringLiteralList = new ArrayList<String>();
	/**
	 * 現在処理中のコメントのリスト。
	 */
	final List<String> commentList = new ArrayList<String>();

	/**
	 * PO 読み込みのエントリーポイント。
	 * @param inStream 入力ストリーム。
	 * @return PO のエントリー・リスト。
	 * @throws IOException 入出力例外が発生した場合。
	 */
	List<SimplePOEntry> process(final InputStream inStream) throws IOException {
		final BufferedReader reader = new BufferedReader(new InputStreamReader(inStream, "UTF-8"));
		process(reader);
		reader.close();

		return entryList;
	}

	void process(final Reader reader) throws IOException {
		String nextCommand = null;
		boolean isNoMeanLine = true;
		for (;;) {
			reader.mark(1);
			final int readNext = reader.read();
			if (readNext < 0) {
				if (nextCommand == null) {
					rememberCommand("x-benten-whitespace");
				} else {
					rememberCommand(nextCommand);
				}
				// メソッドの終了。
				return;
			}

			final char readChar = (char) readNext;
			switch (readChar) {
			case ' ':
				// 無視。
				break;
			case '\n':
				if (isNoMeanLine) {
					if (nextCommand != null) {
						rememberCommand(nextCommand);
						nextCommand = null;
					}
					rememberCommand("x-benten-whitespace");
				}
				isNoMeanLine = true;
				break;
			case '\r':
			case '\t':
				break;
			case '#':
				if (nextCommand != null) {
					rememberCommand(nextCommand);
					nextCommand = null;
				}

				// コメントの処理を行ないます。
				commentList.add(processComment(reader));
				isNoMeanLine = false;
				break;
			case '"':
				stringLiteralList.add(processStringLiteral(reader));
				isNoMeanLine = false;
				break;
			default:
				if (nextCommand != null) {
					rememberCommand(nextCommand);
					nextCommand = null;
				}

				reader.reset();
				final String result = processCommand(reader);
				if (result != null) {
					nextCommand = result;
				}
				break;
			}
		}
	}

	String processComment(final Reader reader) throws IOException {
		final StringBuilder strbuf = new StringBuilder();
		outerloop: for (;;) {
			reader.mark(1);
			final int readNext = reader.read();
			if (readNext < 0) {
				reader.reset();
				break outerloop;
			}

			final char readChar = (char) readNext;
			switch (readChar) {
			case '\n':
			case '\r':
				// 行が終わりました。
				// PO ファイルでは改行でコメントが終了します。
				reader.reset();
				break outerloop;
			}
			strbuf.append(readChar);
		}
		return strbuf.toString();
	}

	String processStringLiteral(final Reader reader) throws IOException {
		final StringBuilder strbuf = new StringBuilder();
		boolean isInEscape = false;
		outerloop: for (;;) {
			reader.mark(1);
			final int readNext = reader.read();
			if (readNext < 0) {
				reader.reset();
				break outerloop;
			}

			final char readChar = (char) readNext;
			switch (readChar) {
			case '"':
				if (isInEscape == false) {
					// 文字列の終了。
					break outerloop;
				} else {
					isInEscape = false;
					// 前のエスケープを出力。
					strbuf.append("\\");
					// 普通に処理。
				}
				break;
			case '\\':
				if (isInEscape == false) {
					// エスケープ開始。
					isInEscape = true;
					// ここでコンティニュー。
					continue;
				} else {
					isInEscape = false;
					// エスケープ中。
					// 前のエスケープを出力。
					strbuf.append("\\");
				}
				break;
			case '\n':
			case '\r':
				reader.reset();
				if (isInEscape == false) {
					// 行が終わりました。
					throw new IllegalArgumentException("想定しない場所で改行が入りました。不正な文字列です。");
				} else {
					isInEscape = false;
					// 文字列の終了。
					break outerloop;
				}
			default:
				if (isInEscape == false) {
				} else {
					isInEscape = false;
					// 前のエスケープを出力。
					strbuf.append("\\");
				}
				break;
			}
			strbuf.append(readChar);
		}

		return strbuf.toString();
	}

	String processCommand(final Reader reader) throws IOException {
		final StringBuilder strbuf = new StringBuilder();
		outerloop: for (;;) {
			reader.mark(1);
			final int readNext = reader.read();
			if (readNext < 0) {
				reader.reset();
				break outerloop;
			}

			final char readChar = (char) readNext;
			switch (readChar) {
			case ' ':
				break outerloop;
			case '\n':
			case '\r':
				// 行が終わりました。
				// PO ファイルでは改行でコマンドが終了します。
				reader.reset();
				break outerloop;
			}
			strbuf.append(readChar);
		}

		final String result = strbuf.toString();
		if (result.trim().length() == 0) {
			return null;
		}

		return result.trim();
	}

	/**
	 * コマンドを記憶します。
	 * 
	 * @param command
	 *            コマンド。
	 */
	void rememberCommand(final String command) {
		SimplePOEntry currentEntry = new SimplePOEntry();
		entryList.add(currentEntry);

		currentEntry.fCommand = command;
		currentEntry.fCommentList.addAll(commentList);
		currentEntry.fStringLiteralList.addAll(stringLiteralList);

		// コマンドが開始したので保留していたリテラルは破棄します。
		stringLiteralList.clear();

		// コメントも破棄します。
		commentList.clear();
	}
}
