/*******************************************************************************
 benten.cat.tm.engine.omegat - Benten default TM
                               (Translation Memory) engine.
 Copyright (C) 2009 Information-technology Promotion Agency, Japan.

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, see <http://www.gnu.org/licenses>.
 *******************************************************************************/
package benten.cat.tm.engine.omegat;

import java.io.File;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.omegat.core.data.LegacyTM;
import org.omegat.core.data.StringEntry;
import org.omegat.core.data.StringEntryWrapper;
import org.omegat.core.matching.FuzzyMatcherWrapper;
import org.omegat.core.matching.NearString;
import org.omegat.util.Language;
import org.omegat.util.TMXReader;

import benten.cat.tm.core.BentenTmEngine;
import benten.cat.tm.core.BentenTmSearchResult;
import benten.cat.tm.engine.omegat.messages.DefaultTmEngineMessages;
import benten.core.io.Files;

/**
 * デフォルトの Benten TM (翻訳メモリー) エンジン。
 *
 * ★基本設計「共通機能: 翻訳メモリー・エンジン機能」に対応します。
 *
 * @author YAMAMOTO Koji
 */
public class DefaultTmEngine implements BentenTmEngine {
	/**
	 * Benten デフォルト TM エンジン・プラグインのためのメッセージ。
	 */
	protected static final DefaultTmEngineMessages fMsg = new DefaultTmEngineMessages();

	/**
	 * デバッグするかどうか。
	 *
	 * <UL>
	 * <LI>Benten 開発者向けのデバッグ・フラグです。
	 * <LI>Java 起動オプションで -Dbenten.debug=true を指定することにより有効となります。
	 * </UL>
	 */
	private static final boolean IS_DEBUG = Boolean.valueOf(System.getProperty("benten.debug")); //$NON-NLS-1$

	/**
	 * 翻訳メモリー・データのリスト。
	 */
	private List<LegacyTM> fTmDataByLegacyTmList = Collections.synchronizedList(new ArrayList<LegacyTM>());

	/**
	 * {@inheritDoc}
	 */
	public synchronized void loadTmx(final File dirTmx) throws IOException {
		final File[] fileTmxs = dirTmx.listFiles();
		if (fileTmxs == null) {
			throw new IllegalArgumentException(fMsg.getMsg0(dirTmx.toString()));
		}

		for (int index = 0; index < fileTmxs.length; index++) {
			final File fileTmx = fileTmxs[index];
			if (fileTmx.isFile() == false) {
				continue;
			}
			if (fileTmx.getName().toLowerCase().endsWith(".tmx") == false) { //$NON-NLS-1$
				// ファイルの拡張子が .tmx のものだけ処理します。
				continue;
			}
			// TMXReader は、BOM 付きの UTF-8 または UTF-16 については、文字エンコーディングを自動判別します。
			// TMXReader の第 1 引数に指定した文字エンコーディングは、自動判別できなかった場合の
			// デフォルトとして使用されます。
			final TMXReader reader = new TMXReader("UTF-8", new Language("en-US"), new Language("ja-JP"), false); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
			try {
				reader.loadFile(fileTmx.getAbsolutePath(), true);
			} catch (Exception e) {
				throw new IllegalArgumentException(fMsg.getMsg9(fileTmx.getAbsolutePath(), e.getMessage()), e);
			}
			final LegacyTM legacyTm = loadLegacyTM(Files.baseName(fileTmx), reader);
			fTmDataByLegacyTmList.add(legacyTm);
		}
	}

	/**
	 * TMXReader を使って、TMX ファイルを LegacyTM に読み込みます。
	 *
	 * @param tmxFileName
	 *            reader にロードした TMX のファイル名。
	 * @param reader
	 *            TMX ファイルがロードされた TMXReader。
	 * @return TMX ファイルが読み込まれた LegacyTM
	 */
	private LegacyTM loadLegacyTM(final String tmxFileName, final TMXReader reader) {
		final int num = reader.numSegments();
		final List<StringEntry> strEntryList = new ArrayList<StringEntry>(num);
		for (int i = 0; i < num; i++) {
			final String src = reader.getSourceSegment(i);
			final String trans = reader.getTargetSegment(i);

			final StringEntryWrapper se = new StringEntryWrapper(src);
			se.setTranslation(trans);
			strEntryList.add(se);
		}

		return new LegacyTM(tmxFileName, strEntryList);
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized List<BentenTmSearchResult> fuzzySearch(final String target) {
		//	検索結果のリスト。
		final List<BentenTmSearchResult> nearList = new ArrayList<BentenTmSearchResult>();

		final FuzzyMatcherWrapper fuzzyMatcher = FuzzyMatcherWrapper.getInstance();
		synchronized (fuzzyMatcher) {
			// シングルトン・インスタンスを共用しているので、同期化して排他利用する必要があります。
			final StringBuffer targetStrBuf = new StringBuffer();
			if (target != null) {
				targetStrBuf.append(target);
			}

			final StringEntry tuEntry = new StringEntry(targetStrBuf.toString());
			final List<NearString> nearStringList = fuzzyMatcher.match(tuEntry, fTmDataByLegacyTmList);

			if (nearStringList.size() > 0) {
				if (IS_DEBUG) {
					System.out.println(fMsg.getMsg1(BigDecimal.valueOf(nearStringList.size())));
					System.out.println(fMsg.getMsg2() + tuEntry.getSrcText());
				}

				for (int i = 0; i < nearStringList.size(); i++) {
					final NearString nearString = nearStringList.get(i);
					final BentenTmSearchResult result = new BentenTmSearchResult();
					result.setMatchQuality(Integer.toString(nearString.score) + "% "); //$NON-NLS-1$
					result.setSource(nearString.str.getSrcText());
					result.setTarget(nearString.str.getTranslation());
					result.setOrigin(nearString.proj);
					nearList.add(result);

					if (IS_DEBUG) {
						System.out.println("------------------------------------"); //$NON-NLS-1$
						System.out.println(fMsg.getMsg3(BigDecimal.valueOf(i + 1)));
						System.out.println(fMsg.getMsg4(BigDecimal.valueOf(nearString.score)));
						System.out.println(nearString.str.getSrcText());
						System.out.println(nearString.str.getTranslation());
						System.out.println(nearString.proj);
					}
				}

				if (IS_DEBUG) {
					System.out.println("------------------------------------"); //$NON-NLS-1$
					System.out.println(""); //$NON-NLS-1$
				}
			}
		}

		return nearList;
	}

	/**
	 * {@inheritDoc}
	 */
	public synchronized void unload() {
		fTmDataByLegacyTmList.clear();
	}
}
