/**
 * Copyright (C) 2023 awk4j - https://ja.osdn.net/projects/awk4j/
 * <p>
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * <p>
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * <p>
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
use crate::Path;
use std::fs::File;
use std::io::{Read, Write};
// use std::io::prelude::*;
use std::sync::mpsc;
use std::thread;

use crate::iomod::get_meta_len;
use crate::iomod::path_to_unix;

// #[derive(Debug, Clone)] // I/O buffer
struct IOBUF {
    buf: Vec<u8>, // this field does not implement `Copy`
    length: usize,
}
impl IOBUF {
    fn to_slice(&self) -> &[u8] {
        &self.buf[..self.length]
    }
}
/**
 * copy maxbuf from to -> length
 * read/write buffer: 1MB, スループット: In + Out
 */
pub fn copymax<P: AsRef<Path>>(from: P, to: P) -> u64 {
    const MAXBUFSIZE: usize = 1024 * 1024; // 1MB
    let f: &Path = from.as_ref();
    let t: &Path = to.as_ref();
    let mut fr = File::open(f).expect(&path_to_unix(f));
    let mut fw = File::create(t).expect(&path_to_unix(t));
    let mut length: usize = 0;
    let mut io = IOBUF {
        buf: vec![0_u8; MAXBUFSIZE],
        length: 0,
    };
    loop {
        io.length = fr.read(&mut io.buf).unwrap();
        if io.length == 0 {
            break;
        }
        let slice: &[u8] = io.to_slice();
        length += fw.write(slice).unwrap();
    }
    let _ = fw.flush();
    let fromsize: usize = get_meta_len(f) as usize;
    assert_eq!(fromsize, length, "(original:result) ");
    length as u64
}

/**
 * copy channel from | to -> length
 * channel buffer: 512k, スループット: max(In, Out)
 */
pub fn copych<P: AsRef<Path>>(from: P, to: P) -> u64 {
    const CHBUFSIZE: usize = 1024 * 1024;
    // const CHBUFSIZE: usize = 1024 * 512;
    let f: &Path = from.as_ref();
    let t: &Path = to.as_ref();
    let mut fr = File::open(f).expect(&path_to_unix(f));
    let mut fw = File::create(t).expect(&path_to_unix(t));
    let (tx, rx) = mpsc::channel();
    thread::spawn(move || {
        loop {
            let mut ch = IOBUF {
                buf: vec![0_u8; CHBUFSIZE],
                length: 0,
            };
            ch.length = fr.read(&mut ch.buf).unwrap(); // ひたすら読み込む
            if ch.length == 0 {
                break;
            }
            tx.send(ch).unwrap(); // 送信
        }
    });
    let mut length: usize = 0; // 受信
    for received in rx {
        let slice: &[u8] = received.to_slice();
        length += fw.write(slice).unwrap(); // ひたすら書き出す
    }
    let _ = fw.flush();
    let fromsize: usize = get_meta_len(f) as usize;
    assert_eq!(fromsize, length, "(original:result) ");
    length as u64
}
