/*
 * Copyright (c) 2003 SOFTWARE AG, All Rights Reserved.
 */


/*
 * InsertUnique.java
 */
package com.softwareag.tamino.db.api.examples.jazz;

import com.softwareag.tamino.db.api.accessor.TAccessLocation;
import com.softwareag.tamino.db.api.accessor.TAccessorException;
import com.softwareag.tamino.db.api.accessor.TInsertException;
import com.softwareag.tamino.db.api.accessor.TQuery;
import com.softwareag.tamino.db.api.accessor.TQueryException;
import com.softwareag.tamino.db.api.accessor.TXMLObjectAccessor;
import com.softwareag.tamino.db.api.common.TAccessFailureMessage;
import com.softwareag.tamino.db.api.common.TException;
import com.softwareag.tamino.db.api.connection.TConnection;
import com.softwareag.tamino.db.api.connection.TConnectionException;
import com.softwareag.tamino.db.api.connection.TConnectionFactory;
import com.softwareag.tamino.db.api.connection.TIsolationLevel;
import com.softwareag.tamino.db.api.connection.TLocalTransaction;
import com.softwareag.tamino.db.api.objectModel.TXMLObject;
import com.softwareag.tamino.db.api.objectModel.dom.TDOMObjectModel;
import com.softwareag.tamino.db.api.response.TResponse;
import java.io.FileInputStream;
import org.w3c.dom.CharacterData;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;


public class InsertUnique {
	
	/** Creates new InsertUnique */
	public InsertUnique (String databaseURI,String collection) throws TConnectionException {
		
		// Obtain the connection factory
		TConnectionFactory connectionFactory = TConnectionFactory.getInstance();
		// Obtain the first connection to the database
		connectionA = connectionFactory.newConnection( databaseURI );
		// Obtain the concrete TXMLObjectAccessor using the DOM object model
		accessorA = connectionA.newXMLObjectAccessor( TAccessLocation.newInstance( collection ) ,
													 TDOMObjectModel.getInstance() );
		// Set local transaction mode to "shared"
		connectionA.setIsolationLevel(TIsolationLevel.SHARED) ;
		
		// Obtain the second connection to the database
		connectionB = connectionFactory.newConnection( databaseURI );
		// Obtain the second accessor
		accessorB = connectionB.newXMLObjectAccessor( TAccessLocation.newInstance( collection ) ,
													 TDOMObjectModel.getInstance() );
		// Set isolation level of connection 2 to "unprotected"
		connectionB.setIsolationLevel(TIsolationLevel.UNPROTECTED) ;
	}
	
	private void performInsert(TXMLObject xmlObject) throws Exception  {
		try  {
			// Invoke the insert operation
			accessorA.insert( xmlObject );
		}
		catch (TInsertException insertException)  {
			showAccessFailure( insertException );
		}
	}
	
	// get number of matching documents
	private int getCount(String path) throws Exception  {
		try  {
			// Construct TQuery object
			TQuery query = TQuery.newInstance("count("+path+")");
			// perform the query
			TResponse response = accessorB.query(query );
			// get the number of documents found
			String s = response.getQueryContentAsString();
			// convert to integer
			return Integer.valueOf(s).intValue();
		}
		catch (TQueryException queryException)  {
			showAccessFailure( queryException );
			return 0;
		}
	}
	
	// Show the reason for the access failure
	private void showAccessFailure(TAccessorException accessorException) throws Exception {
		// Obtain an access failure message telling the exact reason if Tamino request failed
		TAccessFailureMessage accessFailure = accessorException.getAccessFailureException();
		if ( accessFailure != null )
			throw new Exception( "Access failed:" + accessFailure );
		else
			throw new Exception( "Access failed:" + accessorException.getMessage() );
	}
	
	// get text content from element
	private String getText(Element element) {
		return ((CharacterData) element.getFirstChild()).getData();
	}
	
	private void processTransaction(String filename, String key) throws Exception {
		TLocalTransaction myTransaction = null;
		try  {
			// Read file into a DOM Tamino XML object
			// Instantiate an empty TXMLObject instance using the DOM object model
			TXMLObject xmlObject = TXMLObject.newInstance( TDOMObjectModel.getInstance() );
			// Establish the DOM representation by reading the content from a file input stream
			xmlObject.readFrom( new FileInputStream(filename ));
			// get DOM document
			Document doc = (Document) xmlObject.getDocument();
			// get top level element
			Element root = (Element) xmlObject.getElement();
			// get key value
			String keyValue = null;
			// check if key is an attribute or an element
			if (key.startsWith("@")) {
				// get attribute value
				keyValue = root.getAttribute(key.substring(1));
			} else {
				// get element node list
				NodeList nl = doc.getElementsByTagName(key);
				if (nl.getLength() == 0) throw new Exception("Key not found");
				// get only element
				Element elem = (Element) nl.item(0);
				// get element content
				keyValue = getText(elem);
			}
			// Check for proper content
			if (keyValue == "") throw new Exception("Key not found");
			// Start the transaction
			myTransaction = connectionA.useLocalTransactionMode();
			// Insert the document
			performInsert( xmlObject );
			// Get number of matching documents
			int c = getCount( xmlObject.getDoctype() + "["+key+"='" + keyValue + "']" );
			if (c == 1) {
				// Unique - commit the transaction
				myTransaction.commit();
				System.out.println("Transaction committed");
			} else {
				// Bad - roll back the transaction
				myTransaction.rollback();
				throw new Exception("Key not unique: " + c + " occurrences. Transaction aborted.");
			}
		}
		catch (DOMException domException)  {
			if (myTransaction != null) myTransaction.rollback();
			domException.printStackTrace();
		}
		catch (TException taminoException)  {
			taminoException.printStackTrace();
		}
		finally  {
			// Close both connections
			connectionA.close();
			connectionB.close();
		}
	}
	
	/**
	 * @param args the command line arguments
	 */
	public static void main(String[] args) throws Exception  {
		InsertUnique insertUnique = new InsertUnique( DATABASE_URI , COLLECTION );
		insertUnique.processTransaction(args[0],args[1]);
	}
	
	// URI of the Tamino database
	private final static String DATABASE_URI = "http://localhost/tamino/jazz";
	
	// Constant for the collection
	private final static String COLLECTION = "encyclopedia";
	
	// Database connection A
	private TConnection connectionA = null;
	
	// Accessor A
	private TXMLObjectAccessor accessorA = null;
	
	// Database connection B
	private TConnection connectionB = null;
	
	// Accessor B
	private TXMLObjectAccessor accessorB = null;
}

