/*
 * Copyright (c) 2003 SOFTWARE AG, All Rights Reserved.
 */


/*
 * InsertConstraintCheck.java
 */
package com.softwareag.tamino.db.api.examples.jazz;

import com.softwareag.tamino.db.api.accessor.TAccessLocation;
import com.softwareag.tamino.db.api.accessor.TAccessorException;
import com.softwareag.tamino.db.api.accessor.TInsertException;
import com.softwareag.tamino.db.api.accessor.TQuery;
import com.softwareag.tamino.db.api.accessor.TQueryException;
import com.softwareag.tamino.db.api.accessor.TXMLObjectAccessor;
import com.softwareag.tamino.db.api.common.TAccessFailureMessage;
import com.softwareag.tamino.db.api.common.TException;
import com.softwareag.tamino.db.api.connection.TConnection;
import com.softwareag.tamino.db.api.connection.TConnectionException;
import com.softwareag.tamino.db.api.connection.TConnectionFactory;
import com.softwareag.tamino.db.api.connection.TIsolationLevel;
import com.softwareag.tamino.db.api.connection.TLocalTransaction;
import com.softwareag.tamino.db.api.objectModel.TXMLObject;
import com.softwareag.tamino.db.api.objectModel.dom.TDOMObjectModel;
import com.softwareag.tamino.db.api.response.TResponse;
import java.io.FileInputStream;
import java.text.SimpleDateFormat;
import java.util.Date;
import org.w3c.dom.CharacterData;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;


public class InsertConstraintCheck {
	
	/** Creates new InsertUnique */
	public InsertConstraintCheck (String databaseURI,String collection) throws TConnectionException {
		// Obtain the connection factory
		TConnectionFactory connectionFactory = TConnectionFactory.getInstance();
		// Obtain the connection to the database
		connection = connectionFactory.newConnection( databaseURI );
		// Obtain the concrete TXMLObjectAccessor using the DOM object model
		accessor = connection.newXMLObjectAccessor( TAccessLocation.newInstance( collection ) ,
												   TDOMObjectModel.getInstance() );
		// Set isolation level to "shared"
		connection.setIsolationLevel(TIsolationLevel.SHARED) ;
	}
	
	private void performInsert(TXMLObject xmlObject) throws Exception  {
		try  {
			// Invoke the insert operation
			accessor.insert( xmlObject );
		}
		catch (TInsertException insertException)  {
			showAccessFailure( insertException );
		}
	}
	
	// process query
	private TResponse processQuery(String s) throws Exception {
		TQuery query = TQuery.newInstance( s );
		try {
			// Invoke the query operation
			return accessor.query( query );
		}
		catch (TQueryException queryException) {
			// Inform about the reason for the failure
			showAccessFailure(queryException);
			return null;
		}
	}
	
	// Show the reason for the access failure
	private void showAccessFailure(TAccessorException accessorException) throws Exception {
		// Obtain an access failure message telling the exact reason if Tamino request failed
		TAccessFailureMessage accessFailure = accessorException.getAccessFailureException();
		if ( accessFailure != null )
			throw new Exception( "Access failed:" + accessFailure );
		else
			throw new Exception( "Access failed:" + accessorException.getMessage() );
	}
	
	// Convert into date format
	private Date toDate(String s) throws java.text.ParseException {
		// determine formatting string
		String f = (s.indexOf("T",0) >= 0 ?
					"yyyy-MM-dd'T'HH:mm:ss" :
						(s.indexOf(":", 1) >= 0 ?
						 "HH:mm:ss" : "yyyy-MM-dd"));
		// check for explicit time zone
		int p = Math.max (s.indexOf("+", 8),s.indexOf("-", 8));
		if (p >= 0) {
			f += "z"; // indicate time zone in formatting string
			s = s.substring(0,p-1)+"GMT"+s.substring(p);   // keep SimpleDateFormat happy
		}
		else if (s.charAt(s.length()-1) == 'Z') {
			// check for UTC time zone
			f += "z";
			s = s.substring(0,s.length()-1) + "UTC";
		}
		// create SimpleDateFormat object
		SimpleDateFormat df = new SimpleDateFormat ( f );
		// and use it as a parser
		return df.parse(s);
	}
	
	// get text content from element
	private String getText(Element element) {
		return ((CharacterData) element.getFirstChild()).getData();
	}
	
	// process transaction
	private void processTransaction(String filename)  throws Exception {
		TLocalTransaction myTransaction = null;
		boolean abortTransaction = false;
		try  {
			// Read file into a DOM Tamino XML object
			// Instantiate an empty TXMLObject instance using the DOM object model
			TXMLObject collaborationObject = TXMLObject.newInstance( TDOMObjectModel.getInstance() );
			// Establish the DOM representation by reading the content from a file input stream.
			collaborationObject.readFrom( new FileInputStream(filename ));
			// get DOM document
			Document collaborationDoc = (Document) collaborationObject.getDocument();
			// Set local transaction mode and get a transaction object
			myTransaction = connection.useLocalTransactionMode();
			// initialize start date
			Element startDateElement = null;
			// get "from" elements if defined
			NodeList fromList = collaborationDoc.getElementsByTagName("from");
			if (fromList.getLength() > 0) {
				// get the only child
				startDateElement = (Element) fromList.item(0);
			} else {
				// alternatively, get the "time" element
				startDateElement = (Element) collaborationDoc.getElementsByTagName("time").item(0);
			}
			// get start date value
			String startDateValue = getText(startDateElement);
			// convert to Date
			Date startDate = toDate(startDateValue);
			// Get jazzMusician elements
			NodeList collaborateurs = collaborationDoc.getElementsByTagName("jazzMusician");
			// now loop over the "jazzMusician" children
			for (int i=0; i < collaborateurs.getLength(); i++) {
				// get a single "jazzMusician" child
				Element collaborateurElement = (Element) collaborateurs.item(i);
				// check if this item has content
				if (collaborateurElement.hasChildNodes()) {
					// get the string content
					String collaborateurID = getText(collaborateurElement);
					// Perform query for jazzMusicians identified by collaborateurID
					TResponse response = processQuery("jazzMusician[@ID"+"='" + collaborateurID + "']");
					// Process the musician document if we have one
					if (!response.hasFirstXMLObject()) {
						abortTransaction = true;
						System.out.println("Error: Referenced jazzMusician " + collaborateurID +
										   " does not exist");
					} else {
						// get first (and only) result document
						TXMLObject jazzMusicianObject = response.getFirstXMLObject();
						// Get top level DOM element
						Document jazzMusicianDoc = (Document) jazzMusicianObject.getDocument();
						// get birthDate
						Element birthDateElement = (Element) jazzMusicianDoc.getElementsByTagName("birthDate").item(0);
						// get string value
						String birthDateValue = getText(birthDateElement);
						// convert to date
						Date birthDate = toDate(birthDateValue);
						// compare with startDate
						if (startDate.compareTo(birthDate) <= 0) {
							abortTransaction = true;
							// Report violation of integrity constraint
							System.out.println("Error: Collaboration start date before birth date of jazz musician " +
											   collaborateurID);
						}
					}
				}
			}
			if (abortTransaction) {
				myTransaction.rollback();
				// Report abort of operation
				System.out.println("Error: Insert not performed");
			} else {
				performInsert( collaborationObject );
				myTransaction.commit();
				// Show the collection, doctype and id
				System.out.println("Message: Insert succeeded, ino:collection=" +
								   collaborationObject.getCollection() + ", ino:doctype=" +
								   collaborationObject.getDoctype() +
								   ", ino:id=" + collaborationObject.getId() );
			}
		}
		catch (DOMException domException)  {
			if (myTransaction != null) myTransaction.rollback();
			domException.printStackTrace();
		}
		catch (TException taminoException)  {
			taminoException.printStackTrace();
		}
		finally  {
			// Close the connection
			connection.close();
		}
	}
	
	/**
	 * @param args the command line arguments
	 */
	
	public static void main(String[] args) throws Exception  {
		InsertConstraintCheck insertConstraintCheck = new InsertConstraintCheck( DATABASE_URI , COLLECTION );
		// perform the transaction
		insertConstraintCheck.processTransaction(args[0]);
	}
	
	// URI of the Tamino database
	private final static String DATABASE_URI = "http://localhost/tamino/jazz";
	
	// Constant for the collection
	private final static String COLLECTION = "encyclopedia";
	
	// The database connection
	private TConnection connection = null;
	
	// The accessor, here a high-level TXMLObjectAccessor
	private TXMLObjectAccessor accessor = null;
}

