////////////////////////////////////////////////////////////////
// aaoh.kaze
//   Copyright (C) 2008 - 2012 AAOh inc.
// 
// Licensed under the MIT license
//   http://www.opensource.org/licenses/mit-license.php
////////////////////////////////////////////////////////////////

/**
 * @fileOverview aaoh.kaze(アニメーション)
 * @author       $Author: fujishiro $
 */

/**
 * namespace。
 *   aaoh.kaze.animation.Animation
 *   aaoh.kaze.animation.FadeinAnimation
 *   aaoh.kaze.animation.FadeoutAnimation
 *   aaoh.kaze.animation.MoveAnimation
 *   aaoh.kaze.animation.ResizeAnimation
 *   aaoh.kaze.animation.ForegroundColorAnimation
 *   aaoh.kaze.animation.BackgroundColorAnimation
 *   aaoh.kaze.animation.BorderColorAnimation
 */

(function () {
	try {
		if (!aaoh.kaze.animation) {
			/**
			 * aaoh.kaze.animation。
			 *
			 * @namespace
			 */
			aaoh.kaze.animation = {};
		}
	} catch(ex) {
		aaoh.kaze.animation = {};
	}
})();

(function () {
/**
 * Animation。
 *
 * @constructor
 */
aaoh.kaze.animation.Animation = function() {
	this.callObj_       = null;
	this.element_       = null;
	this.speed_         = null;
	this.startValue_    = null;
	this.stopValue_     = null;
	this.stepValue_     = null;
	this.startCallable_ = null;
	this.stopCallable_  = null;
	this.stepCallable_  = null;

	this.flgInit_      = false;
	this.currentValue_ = null;
	this.timer_        = null;
}
aaoh.kaze.animation.Animation.prototype = {
	/**
	 * 初期処理。
	 *
	 * @param {Object}   callObj       呼び出し元オブジェクト
	 * @param {Object}   element       エレメント
	 * @param {Number}   speed         スピード(msec)
	 * @param {Object}   startValue    開始値
	 * @param {Object}   stopValue     終了値
	 * @param {Object}   stepValue     ステップ値
	 * @param {Function} startCallable 開始ファンクション
	 * @param {Function} stopCallable  終了ファンクション
	 * @param {Function} stepCallable  ステップファンクション
	 */
	init: function(
		callObj, element, speed,
		startValue, stopValue, stepValue,
		startCallable, stopCallable, stepCallable
	) {
		this.callObj_       = callObj;
		this.element_       = element;
		this.speed_         = speed;
		this.startValue_    = startValue;
		this.stopValue_     = stopValue;
		this.stepValue_     = stepValue;
		this.startCallable_ = startCallable;
		this.stopCallable_  = stopCallable;
		this.stepCallable_  = stepCallable;
		if (this.callObj_ == null) {
			this.callObj_ = this;
		}

		if (this.element_ == null) {
			throw new Error('element is null');

		} else if (isNaN(this.speed_)) {
			throw new Error('speed is null, element is ' + this.element_.id);

		} else if (this.startValue_ == null) {
			throw new Error(
				'startValue is null, '
				+ 'element is ' + this.element_.id + ', '
				+ 'speed is '   + this.speed_
			);

		} else if (this.stopValue_ == null) {
			throw new Error(
				'stopValue is null, '
				+ 'element is '    + this.element_.id + ', '
				+ 'speed is '      + this.speed_      + ', '
				+ 'startValue is ' + this.startValue_
			);

		} else if (this.stepValue_ == null) {
			throw new Error(
				'stepValue is null, '
				+ 'element is '    + this.element_.id + ', '
				+ 'speed is '      + this.speed_      + ', '
				+ 'startValue is ' + this.startValue_ + ', '
				+ 'stopValue is '  + this.stopValue_
			);
		}

		this.flgInit_ = true;
		this.timer_   = new aaoh.kaze.Timer(this);
	},

	/**
	 * アニメーションの開始。
	 */
	start: function() {
		if (!this.flgInit_) {
			this.init(
				arguments[0], arguments[1], arguments[2], arguments[3], arguments[4],
				arguments[5], arguments[6], arguments[7], arguments[8]
			);
		}
		this.currentValue_ = this.startValue_;

		if (this.startCallable_ != null) {
			this.startCallable_.call(
				this.callObj_, this.element_, this.startValue_
			);
		}

		this.timer_.setInterval(this.step, this.speed_);
	},

	/**
	 * アニメーションの停止。
	 */
	stop: function() {
		this.timer_.clearInterval();

		if (this.stopCallable_ != null) {
			this.stopCallable_.call(
				this.callObj_, this.element_, this.stopValue_
			);
		}

		this.currentValue_ = null;
	},

	/**
	 * アニメーションのステップ実行。
	 */
	step: function() {
		var nextValue = this.nextValue();
		if (nextValue == null) {
			this.stop();
			return;
		}

		if (this.stepCallable_ != null) {
			this.stepCallable_.call(
				this.callObj_, this.element_, nextValue
			);
		}
	},

	/**
	 * 次ステップ値の取得。
	 *
	 * @return {Object} 次ステップ値またはnullを返す
	 */
	nextValue: function() {
		if (isNaN(this.currentValue_)) {
			return null;
		}

		var retValue = this.currentValue_ + this.stepValue_;
		if (retValue >= this.stopValue_) {
			return null;
		}

		this.currentValue_ = retValue;
		return retValue;
	}
}
})();

(function () {
/**
 * FadeinAnimation。
 *
 * @constructor
 * @extends aaoh.kaze.animation.Animation
 * @param {Object} element エレメント
 * @param {Number} speed   スピード(msec)
 */
aaoh.kaze.animation.FadeinAnimation = aaoh.kaze.Gadget.extend(
	aaoh.kaze.animation.Animation, function(
		element, speed
	) {
	this.DEFAULT_SPEED       = 20;
	this.DEFAULT_START_VALUE = 0;
	this.DEFAULT_STOP_VALUE  = 100;
	this.DEFAULT_STEP_VALUE  = 10;

	if (speed == null) {
		speed = this.DEFAULT_SPEED;
	}

	// 初期化
	this.init(
		this, element, speed,
		this.DEFAULT_START_VALUE, this.DEFAULT_STOP_VALUE,
		this.DEFAULT_STEP_VALUE,
		this.startCallable, this.stopCallable, this.stepCallable
	);
});
	/**
	 * 開始処理。
	 *
	 * @param {Object} element    エレメント
	 * @param {Object} startValue 開始値
	 */
	aaoh.kaze.animation.FadeinAnimation.prototype.
	startCallable = function(element, startValue) {
		aaoh.kaze.Element.setOpacity(element, startValue);
	}

	/**
	 * 終了処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} stopValue 終了値
	 */
	aaoh.kaze.animation.FadeinAnimation.prototype.
	stopCallable = function(element, stopValue) {
		aaoh.kaze.Element.setOpacity(element, stopValue);
	}

	/**
	 * ステップ処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} nextValue 現在値
	 */
	aaoh.kaze.animation.FadeinAnimation.prototype.
	stepCallable = function(element, nextValue) {
		aaoh.kaze.Element.setOpacity(element, nextValue);
	}
})();

(function () {
/**
 * FadeoutAnimation。
 *
 * @constructor
 * @extends aaoh.kaze.animation.Animation
 * @param {Object} element エレメント
 * @param {Number} speed   スピード(msec)
 */
aaoh.kaze.animation.FadeoutAnimation = aaoh.kaze.Gadget.extend(
	aaoh.kaze.animation.Animation, function(
		element, speed
	) {
	this.DEFAULT_SPEED       = 20;
	this.DEFAULT_START_VALUE = 100;
	this.DEFAULT_STOP_VALUE  = 0;
	this.DEFAULT_STEP_VALUE  = -10;

	if (speed == null) {
		speed = this.DEFAULT_SPEED;
	}

	// 初期化
	this.init(
		this, element, speed,
		this.DEFAULT_START_VALUE, this.DEFAULT_STOP_VALUE,
		this.DEFAULT_STEP_VALUE,
		this.startCallable, this.stopCallable, this.stepCallable
	);
});
	/**
	 * 開始処理。
	 *
	 * @param {Object} element    エレメント
	 * @param {Object} startValue 開始値
	 */
	aaoh.kaze.animation.FadeoutAnimation.prototype.
	startCallable = function(element, startValue) {
		aaoh.kaze.Element.setOpacity(element, startValue);
	}

	/**
	 * 終了処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} stopValue 終了値
	 */
	aaoh.kaze.animation.FadeoutAnimation.prototype.
	stopCallable = function(element, stopValue) {
		aaoh.kaze.Element.setOpacity(element, stopValue);
	}

	/**
	 * ステップ処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} nextValue 現在値
	 */
	aaoh.kaze.animation.FadeoutAnimation.prototype.
	stepCallable = function(element, nextValue) {
		aaoh.kaze.Element.setOpacity(element, nextValue);
	}

	/**
	 * 次ステップ値の取得。
	 *
	 * @return {Object} 次ステップ値またはnullを返す
	 */
	aaoh.kaze.animation.FadeoutAnimation.prototype.
	nextValue = function() {
		if (isNaN(this.currentValue_)) {
			return null;
		}

		var retValue = this.currentValue_ + this.stepValue_;
		if (retValue <= this.stopValue_) {
			return null;
		}

		this.currentValue_ = retValue;
		return retValue;
	}
})();

(function () {
/**
 * MoveAnimation。
 *
 * @constructor
 * @extends aaoh.kaze.animation.Animation
 * @param {Object} element    エレメント
 * @param {Number} speed      スピード(msec)
 * @param {Object} startValue 開始値
 * @param {Object} stopValue  終了値
 * @param {Object} stepValue  ステップ値
 */
aaoh.kaze.animation.MoveAnimation = aaoh.kaze.Gadget.extend(
	aaoh.kaze.animation.Animation, function(
		element, speed,
		startValue, stopValue, stepValue
	) {
	this.DEFAULT_SPEED       = 20;
	this.DEFAULT_START_VALUE = { x: 0,  y: 0  };
	this.DEFAULT_STOP_VALUE  = { x: 10, y: 10 };
	this.DEFAULT_STEP_VALUE  = { x: 1,  y: 1  };

	if (speed == null) {
		speed = this.DEFAULT_SPEED;
	}
	if (startValue == null) {
		startValue = this.DEFAULT_START_VALUE
	}
	if (stopValue == null) {
		stopValue = this.DEFAULT_STOP_VALUE
	}
	if (stepValue == null) {
		stepValue = this.DEFAULT_STEP_VALUE
	}

	// 初期化
	this.init(
		this, element, speed,
		startValue, stopValue, stepValue,
		this.startCallable, this.stopCallable, this.stepCallable
	);
});
	/**
	 * 開始処理。
	 *
	 * @param {Object} element    エレメント
	 * @param {Object} startValue 開始値
	 */
	aaoh.kaze.animation.MoveAnimation.prototype.
	startCallable = function(element, startValue) {
		aaoh.kaze.Element.setPosition(
			element, startValue.x, startValue.y
		);
	}

	/**
	 * 終了処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} stopValue 終了値
	 */
	aaoh.kaze.animation.MoveAnimation.prototype.
	stopCallable = function(element, stopValue) {
		aaoh.kaze.Element.setPosition(
			element, stopValue.x, stopValue.y
		);
	}

	/**
	 * ステップ処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} nextValue 現在値
	 */
	aaoh.kaze.animation.MoveAnimation.prototype.
	stepCallable = function(element, nextValue) {
		aaoh.kaze.Element.setPosition(
			element, nextValue.x, nextValue.y
		);
	}

	/**
	 * 次ステップ値の取得。
	 *
	 * @return {Object} 次ステップ値またはnullを返す
	 */
	aaoh.kaze.animation.MoveAnimation.prototype.
	nextValue = function() {
		if (this.currentValue_ == null) {
			return null;
		}

		var retValue = { x: 0, y: 0 };
		retValue.x = this.currentValue_.x + this.stepValue_.x;
		retValue.y = this.currentValue_.y + this.stepValue_.y;
		if (retValue.x >= this.stopValue_.x) {
			retValue.x = this.stopValue_.x;
		}
		if (retValue.y >= this.stopValue_.y) {
			retValue.y = this.stopValue_.y;
		}

		if (
			retValue.x >= this.stopValue_.x
			&& retValue.y >= this.stopValue_.y
		) {
			return null;
		}

		this.currentValue_ = retValue;
		return retValue;
	}
})();

(function () {
/**
 * ResizeAnimation。
 *
 * @constructor
 * @extends aaoh.kaze.animation.Animation
 * @param {Object} element    エレメント
 * @param {Number} speed      スピード(msec)
 * @param {Object} startValue 開始値
 * @param {Object} stopValue  終了値
 * @param {Object} stepValue  ステップ値
 */
aaoh.kaze.animation.ResizeAnimation = aaoh.kaze.Gadget.extend(
	aaoh.kaze.animation.Animation, function(
		element, speed,
		startValue, stopValue, stepValue
	) {
	this.DEFAULT_SPEED       = 20;
	this.DEFAULT_START_VALUE = { width: 0,  height: 0 };
	this.DEFAULT_STOP_VALUE  = { width: 10, height: 10 };
	this.DEFAULT_STEP_VALUE  = { width: 1,  height: 1 };

	if (speed == null) {
		speed = this.DEFAULT_SPEED;
	}
	if (startValue == null) {
		startValue = this.DEFAULT_START_VALUE
	}
	if (stopValue == null) {
		stopValue = this.DEFAULT_STOP_VALUE
	}
	if (stepValue == null) {
		stepValue = this.DEFAULT_STEP_VALUE
	}

	// 初期化
	this.init(
		this, element, speed,
		startValue, stopValue, stepValue,
		this.startCallable, this.stopCallable, this.stepCallable
	);
});
	/**
	 * 開始処理。
	 *
	 * @param {Object} element    エレメント
	 * @param {Object} startValue 開始値
	 */
	aaoh.kaze.animation.ResizeAnimation.prototype.
	startCallable = function(element, startValue) {
		aaoh.kaze.Element.setSize(
			element, startValue.width, startValue.height
		);
	}

	/**
	 * 終了処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} stopValue 終了値
	 */
	aaoh.kaze.animation.ResizeAnimation.prototype.
	stopCallable = function(element, stopValue) {
		aaoh.kaze.Element.setSize(
			element, stopValue.width, stopValue.height
		);
	}

	/**
	 * ステップ処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} nextValue 現在値
	 */
	aaoh.kaze.animation.ResizeAnimation.prototype.
	stepCallable = function(element, nextValue) {
		aaoh.kaze.Element.setSize(
			element, nextValue.width, nextValue.height
		);
	}

	/**
	 * 次ステップ値の取得。
	 *
	 * @return {Object} 次ステップ値またはnullを返す
	 */
	aaoh.kaze.animation.ResizeAnimation.prototype.
	nextValue = function() {
		if (this.currentValue_ == null) {
			return null;
		}

		var retValue = { width: 0, height: 0 };
		retValue.width  = this.currentValue_.width  + this.stepValue_.width;
		retValue.height = this.currentValue_.height + this.stepValue_.height;
		if (retValue.width >= this.stopValue_.width) {
			retValue.width = this.stopValue_.width;
		}
		if (retValue.height >= this.stopValue_.height) {
			retValue.height = this.stopValue_.height;
		}

		if (
			retValue.width >= this.stopValue_.width
			&& retValue.height >= this.stopValue_.height
		) {
			return null;
		}

		this.currentValue_ = retValue;
		return retValue;
	}
})();

(function () {
/**
 * BaseColorAnimation。
 *
 * @constructor
 * @extends aaoh.kaze.animation.Animation
 */
aaoh.kaze.animation.BaseColorAnimation = aaoh.kaze.Gadget.extend(
	aaoh.kaze.animation.Animation, function() {
	// 特になし
});
	/**
	 * 次ステップ値の取得。
	 *
	 * @return {Object} 次ステップ値またはnullを返す
	 */
	aaoh.kaze.animation.BaseColorAnimation.prototype.
	nextValue = function() {
		if (this.currentValue_ == null) {
			return null;
		}
		var retValue  = new aaoh.kaze.Color();

		retValue.r_ = this.currentValue_.r_ + this.stepValue_.r_;
		retValue.g_ = this.currentValue_.g_ + this.stepValue_.g_;
		retValue.b_ = this.currentValue_.b_ + this.stepValue_.b_;
		if (retValue.r_ >= this.stopValue_.r_) {
			retValue.r_ = this.stopValue_.r_;
		}
		if (retValue.g_ >= this.stopValue_.g_) {
			retValue.g_ = this.stopValue_.g_;
		}
		if (retValue.b_ >= this.stopValue_.b_) {
			retValue.b_ = this.stopValue_.b_;
		}

		if (
			retValue.r_ >= this.stopValue_.r_
			&& retValue.g_ >= this.stopValue_.g_
			&& retValue.b_ >= this.stopValue_.b_
		) {
			return null;
		}

		this.currentValue_ = retValue;
		return retValue;
	}

	/**
	 * RGBへの変換。
	 *
	 * @return {String} 色値
	 * @return {Object} RGB値
	 */
	aaoh.kaze.animation.BaseColorAnimation.prototype.
	toRGB = function(color) {
		if (color == null) {
			return null;
		}
		var rgb = new aaoh.kaze.Color(color);

		return rgb;
	}
})();

(function () {
/**
 * ForegroundColorAnimation。
 *
 * @constructor
 * @extends aaoh.kaze.animation.BaseColorAnimation
 * @param {Object} element    エレメント
 * @param {Number} speed      スピード(msec)
 * @param {Object} startValue 開始値
 * @param {Object} stopValue  終了値
 * @param {Object} stepValue  ステップ値
 */
aaoh.kaze.animation.ForegroundColorAnimation = aaoh.kaze.Gadget.extend(
	aaoh.kaze.animation.BaseColorAnimation, function(
		element, speed,
		startValue, stopValue, stepValue
	) {
	this.DEFAULT_SPEED       = 20;
	this.DEFAULT_START_VALUE = new aaoh.kaze.Color('#000000');
	this.DEFAULT_STOP_VALUE  = new aaoh.kaze.Color('#ffffff');
	this.DEFAULT_STEP_VALUE  = new aaoh.kaze.Color('#202020');

	startValue = this.toRGB(startValue);
	stopValue  = this.toRGB(stopValue);
	stepValue  = this.toRGB(stepValue);
	if (speed == null) {
		speed = this.DEFAULT_SPEED;
	}
	if (startValue == null) {
		startValue = this.DEFAULT_START_VALUE
	}
	if (stopValue == null) {
		stopValue = this.DEFAULT_STOP_VALUE
	}
	if (stepValue == null) {
		stepValue = this.DEFAULT_STEP_VALUE
	}

	// 初期化
	this.init(
		this, element, speed,
		startValue, stopValue, stepValue,
		this.startCallable, this.stopCallable, this.stepCallable
	);
});
	/**
	 * 開始処理。
	 *
	 * @param {Object} element    エレメント
	 * @param {Object} startValue 開始値
	 */
	aaoh.kaze.animation.ForegroundColorAnimation.prototype.
	startCallable = function(element, startValue) {

		aaoh.kaze.Element.setForeground(element, startValue);
	}

	/**
	 * 終了処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} stopValue 終了値
	 */
	aaoh.kaze.animation.ForegroundColorAnimation.prototype.
	stopCallable = function(element, stopValue) {

		aaoh.kaze.Element.setForeground(element, stopValue);
	}

	/**
	 * ステップ処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} nextValue 現在値
	 */
	aaoh.kaze.animation.ForegroundColorAnimation.prototype.
	stepCallable = function(element, nextValue) {

		aaoh.kaze.Element.setForeground(element, nextValue);
	}
})();

(function () {
/**
 * BackgroundColorAnimation。
 *
 * @constructor
 * @extends aaoh.kaze.animation.BaseColorAnimation
 * @param {Object} element    エレメント
 * @param {Number} speed      スピード(msec)
 * @param {Object} startValue 開始値
 * @param {Object} stopValue  終了値
 * @param {Object} stepValue  ステップ値
 */
aaoh.kaze.animation.BackgroundColorAnimation = aaoh.kaze.Gadget.extend(
	aaoh.kaze.animation.BaseColorAnimation, function(
		element, speed,
		startValue, stopValue, stepValue
	) {
	this.DEFAULT_SPEED       = 20;
	this.DEFAULT_START_VALUE = new aaoh.kaze.Color('#ffffff');
	this.DEFAULT_STOP_VALUE  = new aaoh.kaze.Color('#000000');
	this.DEFAULT_STEP_VALUE  = new aaoh.kaze.Color('#202020');

	startValue = this.toRGB(startValue);
	stopValue  = this.toRGB(stopValue);
	stepValue  = this.toRGB(stepValue);
	if (speed == null) {
		speed = this.DEFAULT_SPEED;
	}
	if (startValue == null) {
		startValue = this.DEFAULT_START_VALUE
	}
	if (stopValue == null) {
		stopValue = this.DEFAULT_STOP_VALUE
	}
	if (stepValue == null) {
		stepValue = this.DEFAULT_STEP_VALUE
	}

	// 初期化
	this.init(
		this, element, speed,
		startValue, stopValue, stepValue,
		this.startCallable, this.stopCallable, this.stepCallable
	);
});
	/**
	 * 開始処理。
	 *
	 * @param {Object} element    エレメント
	 * @param {Object} startValue 開始値
	 */
	aaoh.kaze.animation.BackgroundColorAnimation.prototype.
	startCallable = function(element, startValue) {

		aaoh.kaze.Element.setForeground(element, startValue);
	}

	/**
	 * 終了処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} stopValue 終了値
	 */
	aaoh.kaze.animation.BackgroundColorAnimation.prototype.
	stopCallable = function(element, stopValue) {

		aaoh.kaze.Element.setForeground(element, stopValue);
	}

	/**
	 * ステップ処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} nextValue 現在値
	 */
	aaoh.kaze.animation.BackgroundColorAnimation.prototype.
	stepCallable = function(element, nextValue) {

		aaoh.kaze.Element.setForeground(element, nextValue);
	}
})();

(function () {
/**
 * BorderColorAnimation。
 *
 * @constructor
 * @extends aaoh.kaze.animation.BaseColorAnimation
 * @param {Object} element    エレメント
 * @param {Number} speed      スピード(msec)
 * @param {Object} startValue 開始値
 * @param {Object} stopValue  終了値
 * @param {Object} stepValue  ステップ値
 */
aaoh.kaze.animation.BorderColorAnimation = aaoh.kaze.Gadget.extend(
	aaoh.kaze.animation.BaseColorAnimation, function(
		element, speed,
		startValue, stopValue, stepValue
	) {
	this.DEFAULT_SPEED       = 20;
	this.DEFAULT_START_VALUE = new aaoh.kaze.Color('#000000');
	this.DEFAULT_STOP_VALUE  = new aaoh.kaze.Color('#ffffff');
	this.DEFAULT_STEP_VALUE  = new aaoh.kaze.Color('#202020');

	startValue = this.toRGB(startValue);
	stopValue  = this.toRGB(stopValue);
	stepValue  = this.toRGB(stepValue);
	if (speed == null) {
		speed = this.DEFAULT_SPEED;
	}
	if (startValue == null) {
		startValue = this.DEFAULT_START_VALUE
	}
	if (stopValue == null) {
		stopValue = this.DEFAULT_STOP_VALUE
	}
	if (stepValue == null) {
		stepValue = this.DEFAULT_STEP_VALUE
	}

	// 初期化
	this.init(
		this, element, speed,
		startValue, stopValue, stepValue,
		this.startCallable, this.stopCallable, this.stepCallable
	);
});
	/**
	 * 開始処理。
	 *
	 * @param {Object} element    エレメント
	 * @param {Object} startValue 開始値
	 */
	aaoh.kaze.animation.BorderColorAnimation.prototype.
	startCallable = function(element, startValue) {

		aaoh.kaze.Element.setForeground(element, startValue);
	}

	/**
	 * 終了処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} stopValue 終了値
	 */
	aaoh.kaze.animation.BorderColorAnimation.prototype.
	stopCallable = function(element, stopValue) {

		aaoh.kaze.Element.setForeground(element, stopValue);
	}

	/**
	 * ステップ処理。
	 *
	 * @param {Object} element   エレメント
	 * @param {Object} nextValue 現在値
	 */
	aaoh.kaze.animation.BorderColorAnimation.prototype.
	stepCallable = function(element, nextValue) {

		aaoh.kaze.Element.setForeground(element, nextValue);
	}
})();
