%feature("docstring") OT::KernelSmoothing
"Non parametric continuous distribution estimation by kernel smoothing.

Refer to :ref:`kernel_smoothing`.

Parameters
----------
kernel : :class:`~openturns.Distribution`, optional
    Univariate distribution of the kernel that will be used. By default, the standard Normal distribution is used. 
binned : bool, optional
    Activates bining mechanism only in the univariate or bivariate cases. It allows one to speed up the manipulation of the density function of the resulting distribution.  By default, the mechanism is activated.
binNumber : int, :math:`binNumber \\geq 2`, optional
    Indicates the number of bins used by the bining mechanism. By default, OpenTURNS uses the values stored in the *ResourceMap*.
boundaryCorrection : bool, optional
    Activates the boundary correction using the mirroring technique.
    By default, the correction is not provided.

Notes
-----
The binning mechanism creates a regular grid of *binNumber* intervals in each
dimension, then the unit weight of each point is linearly affected to the vertices
of the bin containing the point (see [wand1994]_ appendix D, page 182).
The `KernelSmoothing-BinNumber` key defines the default value of the 
number of bins used in the _binning_ algorithm to improve the evaluation speed.

The boundary correction is available only in one dimension, and it is done using
the mirroring technique. See the notes of the :meth:`setBoundingOption` method for
details.

When applied to multivariate samples, the kernel is the kernel product of the
univariate distribution specified in the constructor.

Examples
--------
Fit a distribution on data thanks to the kernel smoothing technique:

>>> import openturns as ot
>>> ot.RandomGenerator.SetSeed(0)
>>> sample = ot.Gamma(6.0, 1.0).getSample(100)
>>> ks = ot.KernelSmoothing()
>>> fittedDist = ks.build(sample)
>>> print(fittedDist.getClassName())
Distribution

The :meth:`build` method produces a generic :class:`~openturns.Distribution`
object. Other build methods (detailed below) produce more specific objects.

Get the bandwidth:

>>> bandwidth = ks.getBandwidth()
>>> print(bandwidth)
[0.862207]

The bandwidth was evaluated by the :meth:`build` method.
It could also have been provided by the user.

>>> bandwidth = [0.9]
>>> fittedDist = ks.build(sample, bandwidth)

Compare the PDFs:

>>> graph = fittedDist.drawPDF()
>>> graph.add( ot.Gamma(6.0, 1.0).drawPDF())
>>> graph.setColors(ot.Drawable.BuildDefaultPalette(2))
>>> graph.setLegends(['KS dist', 'Gamma'])

The default values of the parameters of the constructor usually provide good results.
Nevertheless, the parameters can be manually set.

>>> kernel = ot.Uniform()
>>> ks = ot.KernelSmoothing(kernel)
>>> binned = True # by default True
>>> binNumber = 64
>>> ks = ot.KernelSmoothing(kernel, binned, binNumber)
>>> boundaryCorrection = True # by default False
>>> ks = ot.KernelSmoothing(kernel, binned, binNumber, boundaryCorrection)

Variants of the :meth:`build` method can be used when the distribution to build
is expected to be of a certain type. In those cases however,
the bandwidth must be user-specified.
To use :meth:`buildAsTruncatedDistribution`, boundary correction must be enabled.

>>> distribution = ks.buildAsKernelMixture(sample, bandwidth)
>>> print(distribution.getClassName())
KernelMixture
>>> distribution = ks.buildAsMixture(sample, bandwidth)
>>> print(distribution.getClassName())
Mixture
>>> distribution = ks.buildAsTruncatedDistribution(sample, bandwidth)
>>> print(distribution.getClassName())
TruncatedDistribution"

// ---------------------------------------------------------------------
%feature("docstring") OT::KernelSmoothing::buildAsKernelMixture
"Fit a kernel smoothing distribution on data.

Parameters
----------
sample : 2-d sequence of float
    Data on which the distribution is fitted. Any dimension.
bandwidth : :class:`~openturns.Point`
    Contains the bandwidth in each direction.

Returns
-------
fittdDist : :class:`~openturns.KernelMixture`
    The fitted distribution.

Notes
-----
It builds a :math:`~openturns.KernelMixture` using the given data and bandwidth regardless of the binning or boundary treatment flags.

Examples
--------

>>> import openturns as ot
>>> sample = ot.Exponential(1.0).getSample(1000)
>>> smoother = ot.KernelSmoothing()
>>> kernelMixture = smoother.buildAsKernelMixture(sample, [1.0])
"

// ---------------------------------------------------------------------
%feature("docstring") OT::KernelSmoothing::buildAsMixture
"Fit a kernel smoothing distribution on data.

Parameters
----------
sample : 2-d sequence of float
    Data on which the distribution is fitted. Any dimension.
bandwidth : :class:`~openturns.Point`
    Contains the bandwidth in each direction.

Returns
-------
fittdDist : :class:`~openturns.KernelMixture`
    The fitted distribution.

Notes
-----
It builds a :math:`~openturns.Mixture` using the given bandwidth and a binning of the given data regardless of the bin number, the data size, the binning flag or boundary treatment flags. This method is available only for 1D or 2D samples.

Examples
--------

>>> import openturns as ot
>>> sample = ot.Exponential(1.0).getSample(1000)
>>> smoother = ot.KernelSmoothing(ot.Normal(), True, 100, False)
>>> mixture = smoother.buildAsMixture(sample, [1.0])
"

// ---------------------------------------------------------------------
%feature("docstring") OT::KernelSmoothing::buildAsTruncatedDistribution
"Estimate the distribution as :class:`~openturns.TruncatedDistribution`.

Parameters
----------
sample : 2-d sequence of float
    Data on which the distribution is fitted. Any dimension.
bandwidth : :class:`~openturns.Point`
    Contains the bandwidth in each direction.

Returns
-------
fittdDist : :class:`~openturns.TruncatedDistribution`
    The estimated distribution as a :class:`~openturns.TruncatedDistribution`.

Examples
--------

>>> import openturns as ot
>>> sample = ot.Exponential(1.0).getSample(1000)
>>> smoother = ot.KernelSmoothing(ot.Normal(), False, 0, True)
>>> truncated = smoother.buildAsTruncatedDistribution(sample, [1.0])
"

// ---------------------------------------------------------------------
%feature("docstring") OT::KernelSmoothing::build
"Fit a kernel smoothing distribution on data.

Parameters
----------
sample : 2-d sequence of float
    Data on which the distribution is fitted. Any dimension.
bandwidth : :class:`~openturns.Point`, optional
    Contains the bandwidth in each direction. If not specified, the bandwidth is calculated using the mixed rule from data.

Returns
-------
fittdDist : :class:`~openturns.Distribution`
    The fitted distribution.

Notes
-----
According to the dimension of the data and the specified treatments, the resulting distribution differs.

- If the sample is constant, a :class:`~openturns.Dirac` distribution is built.
- If dimension > 2 or if no treatment has been asked for, a :class:`~openturns.KernelMixture` is built by calling *buildAsKernelMixture*. 
- If dimension = 1 and a boundary treatment has been asked for, a :class:`~openturns.TruncatedDistribution` is built by calling *buildAsTruncatedDistribution*
- If dimension = 1 or 2 and no boundary treatment has been asked for, but a binning treatment has been asked for,

    - If the sample size is greater than the bin number, then a :class:`~openturns.Mixture` is built by calling `buildAsMixture`
    - Otherwise a :class:`~openturns.KernelMixture` is built by calling `buildAsKernelMixture`
    
The bandwidth selection depends on the dimension.

- If dimension = 1, then `computeMixedBandwidth` is used.
- Otherwise, then the only multivariate rule `computeSilvermanBandwidth` is used.

Examples
--------
See the effect of the boundary correction:

>>> import openturns as ot
>>> sample = ot.Exponential(1.0).getSample(1000)
>>> smoother = ot.KernelSmoothing()
>>> fittedDistNoCorr = smoother.build(sample)
>>> smoother.setBoundaryCorrection(True)
>>> fittedDistWithCorr = smoother.build(sample)

Compare the PDFs:

>>> graph = ot.Exponential(1.0).drawPDF()
>>> graph.add(fittedDistNoCorr.drawPDF())
>>> graph.add(fittedDistWithCorr.drawPDF())
>>> graph.setColors(['black', 'blue', 'red'])
>>> graph.setLegends(['Exp dist', 'No boundary corr', 'Boundary corr'])
"

// ---------------------------------------------------------------------
%feature("docstring") OT::KernelSmoothing::getBandwidth
"Accessor to the bandwidth used in the kernel smoothing.

Returns
-------
bandwidth : :class:`~openturns.Point`
    Bandwidth used in each direction.
"

// ---------------------------------------------------------------------
%feature("docstring") OT::KernelSmoothing::getKernel
"Accessor to kernel used in the kernel smoothing.

Returns
-------
kernel : :class:`~openturns.Distribution`
    Univariate distribution used to build the kernel.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::KernelSmoothing::setBoundaryCorrection
"Accessor to the boundary correction flag.

Parameters
----------
boundaryCorrection : bool
    Activates the boundary correction using the mirroring technique."

// ---------------------------------------------------------------------

%feature("docstring") OT::KernelSmoothing::setBoundingOption
"Accessor to the boundary correction option.

Parameters
----------
boundingOption : int
    Select the boundary correction option, see notes.

Notes
-----
The possible values for the bounding option are:

    - KernelSmoothing.NONE  or 0: no boundary correction
    - KernelSmoothing.LOWER or 1: apply the boundary correction to the lower bound
    - KernelSmoothing.UPPER or 2: apply the boundary correction to the upper bound
    - KernelSmoothing.BOTH  or 3: apply the boundary correction to both bounds

It applies only to 1D samples. Each bound can be defined by the user or computed
automatically from the sample, see *setLowerBound*, *setUpperBound*,
*setAutomaticLowerBound*, *setAutomaticUpperBound*."

// ---------------------------------------------------------------------

%feature("docstring") OT::KernelSmoothing::setLowerBound
"Accessor to the lower bound for boundary correction.

Parameters
----------
lowerBound : float
    A user-defined lower bound to take into account for boundary correction.

Notes
-----
This method automatically sets the *automaticLowerBound* flag to *False*.
The given value will be taken into account only if *boundingOption* is set to
either 1 or 3. If the algorithm is applied to a sample with a minimum value
less than the user-defined lower bound and the *automaticLowerBound* is set to
*False*, then an exception it raised.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::KernelSmoothing::setUpperBound
"Accessor to the upper bound for boundary correction.

Parameters
----------
upperBound : float
    A user-defined lower bound to take into account for boundary correction.

Notes
-----
This method automatically sets the *automaticLowerBound* flag to *False*.
The given value will be taken into account only if *boundingOption* is set to
either 1 or 3. If the algorithm is applied to a sample with a minimum value
less than the user-defined lower bound and the *automaticLowerBound* is set to
*False*, then an exception it raised.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::KernelSmoothing::setAutomaticLowerBound
"Accessor to the flag for an automatic selection of lower bound.

Parameters
----------
automaticLowerBound : bool
    Flag to tell if the user-defined lower bound has to be taken into account (value *False*) or if the minimum of the given sample has to be used (value *True*)."

// ---------------------------------------------------------------------

%feature("docstring") OT::KernelSmoothing::setAutomaticUpperBound
"Accessor to the flag for an automatic selection of upper bound.

Parameters
----------
automaticUpperBound : bool
    Flag to tell if the user-defined upper bound has to be taken into account (value *False*) or if the maximum of the given sample has to be used (value *True*)."

// ---------------------------------------------------------------------
%feature("docstring") OT::KernelSmoothing::computeSilvermanBandwidth
"Compute the bandwidth according to the Silverman rule.

Returns
-------
bandwidth : :class:`~openturns.Point`
    Bandwidth which components are evaluated according to the Silverman rule
    assuming a normal distribution.
    The bandwidth uses a robust estimate of the
    sample standard deviation, based on the interquartile range introduced
    in :ref:`kernel_smoothing` (rather than the sample standard deviation).
    This method can manage a multivariate sample and produces a
    multivariate bandwidth.
"

// ---------------------------------------------------------------------
%feature("docstring") OT::KernelSmoothing::computePluginBandwidth
"Compute the bandwidth according to the plugin rule.

Returns
-------
bandwidth : :class:`~openturns.Point`
    Bandwidth which components are evaluated according to the plugin rule.

Notes
-----
This plug-in method is based on the *solve-the-equation* rule from [sheather1991]_.
This method can take a lot of time for large samples, as the cost is  quadratic with the sample size.

Several keys of the :class:`~openturns.ResourceMap` are used by the [sheather1991]_ method.

- The key `KernelSmoothing-AbsolutePrecision` is used in the Sheather-Jones algorithm
  to estimate the bandwidth.
  It defines the absolute tolerance used by the solver
  to solve the nonlinear equation.

- The `KernelSmoothing-MaximumIteration` key defines the maximum number of iterations
  used by the solver.

- The `KernelSmoothing-RelativePrecision` key defines the relative tolerance.

- The `KernelSmoothing-AbsolutePrecision` key defines the absolute tolerance. 

- The `KernelSmoothing-ResidualPrecision` key defines the absolute
  tolerance on the residual.

- The `KernelSmoothing-CutOffPlugin` key is the cut-off value
  introduced in :ref:`kernel_smoothing`.

More precisely, the `KernelSmoothing-CutOffPlugin` key of the :class:`~openturns.ResourceMap` controls
the accuracy of the approximation used to estimate the rugosity of the
second derivative of the distribution.
The default value ensures that terms in the sum which weight are lower than
:math:`4 \\times 10^{-6}` are ignored, which can reduce the calculation in some situations.
The properties of the standard gaussian density are so that,
in order to make the computation exact, the value of the
`KernelSmoothing-CutOffPlugin` must be set to 39, but this may increase the
computation time.
"

// ---------------------------------------------------------------------
%feature("docstring") OT::KernelSmoothing::computeMixedBandwidth
"Compute the bandwidth according to a mixed rule.

Returns
-------
bandwidth : :class:`~openturns.Point`
    Bandwidth which components are evaluated according to a mixed rule.

Notes
-----
This method uses the *mixed* rule introduced in :ref:`kernel_smoothing`.
Its goal is to provide an accurate estimator of the bandwidth 
when the sample size is large.

Let :math:`n` be the sample size.
The estimator depends on the threshold sample size :math:`n_t` defined in the
`KernelSmoothing-SmallSize` key of the :class:`~openturns.ResourceMap`.


- If :math:`n \\leq n_t`, i.e. for a small sample, we use the plugin solve-the-equation
  method.

- Otherwise, the *mixed* rule is used.
"
