// @HEADER
// *****************************************************************************
//        MueLu: A package for multigrid based preconditioning
//
// Copyright 2012 NTESS and the MueLu contributors.
// SPDX-License-Identifier: BSD-3-Clause
// *****************************************************************************
// @HEADER

#ifndef MUELU_COARSEMAPFACTORY_DECL_HPP_
#define MUELU_COARSEMAPFACTORY_DECL_HPP_

#include "Xpetra_StridedMapFactory_fwd.hpp"

#include "MueLu_ConfigDefs.hpp"
#include "MueLu_SingleLevelFactoryBase.hpp"
#include "MueLu_CoarseMapFactory_fwd.hpp"

#include "MueLu_Level_fwd.hpp"
#include "MueLu_Aggregates_fwd.hpp"
#include "MueLu_Exceptions.hpp"

namespace MueLu {

/*!
  @class CoarseMapFactory class.
  @brief Factory for generating coarse level map. Used by TentativePFactory.

  Factory is used by TentativePFactory to generate the coarse level map (=domain map) of
  the prolongation operator. The number of DOFs is calculated by the following formula
  \f[
  n_{Aggs}\cdot d_{blocksize}
  \f]
  where \f$ n_{Aggs} \f$ denotes the number of aggregates (generated by the AggregationFactory) and
  \f$ d_{blocksize}\f$ denotes the number of DOFs per node on the coarse level.
  For determing the blocksize, the @c CoarseMapFactory supports strided map information.
  Per default @c stridedBlockId is -1 and the number of nullspace vectors (from @c "Nullspace" ) is used.
  Otherwise, the block size is calculated using the striding information.

  By setting the "Domain GID offsets" parameter, we can specify the GID offset manually for each level.
  If the number of entries in that list is smaller than the number of levels,
  we use the default offset 0 for the remaining coarse levels.

  @ingroup MueLuTransferClasses

  ## Input/output ##

  ### User parameters of this factory ###
  Parameter | type | default | master.xml | validated | requested | description
  ----------|------|---------|:----------:|:---------:|:---------:|------------
  Aggregates | Factory | null | | * | * | Generating factory for aggregates. We basically need only the number of aggregates in this factory.
  Nullspace  | Factory | null |   | * | * | Generating factory of the fine nullspace vectors (of type "MultiVector"). In the default case the same instance of the TentativePFactory is also the generating factory for the null space vectors (on the next coarser levels). Therefore, it is recommended to declare the TentativePFactory to be the generating factory of the "Nullspace" variable globally using the FactoryManager object! For defining the near null space vectors on the finest level one should use the NullspaceFactory.
  Striding info | std::string | {} |  | * | | String containing the striding information, e.g. {2,1} for a fixed block size of 3 with sub blocks of size 2 and 1 in each node.
  Strided block id | LocalOrdinal | -1 |  | * | | Strided block id in strided information vector or -1 (=default) for the full block information.
  Domain GID offsets | std::string | {0} |  | * | | String describing a std::vector with offsets for GIDs for each level. If no offset GID value is given for the level we use 0 as default.

  The * in the @c master.xml column denotes that the parameter is defined in the @c master.xml file.<br>
  The * in the @c validated column means that the parameter is declared in the list of valid input parameters (see @c GetValidParameters() ).<br>
  The * in the @c requested column states that the data is requested as input with all dependencies (see @c DeclareInput() ).

  \note To auto-compute the Domain GID offset on each level, consider to use the @c BlockedCoarseMapFactory .

  ### Variables provided by this factory ###

  After @c Build() the following data is available (if requested)

  Parameter | generated by | description
  ----------|--------------|------------
  | CoarseMap | CoarseMapFactory | Map containing the coarse map used as domain map in the tentative prolongation operator

*/

template <class Scalar        = DefaultScalar,
          class LocalOrdinal  = DefaultLocalOrdinal,
          class GlobalOrdinal = DefaultGlobalOrdinal,
          class Node          = DefaultNode>
class CoarseMapFactory : public SingleLevelFactoryBase {
#undef MUELU_COARSEMAPFACTORY_SHORT
#include "MueLu_UseShortNames.hpp"

 public:
  //! @name Input
  //@{

  RCP<const ParameterList> GetValidParameterList() const override;

  /*!
    @brief Specifies the data that this class needs, and the factories that generate that data.

    If the Build method of this class requires some data, but the generating factory is not specified in DeclareInput,
    then this class will fall back to the settings in FactoryManager.
  */
  void DeclareInput(Level& currentLevel) const override;

  //@}

  //! @name Build methods.
  //@{

  //! Build an object with this factory.
  void Build(Level& currentLevel) const override;

  //@}

  /*! @brief Get information on the fixed block size
   *
   * Returns the full block size (number of DOFs per node) of the domain DOF map (= coarse map).
   * This is the sum of all entries in the striding vector.
   * e.g. for 2 velocity dofs and 1 pressure dof the return value is 3.
   */
  virtual size_t getFixedBlockSize() const {
    // sum up size of all strided blocks (= number of dofs per node)
    size_t blkSize = 0;
    std::vector<size_t>::const_iterator it;
    for (it = stridingInfo_.begin(); it != stridingInfo_.end(); ++it) {
      blkSize += *it;
    }
    return blkSize;
  }

  //! @name Get/Set functions

  /*! @brief getStridingData
   * returns vector with size of striding blocks in the domain DOF map (= coarse map).
   * e.g. for 2 velocity dofs and 1 pressure dof the vector is (2,1)
   */
  virtual std::vector<size_t> getStridingData() const { return stridingInfo_; }

  /*! @brief setStridingData
   * set striding vector for the domain DOF map (= coarse map),
   * e.g. (2,1) for 2 velocity dofs and 1 pressure dof
   */
  virtual void setStridingData(std::vector<size_t> stridingInfo);

  /*! @brief getStridedBlockId
   * returns strided block id for the domain DOF map of Ptent (= coarse map)
   * or -1 if full strided map is stored in the domain map of Ptent (= coarse map)
   */
  virtual LocalOrdinal getStridedBlockId() const {
    const ParameterList& pL = GetParameterList();
    return pL.get<LocalOrdinal>("Strided block id");
  }

  /*! @brief setStridedBlockId
   * set strided block id for the domain DOF map of Ptent (= coarse map)
   * or -1 if full strided map is stored in the domain map of Ptent (= coarse map)
   */
  virtual void setStridedBlockId(LocalOrdinal stridedBlockId) {
    SetParameter("Strided block id", ParameterEntry(stridedBlockId));
  }

  //@}

 protected:
  //! Build the coarse map using the domain GID offset
  virtual void BuildCoarseMap(Level& currentLevel, const GlobalOrdinal domainGIDOffset) const;

  //! Extract domain GID offset from user data
  virtual GlobalOrdinal GetDomainGIDOffset(Level& currentLevel) const;

 private:
  virtual void CheckForConsistentStridingInformation(
      LocalOrdinal stridedBlockId, const size_t nullspaceDimension) const;

  //! Vector with size of strided blocks (dofs)
  mutable std::vector<size_t> stridingInfo_;

};  // class CoarseMapFactory

}  // namespace MueLu

#define MUELU_COARSEMAPFACTORY_SHORT
#endif /* MUELU_COARSEMAPFACTORY_DECL_HPP_ */
